% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/ddbs_ops_unary.R
\name{ddbs_transform}
\alias{ddbs_transform}
\title{Transform coordinate reference system of geometries}
\usage{
ddbs_transform(
  x,
  y,
  conn = NULL,
  name = NULL,
  crs = NULL,
  crs_column = "crs_duckspatial",
  overwrite = FALSE,
  quiet = FALSE
)
}
\arguments{
\item{x}{An \code{sf} spatial object. Alternatively, it can be a string with the
name of a table with geometry column within the DuckDB database \code{conn}.
Data is returned from this object.}

\item{y}{Target CRS. Can be:
\itemize{
\item A character string with EPSG code (e.g., "EPSG:4326")
\item An \code{sf} object (uses its CRS)
\item Name of a DuckDB table (uses its CRS)
}}

\item{conn}{A connection object to a DuckDB database. If \code{NULL}, the function
runs on a temporary DuckDB database.}

\item{name}{A character string of length one specifying the name of the table,
or a character string of length two specifying the schema and table
names. If \code{NULL} (the default), the function returns the result as an
\code{sf} object}

\item{crs}{The coordinates reference system of the data. Specify if the data
doesn't have a \code{crs_column}, and you know the CRS.}

\item{crs_column}{a character string of length one specifying the column
storing the CRS (created automatically by \code{\link{ddbs_write_vector}}).
Set to \code{NULL} if absent.}

\item{overwrite}{Boolean. whether to overwrite the existing table if it exists. Defaults
to \code{FALSE}. This argument is ignored when \code{name} is \code{NULL}.}

\item{quiet}{A logical value. If \code{TRUE}, suppresses any informational messages.
Defaults to \code{FALSE}.}
}
\value{
an \code{sf} object or \code{TRUE} (invisibly) for table creation
}
\description{
Transforms geometries from a DuckDB table to a different coordinate reference system
using the spatial extension. Works similarly to \code{sf::st_transform()}.
Returns the result as an \code{sf} object or creates a new table in the database.
}
\examples{
\dontrun{
## load packages
library(duckspatial)
library(sf)

# create a duckdb database in memory (with spatial extension)
conn <- ddbs_create_conn(dbdir = "memory")

## read data
argentina_sf <- st_read(system.file("spatial/argentina.geojson", package = "duckspatial"))

## store in duckdb
ddbs_write_vector(conn, argentina_sf, "argentina")

## transform to different CRS using EPSG code
ddbs_transform("argentina", "EPSG:3857", conn)

## transform to match CRS of another sf object
argentina_3857_sf <- st_transform(argentina_sf, "EPSG:3857")
ddbs_write_vector(conn, argentina_3857_sf, "argentina_3857")
ddbs_transform("argentina", argentina_3857_sf, conn)

## transform to match CRS of another DuckDB table
ddbs_transform("argentina", "argentina_3857", conn)

## transform without using a connection
ddbs_transform(argentina_sf, "EPSG:3857")
}
}
