#' Calculate the activations for all outcomes in the data.
#' 
#' @description Calculate the activations for all outcomes in the data 
#' per learning event. The activation values are returned as data frame.
#' @export
#' @param wmlist A list with weightmatrices, generated by 
#' \code{\link{RWlearning}} or \code{\link{updateWeights}}, 
#' or a single weightmatrix (matrix). 
#' @param data Data frame with columns \code{Cues} and \code{Outcomes}. 
#' Number of rows should be the same as the number of weightmatrices 
#' in \code{wmlist}.
#' @param split String, separator between cues and/or outcomes.
#' @param select.outcomes Optional selection of outcomes to limit (or expand) 
#' the number of activations that are returned. 
#' The value of NULL (default) will 
#' return all activations (for each outcome in \code{data}). 
#' Note that specified values that are not in 
#' the weightmatrix will return the initial value without error or 
#' warning. Please use \code{\link{getValues}} for returning all 
#' outcomes in the data.
#' @param init.value Value of activations for non-existing connections. 
#' Typically set to 0.
#' @param normalize Logical: whether or not the activation is normalized by 
#' dividing the total activation by the number of cues. Default is FALSE. If 
#' set to TRUE, the activation reflects the average activation per cue.
#' @return Vector or list of activation values (see \code{return.list} 
#' and \code{fun} for the specific conditions, and the examples below).
#' @section Notes:
#' The outcomes are selected based on the data with events, and not 
#' necessarily all outcomes present in the weightmatrices. For example, 
#' when the weightmatrices were first trained on another data set, some 
#' outcomes may be present in the weightmatrices but not in the current 
#' training data. To include these as well, the user can specify these 
#' extra outcomes with the argument \code{select.outcomes}.
#' @author Jacolien van Rij
#' @seealso \code{\link{getWeightsByCue}},
#' \code{\link{getWeightsByOutcome}}
#' @family functions for calculating activations
#' @examples
#' # load example data:
#' data(dat)
#' 
#' # add obligatory columns Cues, Outcomes, and Frequency:
#' dat$Cues <- paste("BG", dat$Shape, dat$Color, sep="_")
#' dat$Outcomes <- dat$Category
#' dat$Frequency <- dat$Frequency1
#' head(dat)
#' 
#' # now use createTrainingData to sample from the specified frequencies: 
#' train <- createTrainingData(dat)
#' head(train)
#' 
#' # this training data can actually be used train network:
#' wm <- RWlearning(train)
#' 
#' # Now we calculate the activations for all outcomes
#' # per event:
#' activations <- activationsOutcomes(wm, train)
#' head(activations)
#' 
#' # Now with selection of outcomes (note that 'dog' does
#' # not occur as outcome in the data):
#' activations2 <- activationsOutcomes(wm, train, 
#'     select.outcomes = c("plant", "vehicle", "dog"))
#' head(activations2)
#' tail(activations2)
#' 


activationsOutcomes <- function(wmlist, data, 
	split="_", select.outcomes=NULL, 
	init.value=0, normalize=FALSE){

	# check wmlist:
	if(!is.list(wmlist)){
		if(is.matrix(wmlist)){
			wmlist <- list(wmlist)
		}else{
			stop("Argument wmlist should be list of weightmatrices or a single weightmatrix.")
		}
	}
	if(length(wmlist)!=nrow(data)){
		stop("Argument wmlist should have same length as the number of rows in data.")
	}
	# check data:
	if(!"Cues" %in% colnames(data)){
		stop("Cannot find column 'Cues' in data.")
	}
	
	# extract all outcomes:
	outcomes <- getValues(data[,"Outcomes"], split=split, unique = TRUE)
	if(!is.null(select.outcomes)){
		outcomes <- select.outcomes
	}

	# calculate raw activations for each of the outcomes:
	out <- mapply(function(cur.wm, cur.cues){
		return(activationsMatrix(wm=cur.wm, cues=cur.cues, 
			select.outcomes=paste(outcomes, sep=split),
			split=split, init.value=init.value, normalize=normalize))
	}, wmlist, as.list(data[,"Cues"]), SIMPLIFY = FALSE)

	out <- as.data.frame(rbindlist(out))
	return(out)
}

