% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/epoxy_transform_inline.R
\name{epoxy_transform_inline}
\alias{epoxy_transform_inline}
\title{Epoxy Inline Transformer}
\usage{
epoxy_transform_inline(
  ...,
  transformer = glue::identity_transformer,
  .and = and::and,
  .or = and::or,
  .incr = sort,
  .decr = function(x) sort(x, decreasing = TRUE),
  .bytes = scales::label_bytes(),
  .date = function(x) format(x, format = "\%F"),
  .time = function(x) format(x, format = "\%T"),
  .datetime = function(x) format(x, format = "\%F \%T"),
  .dollar = scales::label_dollar(prefix = engine_pick("$", "$", "\\\\$")),
  .number = scales::label_number(),
  .comma = scales::label_comma(),
  .ordinal = scales::label_ordinal(),
  .percent = scales::label_percent(suffix = engine_pick("\%", "\%", "\\\\\%")),
  .pvalue = scales::label_pvalue(),
  .scientific = scales::label_scientific(),
  .uppercase = toupper,
  .lowercase = tolower,
  .titlecase = function(x) tools::toTitleCase(as.character(x)),
  .sentence = function(x) `substr<-`(x, 1, 1, toupper(substr(x, 1, 1))),
  .squote = function(x) sQuote(x, q = getOption("epoxy.fancy_quotes", FALSE)),
  .dquote = function(x) dQuote(x, q = getOption("epoxy.fancy_quotes", FALSE)),
  .strong = NULL,
  .emph = NULL,
  .code = NULL
)
}
\arguments{
\item{...}{Additional named inline transformers as functions taking at least
one argument. The evaluated expression from the template expression is
passed as the first argument to the function. The argument names must
include the leading \code{.} to match the syntax used inline.}

\item{transformer}{The transformer to apply to the replacement string. This
argument is used for chaining the transformer functions. By providing a
function to this argument you can apply an additional transformation after
the current transformation. In nearly all cases, you can let
\code{epoxy_transform()} handle this for you. The chain ends when
\code{\link[glue:identity_transformer]{glue::identity_transformer()}} is used as the \code{transformer}.}

\item{.and}{The function to apply to \code{x} when the template is \verb{\{.and x\}}. Default is \code{\link[and:and]{and::and()}}.}

\item{.or}{The function to apply to \code{x} when the template is \verb{\{.or x\}}. Default is \code{\link[and:and]{and::or()}}.}

\item{.incr}{The function to apply to \code{x} when the template is \verb{\{.incr x\}}. Default is \code{\link[=sort]{sort()}}.}

\item{.decr}{The function to apply to \code{x} when the template is \verb{\{.decr x\}}. Default is \code{function(x) sort(x, decreasing = TRUE)}.}

\item{.bytes}{The function to apply to \code{x} when the template is \verb{\{.bytes x\}}. Default is \code{\link[scales:label_bytes]{scales::label_bytes()}}.}

\item{.date}{The function to apply to \code{x} when the template is \verb{\{.date x\}}. Default is \code{function(x) format(x, format = "\%F")}.}

\item{.time}{The function to apply to \code{x} when the template is \verb{\{.time x\}}. Default is \code{function(x) format(x, format = "\%T")}.}

\item{.datetime}{The function to apply to \code{x} when the template is \verb{\{.datetime x\}} or \verb{\{.dttm x\}}. Default is \code{function(x) format(x, format = "\%F \%T")}.}

\item{.dollar}{The function to apply to \code{x} when the template is \verb{\{.dollar x\}}. Default is \code{\link[scales:label_dollar]{scales::label_dollar(prefix = engine_pick("$", "$", "\\\\$"))}}.}

\item{.number}{The function to apply to \code{x} when the template is \verb{\{.number x\}} or \verb{\{.num x\}}. Default is \code{\link[scales:label_number]{scales::label_number()}}.}

\item{.comma}{The function to apply to \code{x} when the template is \verb{\{.comma x\}}. Default is \code{\link[scales:label_number]{scales::label_comma()}}.}

\item{.ordinal}{The function to apply to \code{x} when the template is \verb{\{.ordinal x\}}. Default is \code{\link[scales:label_ordinal]{scales::label_ordinal()}}.}

\item{.percent}{The function to apply to \code{x} when the template is \verb{\{.percent x\}} or \verb{\{.pct x\}}. Default is \code{\link[scales:label_percent]{scales::label_percent(suffix = engine_pick("\%", "\%", "\\\\\%"))}}.}

\item{.pvalue}{The function to apply to \code{x} when the template is \verb{\{.pvalue x\}}. Default is \code{\link[scales:label_pvalue]{scales::label_pvalue()}}.}

\item{.scientific}{The function to apply to \code{x} when the template is \verb{\{.scientific x\}}. Default is \code{\link[scales:label_scientific]{scales::label_scientific()}}.}

\item{.uppercase}{The function to apply to \code{x} when the template is \verb{\{.uppercase x\}} or \verb{\{.uc x\}}. Default is \code{\link[=toupper]{toupper()}}.}

\item{.lowercase}{The function to apply to \code{x} when the template is \verb{\{.lowercase x\}} or \verb{\{.lc x\}}. Default is \code{\link[=tolower]{tolower()}}.}

\item{.titlecase}{The function to apply to \code{x} when the template is \verb{\{.titlecase x\}} or \verb{\{.tc x\}}. Default is \code{function(x) tools::toTitleCase(as.character(x))}.}

\item{.sentence}{The function to apply to \code{x} when the template is \verb{\{.sentence x\}} or \verb{\{.sc x\}}. Default is \code{function(x) `substr<-`(x, 1, 1, toupper(substr(x, 1, 1)))}.}

\item{.squote}{The function to apply to \code{x} when the template is \verb{\{.squote x\}}. Default is \code{function(x) sQuote(x, q = getOption("epoxy.fancy_quotes", FALSE))}.}

\item{.dquote}{The function to apply to \code{x} when the template is \verb{\{.dquote x\}}. Default is \code{function(x) dQuote(x, q = getOption("epoxy.fancy_quotes", FALSE))}.}

\item{.strong}{The function to apply to \code{x} when the template is \verb{\{.strong x\}} or \verb{\{.bold x\}}. Default is chosen internally based on the output format.}

\item{.emph}{The function to apply to \code{x} when the template is \verb{\{.emph x\}} or \verb{\{.italic x\}}. Default is chosen internally based on the output format.}

\item{.code}{The function to apply to \code{x} when the template is \verb{\{.code x\}}. Default is chosen internally based on the output format.}
}
\value{
A function of \code{text} and \code{envir} suitable for the \code{.transformer} argument of
\code{\link[glue:glue]{glue::glue()}}.
}
\description{
This epoxy transformer is heavily inspired by the inline
formatters in the \href{https://cli.r-lib.org}{cli package}. The syntax is quite
similar, but \pkg{epoxy}'s syntax is slightly different to accommodate
reporting use cases.

To transform a template expression inline, you include a keyword, prefixed
with a dot (\code{.}) that is used to format the value of the template expression
in place.

\if{html}{\out{<div class="sourceCode r">}}\preformatted{epoxy("It cost \{.dollar 123456\}.", .transformer = "inline")
#> It cost $123,456.
}\if{html}{\out{</div>}}

The formatters, e.g. \code{.dollar} in the example above, can be customized using
the arguments of \code{epoxy_transform_inline()}. Pass a customized
\code{\link[scales:label_dollar]{scales::label_dollar()}} to \code{.dollar} to achieve a different transformation.

\if{html}{\out{<div class="sourceCode r">}}\preformatted{dollars_nzd <- scales::label_dollar(suffix = " NZD")

epoxy(
  "It cost \{.dollar 123456\}.",
  .transformer = epoxy_transform_inline(.dollar = dollars_nzd)
)
#> It cost $123,456 NZD.
}\if{html}{\out{</div>}}

Note that, unlike
\href{https://cli.r-lib.org/reference/inline-markup.html}{inline markup with cli},
the text within the template expression, other than the keyword, is treated
as an R expression.

\if{html}{\out{<div class="sourceCode r">}}\preformatted{money <- 123456
epoxy("It cost \{.dollar money\}.", .transformer = "inline")
#> It cost $123,456.
}\if{html}{\out{</div>}}

You can also nest inline markup expressions.

\if{html}{\out{<div class="sourceCode r">}}\preformatted{money <- c(123.456, 234.567)
epoxy("It will cost either \{.or \{.dollar money\}\}.", .transformer = "inline")
#> It will cost either $123.46 or $234.57.
}\if{html}{\out{</div>}}

Finally, you can provide your own functions that are applied to the evaluated
expression. In this example, I add a \code{.runif} inline formatter that generates
\code{n} random numbers (taken from the template expression) and sorts them.

\if{html}{\out{<div class="sourceCode r">}}\preformatted{set.seed(4242)

epoxy(
  "Here are three random percentages: \{.and \{.pct \{.runif 3\}\}\}.",
  .transformer = epoxy_transform_inline(
    .runif = function(n) sort(runif(n))
  )
)
#> Here are three random percentages: 23\%, 35\%, and 99\%.
}\if{html}{\out{</div>}}
}
\examples{
revenue <- 0.2123
sales <- 42000.134

# ---- Basic Example with Inline Formatting --------------------------------
epoxy(
  '{.pct revenue} of revenue generates {.dollar sales} in profits.'
)

# With standard {glue} (`epoxy_transform_inline()` is a glue transformer)
glue::glue(
  '{.pct revenue} of revenue generates {.dollar sales} in profits.',
  .transformer = epoxy_transform_inline()
)

# ---- Setting Inline Formatting Options ----------------------------------
# To set inline format options, provide `scales::label_*()` to the supporting
# epoxy_transform_inline arguments.
epoxy(
  '{.pct revenue} of revenue generates {.dollar sales} in profits.',
  .transformer = epoxy_transform_inline(
    .percent = scales::label_percent(accuracy = 0.1),
    .dollar = scales::label_dollar(accuracy = 10)
  )
)

glue::glue(
  '{.pct revenue} of revenue generates {.dollar sales} in profits.',
  .transformer = epoxy_transform_inline(
    .percent = scales::label_percent(accuracy = 0.1),
    .dollar = scales::label_dollar(accuracy = 10)
  )
)

# ---- Custom Inline Formatting ------------------------------------------
# Add your own formatting functions
search <- "why are cats scared of cucumbers"

epoxy_html(
  "https://example.com?q={{ .url_encode search }}>",
  .transformer = epoxy_transform_inline(
    .url_encode = utils::URLencode
  )
)
}
\seealso{
\code{\link[=epoxy_transform]{epoxy_transform()}}, \code{\link[=epoxy_transform_set]{epoxy_transform_set()}}

Other epoxy's glue transformers: 
\code{\link{epoxy_transform_html}()},
\code{\link{epoxy_transform}()}
}
\concept{epoxy's glue transformers}
