.evidence_mean_one_sample <- function(x, mu, sigma, level) {
  x <- stats::na.omit(x)
  n <- length(x)

  if (n < 2L) stop("At least two observations are required.")

  m <- mean(x)
  s <- if (is.null(sigma)) stats::sd(x) else sigma
  se <- s / sqrt(n)

  z <- stats::qnorm((1 + level) / 2)
  interval <- c(m - z * se, m + z * se)

  ll0 <- sum(stats::dnorm(x, mean = mu, sd = s, log = TRUE))
  ll1 <- sum(stats::dnorm(x, mean = m, sd = s, log = TRUE))

  log10_er <- (ll1 - ll0) / log(10)

  .evidence_result(
    estimate = m,
    interval = interval,
    interval_level = level,
    log10_er = log10_er,
    null = paste0("mean = ", mu),
    alternative = "mean != null",
    model = "one-sample normal mean",
    call = match.call()
  )
}

.evidence_mean_two_sample <- function(x, y, paired, level) {
  x <- stats::na.omit(x)
  y <- stats::na.omit(y)

  if (paired) {
    if (length(x) != length(y)) stop("Paired samples must have equal length.")
    return(.evidence_mean_one_sample(x - y, mu = 0, sigma = NULL, level = level))
  }

  nx <- length(x)
  ny <- length(y)

  mx <- mean(x)
  my <- mean(y)

  vx <- stats::var(x)
  vy <- stats::var(y)

  se <- sqrt(vx / nx + vy / ny)
  est <- mx - my

  z <- stats::qnorm((1 + level) / 2)
  interval <- c(est - z * se, est + z * se)

  ll0 <- sum(stats::dnorm(x, mean = mx, sd = sqrt(vx), log = TRUE)) +
         sum(stats::dnorm(y, mean = mx, sd = sqrt(vx), log = TRUE))
  
  ll1 <- sum(stats::dnorm(x, mean = mx, sd = sqrt(vx), log = TRUE)) +
         sum(stats::dnorm(y, mean = my, sd = sqrt(vy), log = TRUE))

  log10_er <- (ll1 - ll0) / log(10)

  .evidence_result(
    estimate = est,
    interval = interval,
    interval_level = level,
    log10_er = log10_er,
    null = "mean difference = 0",
    alternative = "mean difference != 0",
    model = "two-sample normal mean",
    call = match.call()
  )
}

