% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/fairness_check.R
\name{fairness_check}
\alias{fairness_check}
\title{Fairness check}
\usage{
fairness_check(
  x,
  ...,
  protected = NULL,
  privileged = NULL,
  cutoff = NULL,
  label = NULL,
  epsilon = 0.8,
  verbose = TRUE,
  colorize = TRUE
)
}
\arguments{
\item{x}{object created with \code{\link[DALEX]{explain}} or of class \code{fairness_object}.
It can be multiple fairness_objects, multiple explainers, or combination on both, as long as
they predict the same data. If at least one fairness_object is provided there is no need to
pass protected and privileged parameters. Explainers must be binary classification type.}

\item{...}{possibly more objects created with \code{\link[DALEX]{explain}} and/or objects of class \code{fairness_object}}

\item{protected}{factor, protected variable (also called sensitive attribute), containing privileged and unprivileged groups}

\item{privileged}{factor/character, one value of \code{protected}, in regard to what subgroup parity loss is calculated}

\item{cutoff}{numeric, vector of cutoffs (thresholds) for each value of protected variable, affecting only explainers.}

\item{label}{character, vector of labels to be assigned for explainers, default is explainer label.}

\item{epsilon}{numeric, boundary for fairness checking, lowest acceptable ratio of metrics between unprivileged and privileged subgroups. Default value is 0.8. More on the idea behind epsilon in details section.}

\item{verbose}{logical, whether to print information about creation of fairness object}

\item{colorize}{logical, whether to print information in color}
}
\value{
An object of class \code{fairness_object} which is a list with elements:
\item{parity_loss_metric_data}{A \code{data.frame} containing parity loss for various fairness metrics. The metrics include:
  \itemize{
    \item \strong{TPR}: True Positive Rate (Sensitivity, Recall)
    \item \strong{TNR}: True Negative Rate (Specificity)
    \item \strong{PPV}: Positive Predictive Value (Precision)
    \item \strong{NPV}: Negative Predictive Value
    \item \strong{FNR}: False Negative Rate
    \item \strong{FPR}: False Positive Rate
    \item \strong{FDR}: False Discovery Rate
    \item \strong{FOR}: False Omission Rate
    \item \strong{TS}: Threat Score
    \item \strong{STP}: Statistical Parity
    \item \strong{ACC}: Accuracy
    \item \strong{F1}: F1 Score
  }
}
\item{groups_data}{Metrics across levels in the protected variable.}
\item{groups_confusion_matrices}{Confusion matrices for each subgroup.}
\item{explainers}{A list of \code{DALEX} explainers used to create the object.}
\item{cutoffs}{A list of cutoffs for each explainer and subgroup.}
\item{fairness_check_data}{A \code{data.frame} used for plotting the \code{fairness_object}.}
\item{...}{Other parameters passed to the function.}
}
\description{
Fairness check creates \code{fairness_object} which measures different fairness metrics and wraps data, explainers and parameters in useful object. This is fundamental object in this package.
It enables to visualize fairness metrics and models in many ways and compare models on both fairness and performance level. Fairness check acts as merger and wrapper for explainers and fairness objects.
While other fairness objects values are not changed, fairness check assigns cutoffs and labels to provided explainers so same explainers with changed labels/cutoffs might be gradually added to fairness object.
Users through print and plot methods may quickly check values of most popular fairness metrics. More on that topic in details.
}
\details{
Fairness check

Metrics used are made for each subgroup, then base metric score is subtracted leaving loss of particular metric.
If absolute loss of metrics ratio is not within acceptable boundaries than such metric is marked as "not passed". It means that values of metrics should be within (epsilon, 1/epsilon) boundary.
The default ratio is set to 0.8 which adhere to US 80% rule (more on it here: \url{https://en.wikipedia.org/wiki/Disparate_impact#The_80%_rule}). It means that unprivileged subgroups should have at least 80%
score achieved in metrics by privileged subgroup. For example if TPR_unprivileged/TPR_privileged is less than 0.8 then such ratio is sign of discrimination. On the other hand if
TPR_privileged/TPR_unprivileged is more than 1.25 (1/0.8) than there is discrimination towards privileged group.
Epsilon value can be adjusted to user's needs. It should be interpreted as the lowest ratio of metrics allowed.  There are some metrics that might be derived from existing metrics (For example Equalized Odds - equal TPR and FPR for all subgroups).
That means passing 5 metrics in fairness check asserts that model is even more fair. In \code{fairness_check} models must always predict positive result. Not adhering to this rule
may lead to misinterpretation of the plot. More on metrics and their equivalents:
\url{https://fairware.cs.umass.edu/papers/Verma.pdf}
\url{https://en.wikipedia.org/wiki/Fairness_(machine_learning)}

Parity loss - visualization tool

Parity loss is computed as follows:
M_parity_loss = sum(abs(log(metric/metric_privileged)))

where:

M - some metric mentioned above

metric - vector of metric scores from each subgroup
metric_privileged - value of metric vector for privileged subgroup

base_metric - scalar, value of metric for base subgroup
}
\examples{
data("german")

y_numeric <- as.numeric(german$Risk) - 1

lm_model <- glm(Risk ~ .,
  data = german,
  family = binomial(link = "logit")
)

explainer_lm <- DALEX::explain(lm_model, data = german[, -1], y = y_numeric)

fobject <- fairness_check(explainer_lm,
  protected = german$Sex,
  privileged = "male"
)
plot(fobject)
\donttest{
rf_model <- ranger::ranger(Risk ~ .,
  data = german,
  probability = TRUE,
  max.depth = 3,
  num.trees = 100,
  seed = 1
)


explainer_rf <- DALEX::explain(rf_model,
  data = german[, -1],
  y = y_numeric
)

fobject <- fairness_check(explainer_rf, fobject)

plot(fobject)

# custom print
plot(fobject, fairness_metrics = c("ACC", "TPR"))
}

}
\references{
Zafar,Valera, Rodriguez, Gummadi (2017)  \url{https://arxiv.org/pdf/1610.08452.pdf}

Hardt, Price, Srebro (2016) \url{https://arxiv.org/pdf/1610.02413.pdf}

Verma, Rubin (2018) \url{https://fairware.cs.umass.edu/papers/Verma.pdf}

Barocas, Hardt, Narayanan (2019) \url{https://fairmlbook.org/}
}
