% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/ConcurReg.R
\name{ConcurReg}
\alias{ConcurReg}
\title{Functional Concurrent Regression using 2D smoothing}
\usage{
ConcurReg(
  vars,
  outGrid,
  userBwMu = NULL,
  userBwCov = NULL,
  kern = "gauss",
  measurementError = FALSE,
  diag1D = "all",
  useGAM = FALSE,
  returnCov = TRUE
)
}
\arguments{
\item{vars}{A list of input functional/scalar covariates.
Each field corresponds to a functional (a list) or scalar (a vector) covariate.
The last entry is assumed to be the response if no entry is named 'Y'.
If a field corresponds to a functional covariate, it should have two fields: 'Lt', a list of time points, and 'Ly', a list of functional values.}

\item{outGrid}{A vector of output time points.}

\item{userBwMu}{A scalar/vector bandwidth used for smoothing the mean function. Each entry in the vector represents the bandwidth used for the corresponding covariate in vars. For the scalar covariates, you can input 0 as a placeholder. If you only input a scalar, the function will use the same bandwidth to smooth all mean functions. --- a scalar/vector of positive numeric -
default: NULL --- if no scalar/vector value is provided, the bandwidth value for the smoothed mean function is chosen using 'GCV';}

\item{userBwCov}{A scalar/vector bandwidth used for smoothing the auto or cross-covariances. If you use 1D smoothing for the diagonal line of the covariance (diag1D="all"), only one scalar input is needed. If you use 2D smoothing for the covariance (diag1D="none"), a vector of bandwidth is required. Each entry in the vector represents the bandwidth used for the corresponding covariate in vars. For the scalar covariates, you can input 0 as a placeholder. --- a scalar/vector of positive numeric -
default: NULL --- if no scalar/vector is provided, the bandwidth value for the smoothed cross-covariance function is chosen using 'GCV';}

\item{kern}{Smoothing kernel choice, common for mu and covariance; "rect", "gauss", "epan", "gausvar", "quar" - default: "gauss".}

\item{measurementError}{Assume measurement error in the data; logical - default: FALSE. If TRUE the diagonal raw covariance will be removed when smoothing.}

\item{diag1D}{A string specifying whether to use 1D smoothing for the diagonal line of the covariance.
'none': don't use 1D smoothing; 'all': use 1D for both auto- and cross-covariances. (default : 'all')}

\item{useGAM}{Use GAM smoothing instead of local linear smoothing (semi-parametric option);  logical - default: FALSE.}

\item{returnCov}{Return the covariance surfaces, which is a four dimensional array. The first two dimensions correspond to outGrid
and the last two correspond to the covariates and the response, i.e. (i, j, k, l) entry being Cov(X_k(t_i), X_l(t_j));  logical - default: FALSE.}
}
\value{
A list containing the following fields:
\item{beta}{A matrix for the concurrent regression effects, where rows correspond to different predictors and columns to different time points.}
\item{beta0}{A vector containing the time-varying intercept.}
\item{outGrid}{A vector of the output time points.}
\item{cov}{A 4-dimensional array for the (cross-)covariance surfaces, with the (i, j, k, l) entry being Cov(X_k(t_i), X_l(t_j))}
\item{R2}{A vector of the time-varying R2.}
\item{n}{The sample size.}
}
\description{
Functional concurrent regression with dense or sparse functional data for scalar or functional dependent variables. Note: function-to-scalar regression can also be handled using the VCAM function in fdapace.
}
\details{
If measurement error is assumed, the diagonal elements of the raw covariance will be removed. This could result in highly unstable estimate if the design is very sparse, or strong seasonality presents.
WARNING! For very sparse functional data, setting measurementError = TRUE is not recommended.
}
\examples{
# Y(t) = beta_0(t) + beta_1(t) X_1(t) + beta_2(t) Z_2 + epsilon
#
# Settings
set.seed(1)
n <- 30
nGridIn <- 50
sparsity <- 5:10 # Sparse data sparsity
T <- round(seq(0, 1, length.out=nGridIn), 4) # Functional data support
bw <- 0.1
outGrid <- round(seq(min(T), 1, by=0.05), 2)
 outGrid <- seq(min(T), max(T), by=0.05)
# Simulate functional data
mu <- T * 2 # mean function for X_1
sigma <- 1

beta_0 <- 0
beta <- rbind(cos(T), 1.5 + sin(T))
beta_2 <- 1

Z <- MASS::mvrnorm(n, rep(0, 2), diag(2))
X_1 <- Z[, 1, drop=FALSE] \%*\% matrix(1, 1, nGridIn) + matrix(mu, n, nGridIn, byrow=TRUE)
epsilon <- rnorm(n, sd=sigma)
Y <- matrix(NA, n, nGridIn)
for (i in seq_len(n)) {
  Y[i, ] <- beta_0 + beta[1,]*X_1[i, ] + beta[2,]*Z[i, 2] + epsilon[i]
}

# Sparsify functional data
set.seed(1)
X_1sp <- fdapace::Sparsify(X_1, T, sparsity)
set.seed(1)
Ysp <- fdapace::Sparsify(Y, T, sparsity)
vars <- list(X_1=X_1sp, Z_2=Z[, 2], Y=Ysp)
res2 <- ConcurReg(vars, outGrid, userBwMu=c(.5,0,.5), userBwCov=c(.5,0,.5), kern='gauss',
 measurementError=TRUE, diag1D='none', useGAM = FALSE, returnCov=TRUE)
}
\references{
\itemize{
\item \cite{Yao, F., Müller, H.G., Wang, J.L. "Functional Linear Regression Analysis for Longitudinal Data." Annals of Statistics 33, (2005): 2873-2903.(Dense data)}
\item \cite{Sentürk, D., Müller, H.G. "Functional varying coefficient models for longitudinal data." J. American Statistical Association, 10, (2010): 1256--1264.}
\item \cite{Sentürk, D., Nguyen, D.V. "Varying Coefficient Models for Sparse Noise-contaminated Longitudinal Data", Statistica Sinica 21(4), (2011): 1831-1856. (Sparse data)}
}
}
