#' MFL players library
#'
#' A cached table of MFL players. Will store in memory for each session!
#' (via memoise in zzz.R)
#'
#' @param conn optionally, pass in a conn object generated by ff_connect to receive league-specific custom players
#'
#' @examples
#' \donttest{
#' try({ # try only shown here because sometimes CRAN checks are weird
#'   player_list <- mfl_players()
#'   dplyr::sample_n(player_list, 5)
#' }) # end try
#' }
#'
#' @return a dataframe containing all ~2000+ players in the MFL database
#'
#' @export

mfl_players <- function(conn = NULL) {
  if (!is.null(conn) && !inherits(conn, "mfl_conn")) {
    stop("conn must be generated by 'mfl_connect()' and have type 'mfl_conn'")
  }

  if (is.null(conn)) {
    conn <- mfl_connect(.fn_choose_season())
  }

  conn %>%
    mfl_getendpoint("players", DETAILS = 1) %>%
    purrr::pluck("content", "players", "player") %>%
    tibble::tibble() %>%
    tidyr::unnest_wider(1) %>%
    dplyr::mutate_at(
      "birthdate", ~ as.numeric(.x) %>%
        .as_datetime() %>%
        .as_date()
    ) %>%
    dplyr::mutate("age" = round(as.numeric(Sys.Date() - .data$birthdate) / 365.25, 1)) %>%
    dplyr::select(
      dplyr::any_of(c(
        "player_id" = "id",
        "player_name" = "name",
        "pos" = "position",
        "age",
        "team",
        "status"
      )),
      dplyr::starts_with("draft_"),
      dplyr::ends_with("_id"),
      dplyr::everything()
    )
}
