% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/format_numbers.R
\name{format_numbers}
\alias{format_numbers}
\title{Format numbers}
\usage{
format_numbers(
  x,
  digits = 4,
  format = "engr",
  ...,
  omit_power = c(-1, 2),
  set_power = NULL,
  delim = formatdown_options("delim"),
  size = formatdown_options("size"),
  decimal_mark = formatdown_options("decimal_mark"),
  big_mark = formatdown_options("big_mark"),
  big_interval = formatdown_options("big_interval"),
  small_mark = formatdown_options("small_mark"),
  small_interval = formatdown_options("small_interval"),
  whitespace = formatdown_options("whitespace")
)
}
\arguments{
\item{x}{Number or numbers to be formatted. Can be a single number, a vector,
or a column of a data frame.}

\item{digits}{Integer from 1 through 20 that controls the number of
significant digits in printed numeric values. Passed to \code{signif()}.
Default is 4.}

\item{format}{Character, length 1, defines the type of notation. Possible
values are \code{"engr"} (default) for engineering power-of-ten notation,
\code{"sci"} for scientific power-of-ten notation, and \code{"dcml"} for decimal
notation.}

\item{...}{Not used for values; forces subsequent arguments to be
referable only by name.}

\item{omit_power}{Numeric vector \code{c(p, q)} with \code{p <= q}, specifying
the range of exponents over which power-of-ten notation is omitted in
either scientific or engineering format. Default is
\code{c(-1, 2)}. If a single value is assigned, i.e., \code{omit_power = p}, the
argument is interpreted as \code{c(p, p)}. If \code{NULL} or \code{NA}, all elements
are formatted in power-of-ten notation. Argument is overridden by
specifying \code{set_power} or decimal notation.}

\item{set_power}{Integer, length 1. Formats all values in \code{x} with the same
power-of-ten exponent. Default NULL. Overrides \code{format} and \code{omit_power}
arguments.}

\item{delim}{Character, length 1 or 2, to define the left and right math
markup delimiters. The default setting, \code{delim = "$"}, produces
left and right delimiters \verb{$...$}. The alternate built-in
setting, \code{delim = "\\("}, produces left and right delimiters
\verb{\\\\( ... \\\\)}. Custom delimiters can be assigned in a vector of
length 2 with left and right delimiter symbols, e.g.,
\verb{c("\\\[", "\\\]")}. Special characters typically must be escaped.}

\item{size}{Character, length 1, to assign a font size. If not empty, adds
a font size macro to the markup inside the math delimiters. Possible
values are \code{"scriptsize"}, \code{"small"}, \code{"normalsize"}, \code{"large"}, and
\code{"huge"}. One may also assign the equivalent LaTeX-style markup itself,
e.g., \code{"\\\\scriptsize"}, \code{"\\\\small"}, etc. Default is NULL.}

\item{decimal_mark}{Character, length 1, to assign the decimal marker.
Possible values are a period \code{"."} (default) or a comma \code{","}. Passed
to \code{formatC(decimal.mark)}.}

\item{big_mark}{Character, length 1, used as the mark between every
\code{big_interval} number of digits to the left of the decimal marker to
improve readability. Possible values are empty \code{""} (default) or
\code{"thin"} to produce a LaTeX-style thin, horizontal space. One may also
assign the thin-space markup itself \code{"\\\\\\\\,"}.
Passed to \code{formatC(big.mark)}.}

\item{big_interval}{Integer, length 1, that defines the number of digits
(default 3) in groups separated by \code{big_mark}. Passed to
\code{formatC(big.interval)}.}

\item{small_mark}{Character, length 1, used as the mark between every
\code{small_interval} number of digits to the right of the decimal marker to
improve readability. Possible values are empty \code{""} (default) or
\code{"thin"} to produce a LaTeX-style thin, horizontal space. One may also
assign the thin-space markup itself \code{"\\\\\\\\,"}.
Passed to \code{formatC(small.mark)}.}

\item{small_interval}{Integer, length 1, that defines the number of digits
(default 5) in groups separated by \code{small_mark}. Passed to
\code{formatC(small.interval)}.}

\item{whitespace}{Character, length 1, to define the LaTeX-style
math-mode macro to preserve a horizontal space between words of text or
between physical-unit abbreviations when formatting numbers of class
"units". Default is \code{"\\\\\\\\>"}. Alternatives include \code{"\\\\\\\\:"} or
"\verb{\\\\\\\\ }".}
}
\value{
A character vector in which numbers are formatted in power-of-ten
or decimal notation and delimited for rendering as inline equations
in an R markdown document.
}
\description{
Convert a numeric vector to a character vector in which the numbers are
formatted in power-of-ten notation in scientific or engineering form and
delimited for rendering as inline equations in an R markdown document.
Decimal numbers can be similarly formatted, without the power-of-ten
notation.
}
\details{
Given a number, a numerical vector, or a numerical column from a data frame,
\code{format_numbers()} converts the numbers to character strings of the form,
\code{"$a \\\\times 10^{n}$"}, where \code{a} is the coefficient to a specified
number of significant digits and \code{n} is the exponent. When used for decimal
notation, \code{format_numbers()} converts numbers to character strings of the
form \code{"$a$"}.

Powers-of-ten notation is omitted over a range of exponents via \code{omit_power}
such that numbers so specified are converted to decimal notation. For
example, the default \code{omit_power = c(-1, 2)} formats numbers such as 0.123,
1.23, 12.3, and 123 in decimal form. To cancel these exceptions and convert
all numbers to powers-of-ten notation, set the \code{omit_power} argument to NULL
or NA.

Delimiters for inline math markup can be edited if necessary. If the default
argument fails, try using \code{"\\\\("} as an alternative. If using a custom
delimiter to suit the markup environment, be sure to escape all special
symbols.

When inputs are of class "units" (created with the units package), a
math-text macro of the form \verb{\\\\mathrm\{<units_string>\}} is appended
to the formatted numerical value inside the math delimiters.

Arguments after the dots (\code{...}) must be referred to by name.
}
\examples{
# input: single number
x <- 6.0221E+23
format_numbers(x)

# input: units class
x <- 103400
units(x) <- "N m2 C-2"
format_numbers(x)

# input: vector
data("metals", package = "formatdown")
x <- metals$dens
format_numbers(x)

# significant digits
x <- 9.75358e+5
format_numbers(x, 2)
format_numbers(x, 3)
format_numbers(x, 4)

# format & wrappers: format_engr(), format_sci(), format_dcml()
x <- 6.0221E+23
format_numbers(x, format = "engr")
format_engr(x)

format_numbers(x, format = "sci")
format_sci(x)

x <- 103400
format_numbers(x, format = "dcml")
format_dcml(x)

# input: data frame
x <- metals[, c("thrm_exp", "thrm_cond")]
as.data.frame(apply(x, 2, format_sci, digits = 3))

# omit_power
x <- 103400
format_sci(x, omit_power = c(-1, 2)) # default
format_sci(x, omit_power = c(-1, 5))
format_sci(x, omit_power = 5) # equivalent to omit_power = c(5, 5)
x <- 1.2
format_sci(x, omit_power = NULL)

# set_power
format_sci(x, set_power = NULL) # default
format_sci(x, set_power = 3)

# set_power overrides format
x <- 6.0221E+23
format_engr(x)
format_engr(x, set_power = 24L)
format_sci(x)
format_sci(x, set_power = 24L)

# set_power overrides omit_power
x <- 101300
format_sci(x, omit_power = 5)
format_sci(x, omit_power = 5, set_power = 2)
format_sci(x, omit_power = 2)
format_sci(x, omit_power = 2, set_power = 2)

# decimal format ignores set_power
x <- 103400
format_numbers(x, format = "dcml")
format_numbers(x, format = "dcml", set_power = 3)

}
\seealso{
Other format_*: 
\code{\link{format_dcml}()},
\code{\link{format_engr}()},
\code{\link{format_sci}()},
\code{\link{format_text}()}
}
\concept{format_*}
