context("Correct inputs to functions")
library(geodiv)
data(normforest)
normforest <- terra::unwrap(normforest)
ba_func <- bearing_area(normforest)

test_that("incorrect inputs return errors in aacf", {
  expect_error(aacf('a'), 'x must be a raster or matrix.')
  expect_error(scl('a'), 'x must be a raster or matrix.')
  expect_error(stxr('a'), 'x must be a raster or matrix.')
  expect_error(scl(normforest, threshold = -0.1, create_plot = FALSE), 'threshold values cannot be less than 0.')
  expect_error(scl(normforest, threshold = 'a', create_plot = FALSE), 'threshold must be numeric.')
  expect_error(scl(normforest, create_plot = 'a'), 'create_plot argument must be TRUE/FALSE.')
  expect_error(stxr(normforest, threshold = -0.1), 'threshold values cannot be less than 0.')
  expect_error(stxr(normforest, threshold = 'a'), 'threshold must be numeric.')
})

test_that("aacf returns correct value", {
  expect_that(dim(aacf(normforest)), equals(c(371, 371, 1)))
})

test_that("incorrect inputs return errors in basics", {
  expect_error(sa('a'), 'x must be a raster or matrix.')
  expect_error(sq('a'), 'x must be a raster or matrix.')
  expect_error(ssk('a'), 'x must be a raster or matrix.')
  expect_error(sku('a'), 'x must be a raster or matrix.')
  expect_error(sv('a'), 'x must be a raster or matrix.')
  expect_error(sph('a'), 'x must be a raster or matrix.')
  expect_error(smean('a'), 'x must be a raster or matrix.')
  expect_error(ssk(normforest, adj = 'a'), 'adj argument must be TRUE/FALSE.')
  expect_error(sku(normforest, excess = 'a'), 'excess argument must be TRUE/FALSE.')
})

test_that("incorrect inputs return errors in bearingarea", {
  expect_error(bearing_area('a'), 'x must be a raster or matrix.')
  expect_error(plot_ba_curve('a'), 'x must be a raster or matrix.')
  expect_error(plot_ba_curve(normforest, divisions = 'a'), 'divisions argument must be TRUE/FALSE.')
  expect_error(find_flat('a'), 'x must be a raster or matrix.')
  expect_error(find_flat(normforest, perc = 'a'), 'perc must be numeric.')
  expect_error(find_flat(normforest, perc = c(0.1, 0.2)), 'too many values supplied to perc.')
  expect_error(find_flat(normforest, perc = 8), 'perc must be between 0 and 1.')
  expect_error(height_ba('a'), 'x must be a raster or matrix.')
  expect_error(height_ba(normforest, xval = 'a'), 'xval must be numeric.')
  expect_error(height_ba(normforest, xval = c(0.1, 0.2)), 'too many values supplied to xval.')
  expect_error(height_ba(normforest, xval = 8), 'xval must be between 0 and 1.')
  expect_error(sdc('a', low = 0.1, high = 0.4), 'x must be a raster or matrix.')
  expect_error(sdc(normforest, low = 'a', high = 0.4), 'low value must be numeric.')
  expect_error(sdc(normforest, low = 0.1, high = 'a'), 'high value must be numeric.')
  expect_error(sdc(normforest, low = c(0.1, 0.2), high = 0.4), 'too many values supplied to low.')
  expect_error(sdc(normforest, low = 0.1, high = c(0.3, 0.4)), 'too many values supplied to high.')
  expect_error(sdc(normforest, low = 8, high = 10), 'low value must be between 0 and 1.')
  expect_error(sdc(normforest, low = 0.1, high = 10), 'high value must be between 0 and 1.')
  expect_error(sdc(normforest, low = 0.4, high = 0.1), 'high value must be greater than low value.')
  expect_error(sbi('a'), 'x must be a raster or matrix.')
  expect_error(svi('a'), 'x must be a raster or matrix.')
  expect_error(sci('a'), 'x must be a raster or matrix.')
  expect_error(sk('a'), 'x must be a raster or matrix.')
  expect_error(svk('a'), 'x must be a raster or matrix.')
  expect_error(spk('a'), 'x must be a raster or matrix.')
})

test_that("incorrect inputs return errors in bestfit", {
  expect_error(slopecalc('a', h = 0.001, f = bearing_area(normforest)), 'x must be numeric.')
  expect_error(slopecalc(seq(0, 1, 1000), h = 'a', f = bearing_area(normforest)), 'h must be numeric.')
  expect_error(slopecalc(seq(0, 1, 1000), h = 0.001, f = 'a'), 'f was not produced with bearing_area function.')
  expect_error(slopecalc(seq(0, 1, 1000), h = c(0.001, 0.002), f = bearing_area(normforest)), 'too many values for h.')
  expect_error(slopecalc(seq(0, 1, 1000), h = -0.001, f = bearing_area(normforest)), 'h must be less than 1 and greater than 0.')
  expect_error(slopemeans(slopes = 'a', l = 0.4), 'slopes must be a dataframe.')
  expect_error(slopemeans(slopes = data.frame(slope = 1, x = 1), l = 'a'), 'l must be numeric.')
  expect_error(slopemeans(slopes = data.frame(slope = 1, x = 1), l = c(0.3, 0.4)), 'too many values for l.')
  expect_error(slopemeans(slopes = data.frame(slope = 1, x = 1), l = 1.2), 'l must be less than 1 and greater than 0.')
  expect_error(slopemeans(slopes = data.frame(wrong = 1, x = 1), l = 0.4), 'incorrect column names for slopes dataframe -- need slope and x.')
})

test_that("incorrect inputs return errors in fftshift", {
  expect_error(fftshift('a', dim = -1), 'x must be a matrix.')
  expect_error(fftshift(fft(terra::as.matrix(normforest)), dim = 'a'), 'dim must be numeric.')
  expect_error(fftshift(fft(terra::as.matrix(normforest)), dim = c(-1, 1)), 'too many values provided for dim.')
  expect_error(fftshift(base::Re(stats::fft(terra::as.matrix(normforest))), dim = 8), 'invalid value for dim -- must be -1, 1, or 2.')
})

test_that("incorrect inputs return errors in fourier", {
  expect_error(std('a', create_plot = FALSE), 'x must be a raster or matrix.')
  expect_error(std(normforest, create_plot = 'a'), 'create_plot must be logical.')
  expect_error(srw('a', create_plot = FALSE), 'x must be a raster or matrix.')
  expect_error(srw(normforest, create_plot = 'a'), 'create_plot must be logical.')
})

test_that("incorrect inputs return errors in localsurface", {
  expect_error(findpeaks('a'), 'x must be a raster or matrix.')
  expect_error(findvalleys('a'), 'x must be a raster or matrix.')
  expect_error(ssc('a'), 'x must be a raster or matrix.')
  expect_error(sds('a'), 'x must be a raster or matrix.')
  expect_error(s10z('a'), 'x must be a raster or matrix.')
})

test_that("incorrect inputs return errors in sdq", {
  expect_error(sdq('a'), 'x must be a raster or matrix.')
  expect_error(sdq6('a'), 'x must be a raster or matrix.')
})

test_that("incorrect inputs return errors in simpsons", {
  expect_error(area_above(f = 'a', a = 0.1, b = 0.4, n = 100), 'f must be a function.')
  expect_error(area_above(f = bearing_area(normforest), a = 'a', b = 0.4, n = 100), 'a must be numeric.')
  expect_error(area_above(f = bearing_area(normforest), a = 0.1, b = 'a', n = 100), 'b must be numeric.')
  expect_error(area_above(f = bearing_area(normforest), a = 0.1, b = 0.4, n = 'a'), 'n must be numeric.')
  expect_error(area_above(f = bearing_area(normforest), a = c(0.1, 0.2), b = 0.4, n = 100), 'too many values supplied to a.')
  expect_error(area_above(f = bearing_area(normforest), a = 0.1, b = c(0.4, 0.5), n = 100), 'too many values supplied to b.')
  expect_error(area_above(f = bearing_area(normforest), a = 0.1, b = 0.4, n = c(100, 200)), 'too many values supplied to n.')
  expect_error(area_above(f = bearing_area(normforest), a = 0.1, b = 0.4, n = -100), 'n must be greater than 0.')
  expect_error(area_above(f = bearing_area(normforest), a = 0.4, b = 0.1, n = 100), 'b must be greater than a.')
  expect_error(simpsons(f = 'a', a = 0.1, b = 0.4, n = 100), 'f must be a function.')
  expect_error(simpsons(f = bearing_area(normforest), a = 'a', b = 0.4, n = 100), 'a must be numeric.')
  expect_error(simpsons(f = bearing_area(normforest), a = 0.1, b = 'a', n = 100), 'b must be numeric.')
  expect_error(simpsons(f = bearing_area(normforest), a = 0.1, b = 0.4, n = 'a'), 'n must be numeric.')
  expect_error(simpsons(f = bearing_area(normforest), a = c(0.1, 0.2), b = 0.4, n = 100), 'too many values supplied to a.')
  expect_error(simpsons(f = bearing_area(normforest), a = 0.1, b = c(0.4, 0.5), n = 100), 'too many values supplied to b.')
  expect_error(simpsons(f = bearing_area(normforest), a = 0.1, b = 0.4, n = c(100, 200)), 'too many values supplied to n.')
  expect_error(simpsons(f = bearing_area(normforest), a = 0.1, b = 0.4, n = -100), 'n must be greater than 0.')
  expect_error(simpsons(f = bearing_area(normforest), a = 0.4, b = 0.1, n = 100), 'b must be greater than a.')
})

test_that("incorrect inputs return errors in surfacearea", {
  expect_error(flatsa('a'), 'x must be a raster or matrix.')
  expect_error(surface_area('a'), 'x must be a raster or matrix.')
  expect_error(sdr('a'), 'x must be a raster or matrix.')
})

test_that("incorrect inputs return errors in utilities", {
  expect_error(fitplane('a', order = 1), 'x must be a raster or matrix.')
  expect_error(fitplane(normforest, order = 'a'), 'order must be numeric or integer.')
  expect_error(fitplane(normforest, order = c(1, 2)), 'too many values supplied to order.')
  expect_error(fitplane(normforest, order = -1), 'order must be >= 0.')
  expect_error(bestfitplane('a'), 'x must be a raster or matrix.')
  expect_error(remove_plane('a'), 'x must be a raster or matrix.')
})

test_that("incorrect inputs return errors in zshift", {
  expect_error(zshift('a', xdist = 1, ydist = 1, scale = FALSE), 'r must be a raster.')
  expect_error(zshift(normforest, xdist = 'a', ydist = 1, scale = FALSE), 'xdist must be numeric.')
  expect_error(zshift(normforest, xdist = 1, ydist = 'a', scale = FALSE), 'ydist must be numeric.')
  expect_error(zshift(normforest, xdist = 1, ydist = 1, xrm = 'a', scale = FALSE), 'xrm must be numeric.')
  expect_error(zshift(normforest, xdist = 1, ydist = 1, yrm = 'a', scale = FALSE), 'yrm must be numeric.')
  expect_error(zshift(normforest, xdist = 1, ydist = 1, scale = 'a'), 'scale must be logical.')
  expect_error(zshift(normforest, xdist = c(1, 2), ydist = 1, scale = FALSE), 'too many values supplied to xdist.')
  expect_error(zshift(normforest, xdist = 1, ydist = c(1, 2), scale = FALSE), 'too many values supplied to ydist.')
})

test_that("incorrect inputs return errors in sfd", {
  expect_error(sfd(1), 'x must be a raster or matrix.')
})
