% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/scenarios-fr.R
\name{sim-frmfr}
\alias{sim-frmfr}
\alias{r_frm_fr}
\alias{nl_dev}
\title{Sampling functional regression models with functional responses}
\usage{
r_frm_fr(n, scenario = 3, X_fdata = NULL, error_fdata = NULL,
  beta = NULL, s = seq(0, 1, l = 101), t = seq(0, 1, l = 101),
  std_error = 0.15, nonlinear = NULL, concurrent = FALSE,
  int_rule = "trapezoid", n_fpc = 50, verbose = FALSE, ...)

nl_dev(X_fdata, t = seq(0, 1, l = 101), nonlinear = NULL,
  int_rule = "trapezoid", equispaced = equispaced, verbose = FALSE)
}
\arguments{
\item{n}{sample size, only required when \code{scenario} is given.}

\item{scenario}{an index from \code{1} to \code{3} (default) denoting
one of the scenarios (S1, S2 or S3) simulated in
García-Portugués et al. (2021) (see details below). If
\code{scenario = NULL}, \code{X_fdata}, \code{error_fdata}, and \code{beta}
have to be provided. Otherwise, \code{X_fdata}, \code{error_fdata}, and
\code{beta} will be ignored.}

\item{X_fdata}{sample of functional covariates \eqn{X(s)} as
\code{\link[fda.usc]{fdata}} objects of length \code{n}, with \eqn{s} in
\eqn{[a, b]}. Defaults to \code{NULL}.}

\item{error_fdata}{sample of functional errors \eqn{\varepsilon(t)} as
\code{\link[fda.usc]{fdata}} objects of length \code{n}, with \eqn{t} in
\eqn{[c, d]}. If \code{concurrent = TRUE}, \code{X_fdata} and
\code{error_fdata} must be valued in the same grid. Defaults to \code{NULL}.}

\item{beta}{matrix containing the values \eqn{\beta(s, t)}, for each grid
point \eqn{s} in \eqn{[a, b]} and \eqn{t} in \eqn{[c, d]}. If
\code{concurrent = TRUE} (see details below), a row/column vector
must be introduced, valued in the same grid as \code{error_fdata}.
If \code{beta = NULL} (default), \code{scenario != NULL} is required.}

\item{s, t}{grid points. If \code{X_fdata}, \code{error_fdata} and
\code{beta} are provided, \code{s} and \code{t} are ignored. Default to
\code{s = seq(0, 1, l = 101)} and \code{t = seq(0, 1, l = 101)},
respectively.}

\item{std_error}{standard deviation of the random variables
involved in the generation of the functional error \code{error_fdata}.
Defaults to \code{0.15}.}

\item{nonlinear}{nonlinear term. Either a character string (\code{"exp"},
\code{"quadratic"} or \code{"sin"}) or an \code{\link[fda.usc]{fdata}}
object of length \code{n}, valued in the same grid as \code{error_fdata}.
If \code{nonlinear = NULL} (default), the nonlinear part is set to zero.}

\item{concurrent}{flag to consider a concurrent FLRFR (degenerate case).
Defaults to \code{FALSE}.}

\item{int_rule}{quadrature rule for approximating the definite
unidimensional integral: trapezoidal rule (\code{int_rule = "trapezoid"})
and extended Simpson rule (\code{int_rule = "Simpson"}) are available.
Defaults to \code{"trapezoid"}.}

\item{n_fpc}{number of components to be considered for the generation of
functional variables. Defaults to \code{50}.}

\item{verbose}{flag to display information about the sampling procedure.
Defaults to \code{FALSE}.}

\item{...}{further parameters passed to
\code{\link[goffda:elem-flmfr]{r_cm2013_flmfr}},
\code{\link[goffda:elem-flmfr]{r_gof2021_flmfr}} and\cr
\code{\link[goffda:elem-flmfr]{r_ik2018_flmfr}}, depending on the
chosen \code{scenario}.}

\item{equispaced}{flag to indicate if \code{X_fdata$data} is valued in
an equispaced grid or not. Defaults to \code{FALSE}.}
}
\value{
A list with the following elements:
\item{\code{X_fdata}}{functional covariates, an
\code{\link[fda.usc]{fdata}} object of length \code{n}.}
\item{\code{Y_fdata}}{functional responses, an
\code{\link[fda.usc]{fdata}} object of length \code{n}.}
\item{\code{error_fdata}}{functional errors, an
\code{\link[fda.usc]{fdata}} object of length \code{n}.}
\item{\code{beta}}{either the matrix with \eqn{\beta(s, t)} evaluated at
the \code{argvals} of \code{X_fdata} and \code{Y_fdata} (if
\code{concurrent = FALSE}) or a vector with \eqn{\beta(t)}
evaluated at the \code{argvals} of \code{X_fdata} (if
\code{concurrent = TRUE}).}
\item{\code{nl_dev}}{nonlinear term, an \code{\link[fda.usc]{fdata}}
object of length \code{n}.}
}
\description{
Simulation of a Functional Regression Model with Functional
Response (FRMFR) comprised of an additive mix of a linear and nonlinear
terms:
\deqn{Y(t) = \int_a^b X(s) \beta(s,t) ds + \Delta(X)(t) + \varepsilon(t),}{
Y(t) = \int_a^b X(s) \beta(s,t) ds + \Delta(X)(t) + \epsilon(t),}
where \eqn{X} is a random variable in the Hilbert space of
square-integrable functions in \eqn{[a, b]}, \eqn{L^2([a, b])},
\eqn{\beta} is the bivariate kernel of the FRMFR,
\eqn{\varepsilon}{\epsilon} is a random variable in \eqn{L^2([c, d])},
and \eqn{\Delta(X)}{\Delta(X)} is a nonlinear term.

In particular, the scenarios considered in García-Portugués et al. (2021)
can be easily simulated.
}
\details{
\itemize{
  \item{\code{r_frm_fr} samples the above regression model,
  where the nonlinear term \eqn{\Delta(X)} is computed by \code{nl_dev}.
  Functional covariates, errors, and \eqn{\beta} are generated
  automatically from the scenarios in García-Portugués et al. (2021) when
  \code{scenario != NULL} (see the documentation of
  \code{\link{r_gof2021_flmfr}}). If \code{scenario = NULL},
  covariates, errors and \eqn{\beta} must be provided.

  When \code{concurrent = TRUE}, the concurrent FRMFR
  \deqn{Y(t) = X(t) \beta(t) +
  \Delta(X)(t) + \varepsilon(t)}{Y(t) = X(t) * \beta(t)
  + \Delta(X)(t) + \epsilon(t)}
  is considered.}
  \item{\code{nl_dev} computes a nonlinear deviation
  \eqn{\Delta(X)}{\Delta(X)}:
  \eqn{\exp(\sqrt{X(a + (t - c) ((b - a) / (d - c)))})}{
  exp(\sqrt(X(a + (t - c) * ((b - a) / (d - c)))))}
  (for \code{"exp"}),
  \eqn{(X^2 (a + (t - c) ((b - a) / (d - c))) - 1)}{
  (X^2 * (a + (t - c) * ((b - a) / (d - c))) - 1)}
  (\code{"quadratic"}) or
  \eqn{(\sin(2\pi t) - \cos(2 \pi t)) \| X \|^2}{
  (sin(2 \pi t) - cos(2 \pi t)) * ||X||^2}
  (\code{"sin"}). Also, \eqn{\Delta(X)} can be manually set as an
  \code{\link[fda.usc]{fdata}} object of length \code{n} and valued in
  the same grid as \code{error_fdata}.}
}
}
\examples{
## Generate samples for the three scenarios

# Equispaced grids and Simpson's rule

s <- seq(0, 1, l = 101)
samp <- list()
old_par <- par(mfrow = c(3, 5))
for (i in 1:3) {
  samp[[i]] <- r_frm_fr(n = 100, scenario = i, s = s, t = s,
                        int_rule = "Simpson")
  plot(samp[[i]]$X_fdata)
  plot(samp[[i]]$error_fdata)
  plot(samp[[i]]$Y_fdata)
  plot(samp[[i]]$nl_dev)
  image(x = s, y = s, z = samp[[i]]$beta, col = viridisLite::viridis(20))
}
par(old_par)

## Linear term as a concurrent model

# The grids must be have the same number of grid points for a given
# nonlinear term and a given beta function

s <- seq(1, 2, l = 101)
t <- seq(0, 1, l = 101)
samp_c_1 <- r_frm_fr(n = 100, scenario = 3, beta = sin(t) - exp(t),
                     s = s, t = t, nonlinear = fda.usc::fdata(mdata =
                       t(matrix(rep(sin(t), 100), nrow = length(t))),
                       argvals = t),
                     concurrent = TRUE)
old_par <- par(mfrow = c(3, 2))
plot(samp_c_1$X_fdata)
plot(samp_c_1$error_fdata)
plot(samp_c_1$Y_fdata)
plot(samp_c_1$nl_dev)
plot(samp_c_1$beta)
par(old_par)

## Sample for given X_fdata, error_fdata, and beta

# Non equispaced grids with sinusoidal nonlinear term and intensity 0.5
s <- c(seq(0, 0.5, l = 50), seq(0.51, 1, l = 101))
t <- seq(2, 4, len = 151)
X_fdata <- r_ou(n = 100, t = s, alpha = 2, sigma = 4, x0 = 1:100)
error_fdata <- r_ou(n = 100, t = t, alpha = 1, sigma = 1, x0 = 1:100)
beta <- r_gof2021_flmfr(n = 100, s = s, t = t)$beta
samp_Xeps <- r_frm_fr(scenario = NULL, X_fdata = X_fdata,
                      error_fdata = error_fdata, beta = beta,
                      nonlinear = "exp", int_rule = "trapezoid")
old_par <- par(mfrow = c(3, 2))
plot(samp_Xeps$X_fdata)
plot(samp_Xeps$error_fdata)
plot(samp_Xeps$Y_fdata)
plot(samp_Xeps$nl_dev)
image(x = s, y = t, z = beta, col = viridisLite::viridis(20))
par(old_par)
}
\references{
García-Portugués, E., Álvarez-Liébana, J., Álvarez-Pérez, G. and
Gonzalez-Manteiga, W. (2021). A goodness-of-fit test for the functional
linear model with functional response. \emph{Scandinavian Journal of
Statistics}, 48(2):502--528. \doi{10.1111/sjos.12486}
}
\author{
Javier Álvarez-Liébana.
}
