% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/plot_graphTable.R
\name{plot_graphTable}
\alias{plot_graphTable}
\title{Generate benchmarking graphics in a PDF file}
\usage{
plot_graphTable(
  graphTable,
  pdf_file,
  ori_plot_flag = TRUE,
  adj_plot_flag = TRUE,
  GR_plot_flag = TRUE,
  GR_table_flag = FALSE,
  add_bookmarks = TRUE
)
}
\arguments{
\item{graphTable}{(mandatory)

Data frame (object of class "data.frame") corresponding to the benchmarking function output\code{graphTable} data frame.}

\item{pdf_file}{(mandatory)

Name (and path) of the PDF file that will contain the benchmarking graphics. The name should include the ".pdf"
file extension. The PDF file is created in the R session working directory (as returned by \code{getwd()}) if a path
is not specified. Specifying \code{NULL} would cancel the creation of a PDF file.}

\item{ori_plot_flag, adj_plot_flag, GR_plot_flag, GR_table_flag}{(optional)

Logical arguments indicating whether or not the corresponding type of benchmarking graphic should be generated.
All three plots are generated by default but not the growth rates tables.

\strong{Default values} are \code{ori_plot_flag = TRUE}, \code{adj_plot_flag = TRUE}, \code{GR_plot_flag = TRUE} and
\code{GR_table_flag = FALSE}.}

\item{add_bookmarks}{Logical argument indicating whether or not bookmarks should be added to the PDF file. See \strong{Bookmarks} in section
\strong{Details} for more information.

\strong{Default value} is \code{add_bookmarks = TRUE}.}
}
\value{
In addition to creating a PDF file containing the benchmarking graphics (except when \code{pdf_file = NULL}), this function
also invisibly returns a list with the following elements:
\itemize{
\item \code{graphTable}: Character string (character vector of length one) that contains the complete name and path of the PDF
file if it was successfully created and \code{invisible(NA_character_)} otherwise or if \code{pdf_file = NULL} was specified.
\item \code{graph_list}: List of the generated benchmarking graphics (one per series) with the following elements:
\itemize{
\item \code{name}: Character string describing the series (matches the bookmark name in the PDF file).
\item \code{page}: Integer representing the sequence number of the first graphic for the series in the entire sequence of
graphics for all series (matches the page number in the PDF file).
\item \code{ggplot_list}: List of ggplot objects (one per graphic or page in the PDF file) corresponding to the generated
benchmarking graphics for the series. See section \strong{Value} in \link{bench_graphs} for details.
}
}

Note that the returned ggplot objects can be displayed \emph{manually} with \code{\link[=print]{print()}}, in which case some updates to the
ggplot2 theme defaults are recommended in order to produce graphics with a similar look and feel as those generated in
the PDF file (see section \strong{Value} in \link{bench_graphs} for details). Also keep in mind that these graphics are optimized
for the US Letter paper size format in landscape view (as displayed in the PDF file), i.e., 11in wide (27.9cm, 1056px
with 96 DPI) and 8.5in tall (21.6cm, 816px with 96 DPI).
}
\description{
\if{html,text}{(\emph{version française: 
\url{https://StatCan.github.io/gensol-gseries/fr/reference/plot_graphTable.html}})}

Create a PDF file (US Letter paper size format in landscape view) containing benchmarking graphics for the set of series
contained in the specified benchmarking function (\code{\link[=benchmarking]{benchmarking()}} or \code{\link[=stock_benchmarking]{stock_benchmarking()}}) output \code{graphTable}
data frame. Four types of benchmarking graphics can be generated for each series:
\itemize{
\item \strong{Original Scale Plot} (argument \code{ori_plot_flag}) - overlay graph of:
\itemize{
\item Indicator series
\item Average indicator series
\item Bias corrected indicator series (when \eqn{\rho < 1})
\item Benchmarked series
\item Average benchmark
}
\item \strong{Adjustment Scale Plot} (argument \code{adj_plot_flag}) - overlay graph of:
\itemize{
\item Benchmarking adjustments
\item Average benchmarking adjustments
\item Bias line (when \eqn{\rho < 1})
}
\item \strong{Growth Rates Plot} (argument \code{GR_plot_flag}) - bar chart of the indicator and benchmarked series growth rates.
\item \strong{Growth Rates Table} (argument \code{GR_table_flag}) - table of the indicator and benchmarked series growth rates.
}

These graphics can be useful to assess the quality of the benchmarking results. Any of the four types of
benchmarking graphics can be enabled or disabled with the corresponding flag. The first three types of graphics
(the plots) are generated by default while the fourth (growth rates table) is not.
}
\details{
List of the \code{graphTable} data frame variables corresponding to each element of the four types of benchmarking
graphics:
\itemize{
\item Original Scale Plot (argument \code{ori_plot_flag})
\itemize{
\item \code{subAnnual} for the \emph{Indicator Series} line
\item \code{avgSubAnnual} for the \emph{Avg. Indicator Series} segments
\item \code{subAnnualCorrected} for the \emph{Bias Corr. Indicator Series} line (when \eqn{\rho < 1})
\item \code{benchmarked} for the \emph{Benchmarked Series} line
\item \code{avgBenchmark} for the \emph{Average Benchmark} segments
}
\item Adjustment Scale Plot (argument \code{adj_plot_flag})
\itemize{
\item \code{benchmarkedSubAnnualRatio} for the \emph{BI Ratios (Benchmarked Series / Indicator Series)} line \eqn{^{(*)}}{(*)}
\item \code{avgBenchmarkSubAnnualRatio} for the \emph{Average BI Ratios} segments \eqn{^{(*)}}{(*)}
\item \code{bias} for the \emph{Bias} line (when \eqn{\rho < 1})
}
\item Growth Rates Plot (argument \code{GR_plot_flag})
\itemize{
\item \code{growthRateSubAnnual} for the \emph{Growth R. in Indicator Series} bars \eqn{^{(*)}}{(*)}
\item \code{growthRateBenchmarked} for the \emph{Growth R. in Benchmarked Series} bars \eqn{^{(*)}}{(*)}
}
\item Growth Rates Table (argument \code{GR_table_flag})
\itemize{
\item \code{year} for the \emph{Year} column
\item \code{period} for the \emph{Period} column
\item \code{subAnnual} for the \emph{Indicator Series} column
\item \code{benchmarked} for the \emph{Benchmarked Series} column
\item \code{growthRateSubAnnual} for the \emph{Growth Rate in Indicator Series} column \eqn{^{(*)}}{(*)}
\item \code{growthRateBenchmarked} for the \emph{Growth Rate in Benchmarked Series} column \eqn{^{(*)}}{(*)}
}
}

\eqn{^{(*)}}{(*)} \emph{BI ratios} and \emph{growth rates} actually correspond to \emph{differences} when \eqn{\lambda = 0} (additive
benchmarking).

The function uses the extra columns of the \code{graphTable} data frame (columns not listed in the \strong{Value} section of
\code{\link[=benchmarking]{benchmarking()}} and \code{\link[=stock_benchmarking]{stock_benchmarking()}}), if any, to build BY-groups. See section \strong{Benchmarking Multiple Series}
of \code{\link[=benchmarking]{benchmarking()}} for more details.
\subsection{Performance}{

The two types of growth rates graphics, i.e., the bar chart (\code{GR_plot_flag}) and table (\code{GR_table_flag}), often requires
the generation of several pages in the PDF file, especially for long monthly series with several years of data. This creation of
extra pages slows down the execution of \code{\link[=plot_graphTable]{plot_graphTable()}}. This is why only the bar chart is generated by default
(\code{GR_plot_flag = TRUE} and \code{GR_table_flag = FALSE}). Deactivating both types of growth rates graphics (\code{GR_plot_flag = FALSE}
and \code{GR_table_flag = FALSE}) or reducing the size of the input \code{graphTable} data frame for very long series (e.g., keeping
only recent years) could therefore improve execution time. Also note that the impact of benchmarking on the growth rates can be
deduced from the adjustment scale plot (\code{adj_plot_flag}) by examining the extent of vertical movement (downward or upward) of the
benchmarking adjustments between adjacent periods: the greater the vertical movement, the greater the impact on corresponding growth
rate. Execution time of \code{\link[=plot_graphTable]{plot_graphTable()}} could therefore be reduced, if needed, by only generating the first two types of graphics
while focusing on the adjustment scale plot to assess period-to-period movement preservation, i.e., the impact of benchmarking on
the initial growth rates.
}

\subsection{ggplot2 themes}{

The plots are generated with the ggplot2 package which comes with a convenient set of \href{https://ggplot2.tidyverse.org/reference/ggtheme.html}{complete themes} for the general look and feel of the plots (with \code{theme_grey()}
as the default theme). Use function \code{theme_set()} to change the theme applied to the plots generated by \code{\link[=plot_graphTable]{plot_graphTable()}}
(see the \strong{Examples}).
}

\subsection{Bookmarks}{

Bookmarks are added to the PDF file with \code{xmpdf::set_bookmarks()} when argument \code{add_bookmarks = TRUE} (default), which
requires a command-line tool such as \href{https://www.ghostscript.com/}{Ghostscript}  or
\href{https://www.pdflabs.com/tools/pdftk-the-pdf-toolkit/}{PDFtk}. See section \strong{Installation} in
\code{vignette("xmpdf", package = "xmpdf")} for details.

\strong{Important}: bookmarks will be successfully added to the PDF file \strong{if and only if} \ifelse{latex}{\code{xmpdf::supports 
_set_bookmarks()}}{\code{xmpdf::supports_set_bookmarks()}} returns \code{TRUE} \strong{and} the execution of \code{xmpdf::set_bookmarks()} is
successful. If Ghostscript is installed on your machine but \code{xmpdf::supports_set_bookmarks()} still returns \code{FALSE}, try
specifying the path of the Ghostscript executable in environment variable \code{R_GSCMD} (e.g.,
\code{Sys.setenv(R_GSCMD = "C:/Program Files/.../bin/gswin64c.exe")} on Windows). On the other hand, if
\verb{xmpdf::supports_set_bookmarks()\}} returns \code{TRUE} but you are experiencing (irresolvable) issues with \code{xmpdf::set_bookmarks()}
(e.g., error related to the Ghostscript executable), bookmarks creation can be disabled by specifying \code{add_bookmarks = FALSE}.
}
}
\examples{
# Set the working directory (for the PDF files)
iniwd <- getwd() 
setwd(tempdir())

# Quarterly car and van sales (indicator series)
qtr_ind <- ts_to_tsDF(
  ts(matrix(c(# Car sales
              1851, 2436, 3115, 2205, 1987, 2635, 3435, 2361, 2183, 2822,
              3664, 2550, 2342, 3001, 3779, 2538, 2363, 3090, 3807, 2631,
              2601, 3063, 3961, 2774, 2476, 3083, 3864, 2773, 2489, 3082,
              # Van sales
              1900, 2200, 3000, 2000, 1900, 2500, 3800, 2500, 2100, 3100,
              3650, 2950, 3300, 4000, 3290, 2600, 2010, 3600, 3500, 2100,
              2050, 3500, 4290, 2800, 2770, 3080, 3100, 2800, 3100, 2860),
            ncol = 2),
     start = c(2011, 1),
     frequency = 4,
     names = c("car_sales", "van_sales")))

# Annual car and van sales (benchmarks)
ann_bmk <- ts_to_bmkDF(
  ts(matrix(c(# Car sales
              10324, 10200, 10582, 11097, 11582, 11092,
              # Van sales
              12000, 10400, 11550, 11400, 14500, 16000),
            ncol = 2),
     start = 2011,
     frequency = 1,
     names = c("car_sales", "van_sales")), 
  ind_frequency = 4)

# Proportional benchmarking without bias correction
out_bench <- benchmarking(qtr_ind, ann_bmk, 
                          rho = 0.729, lambda = 1, biasOption = 1,
                          allCols = TRUE,
                          quiet = TRUE)


# Default set of graphics (the first 3 types of plots)
\donttest{plot_graphTable(out_bench$graphTable, "bench_graphs.pdf")}

# Temporarily use ggplot2 `theme_bw()` for the plots
library(ggplot2)
ini_theme <- theme_get()
theme_set(theme_bw())
\donttest{plot_graphTable(out_bench$graphTable, "bench_graphs_bw.pdf")}
theme_set(ini_theme)

# Generate all 4 types of graphics (including the growth rates table)
\donttest{plot_graphTable(out_bench$graphTable, "bench_graphs_with_GRTable.pdf",
                GR_table_flag = TRUE)}

# Reduce execution time by disabling both types of growth rates graphics
\donttest{plot_graphTable(out_bench$graphTable, "bench_graphs_no_GR.pdf",
                GR_plot_flag = FALSE)}


# Reset the working directory to its initial location
setwd(iniwd)
}
\seealso{
\link{bench_graphs} \code{\link[=plot_benchAdj]{plot_benchAdj()}} \code{\link[=benchmarking]{benchmarking()}} \code{\link[=stock_benchmarking]{stock_benchmarking()}}
}
