% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/filter.R
\name{filter}
\alias{filter}
\alias{filter.default}
\alias{filter.Arma}
\alias{filter.Ma}
\alias{filter.Sos}
\alias{filter.Zpg}
\title{Filter a signal}
\usage{
filter(filt, ...)

\method{filter}{default}(filt, a, x, zi = NULL, ...)

\method{filter}{Arma}(filt, x, ...)

\method{filter}{Ma}(filt, x, ...)

\method{filter}{Sos}(filt, x, ...)

\method{filter}{Zpg}(filt, x, ...)
}
\arguments{
\item{filt}{For the default case, the moving-average coefficients of an ARMA
filter (normally called \code{b}), specified as a numeric or complex
vector. Generically, \code{filt} specifies an arbitrary filter operation.}

\item{...}{additional arguments (ignored).}

\item{a}{the autoregressive (recursive) coefficients of an ARMA filter,
specified as a numeric or complex vector. If \code{a[1]} is not equal to 1,
then filter normalizes the filter coefficients by \code{a[1]}. Therefore,
\code{a[1]} must be nonzero.}

\item{x}{the input signal to be filtered, specified as a numeric or complex
vector or matrix. If \code{x} is a matrix, each column is filtered.}

\item{zi}{If \code{zi} is provided, it is taken as the initial state of the
system and the final state is returned as zf. The state vector is a vector
or a matrix (depending on \code{x}) whose length or number of rows is equal
to the length of the longest coefficient vector \code{b} or \code{a} minus
one. If \code{zi} is not supplied (NULL), the initial state vector is set
to all zeros. Alternatively, \code{zi} may be the character string
\code{"zf"}, which specifies to return the final state vector even though
the initial state vector is set to all zeros. Default: NULL.}
}
\value{
The filtered signal, of the same dimensions as the input signal. In
  case the \code{zi} input argument was specified, a list with two elements
  is returned containing the variables \code{y}, which represents the output
  signal, and \code{zf}, which contains the final state vector or matrix.
}
\description{
Apply a 1-D digital filter compatible with 'Matlab' and 'Octave'.
}
\details{
The filter is a direct form II transposed implementation of the standard
linear time-invariant difference equation:
\if{latex}{
  \deqn{\sum_{k=0}^{N} a(k+1) y(n-k) + \sum_{k=0}^{M} b(k+1) x(n-k) = 0; 1
  \le n \le length(x)}
}
\if{html}{\preformatted{
  N                  M
 SUM a(k+1)y(n-k) + SUM b(k+1)x(n-k) = 0;   1 <= n <= length(x)
 k=0                k=0
}}
where \code{N = length(a) - 1} and \code{M = length(b) - 1}.

The initial and final conditions for filter delays can be used to filter data
in sections, especially if memory limitations are a consideration. See the
examples.
}
\examples{
bf <- butter(3, 0.1)                                 # 10 Hz low-pass filter
t <- seq(0, 1, len = 100)                            # 1 second sample
x <- sin(2* pi * t * 2.3) + 0.25 * rnorm(length(t))  # 2.3 Hz sinusoid+noise
z <- filter(bf, x)                                   # apply filter
plot(t, x, type = "l")
lines(t, z, col = "red")

## specify initial conditions
## from Python scipy.signal.lfilter() documentation
t <- seq(-1, 1, length.out =  201)
x <- (sin(2 * pi * 0.75 * t * (1 - t) + 2.1)
      + 0.1 * sin(2 * pi * 1.25 * t + 1)
      + 0.18 * cos(2 * pi * 3.85 * t))
h <- butter(3, 0.05)
lab <- max(length(h$b), length(h$a)) - 1
zi <- filtic(h$b, h$a, rep(1, lab), rep(1, lab))
z1 <- filter(h, x)
z2 <- filter(h, x, zi * x[1])
plot(t, x, type = "l")
lines(t, z1, col = "red")
lines(t, z2$y, col = "green")
legend("bottomright", legend = c("Original signal",
        "Filtered without initial conditions",
        "Filtered with initial conditions"),
       lty = 1, col = c("black", "red", "green"))

}
\seealso{
\code{\link{filter_zi}}, \code{\link{sosfilt}} (preferred because it
  avoids numerical problems).
}
\author{
Geert van Boxtel, \email{G.J.M.vanBoxtel@gmail.com}.
}
