% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/format_data.R
\name{fmt_number_si}
\alias{fmt_number_si}
\title{Format numbers with SI prefixes}
\usage{
fmt_number_si(
  data,
  columns = everything(),
  rows = everything(),
  unit = NULL,
  prefix_mode = c("engineering", "decimal"),
  decimals = 2,
  n_sigfig = NULL,
  drop_trailing_zeros = FALSE,
  drop_trailing_dec_mark = TRUE,
  use_seps = TRUE,
  scale_by = 1,
  pattern = "{x}",
  sep_mark = ",",
  dec_mark = ".",
  force_sign = FALSE,
  incl_space = TRUE,
  locale = NULL
)
}
\arguments{
\item{data}{\emph{The gt table data object}

\verb{obj:<gt_tbl>} // \strong{required}

This is the \strong{gt} table object that is commonly created through use of the
\code{\link[=gt]{gt()}} function.}

\item{columns}{\emph{Columns to target}

\code{\link[=rows-columns]{<column-targeting expression>}} // \emph{default:} \code{everything()}

Can either be a series of column names provided in \code{c()}, a vector of
column indices, or a select helper function (e.g. \code{\link[=starts_with]{starts_with()}},
\code{\link[=ends_with]{ends_with()}}, \code{\link[=contains]{contains()}}, \code{\link[=matches]{matches()}}, \code{\link[=num_range]{num_range()}} and \code{\link[=everything]{everything()}}).}

\item{rows}{\emph{Rows to target}

\code{\link[=rows-columns]{<row-targeting expression>}} // \emph{default:} \code{everything()}

In conjunction with \code{columns}, we can specify which of their rows should
undergo formatting. The default \code{\link[=everything]{everything()}} results in all rows in
\code{columns} being formatted. Alternatively, we can supply a vector of row
captions within \code{c()}, a vector of row indices, or a select helper
function (e.g. \code{\link[=starts_with]{starts_with()}}, \code{\link[=ends_with]{ends_with()}}, \code{\link[=contains]{contains()}}, \code{\link[=matches]{matches()}},
\code{\link[=num_range]{num_range()}}, and \code{\link[=everything]{everything()}}). We can also use expressions to filter
down to the rows we need (e.g., \verb{[colname_1] > 100 & [colname_2] < 50}).}

\item{unit}{\emph{Unit to append to formatted values}

\verb{scalar<character>} // \emph{default:} \code{NULL} (\code{optional})

A character string specifying the unit to append after the SI prefix
(e.g., \code{"g"} for grams, \code{"W"} for watts, \code{"Hz"} for hertz, \code{"m"} for meters).
If \code{NULL}, only the prefix will be shown. The unit can also be dynamically
specified per row using \code{\link[=from_column]{from_column()}}.}

\item{prefix_mode}{\emph{Type of SI prefixes to use}

\verb{singl-kw:[engineering|decimal]} // \emph{default:} \code{"engineering"}

The type of SI prefixes to use. Options are \code{"engineering"} (powers of
1000 only) or \code{"decimal"} (all SI prefixes including powers of 10 and 100).
See the \emph{SI Prefix Modes} section for details.}

\item{decimals}{\emph{Number of decimal places}

\code{scalar<numeric|integer>(val>=0)} // \emph{default:} \code{2}

The exact number of decimal places to display in the mantissa. If both
\code{decimals} and \code{n_sigfig} are provided, \code{n_sigfig} takes precedence.}

\item{n_sigfig}{\emph{Number of significant figures}

\code{scalar<numeric|integer>(val>=1)} // \emph{default:} \code{NULL} (\code{optional})

Format numbers to \emph{n} significant figures. This is often preferred in
scientific contexts to maintain consistent precision across different
magnitudes. When specified, the \code{decimals} argument is ignored.}

\item{drop_trailing_zeros}{\emph{Drop trailing zeros}

\verb{scalar<logical>} // \emph{default:} \code{FALSE}

Remove trailing zeros after the decimal point (e.g., "1.50" becomes "1.5").}

\item{drop_trailing_dec_mark}{\emph{Drop trailing decimal mark}

\verb{scalar<logical>} // \emph{default:} \code{TRUE}

Remove the decimal mark if all decimal places are zero (e.g., "1." becomes
"1").}

\item{use_seps}{\emph{Use digit group separators}

\verb{scalar<logical>} // \emph{default:} \code{TRUE}

Enable or disable the use of digit separators (e.g., thousands separators).}

\item{scale_by}{\emph{Scale values by a fixed multiplier}

\verb{scalar<numeric|integer>} // \emph{default:} \code{1}

All numeric values will be multiplied by the \code{scale_by} value before
undergoing formatting. Since the \code{default} value is \code{1}, no values will be
changed unless a different multiplier value is supplied. This is useful
for unit conversions, such as using \code{\link[=unit_conversion]{unit_conversion()}} to convert
horsepower to watts before formatting with SI prefixes.}

\item{pattern}{\emph{Decoration pattern}

\verb{scalar<character>} // \emph{default:} \code{"{x}"}

A formatting pattern for decorating values. Use \code{{x}} to represent the
formatted value (including prefix and unit).}

\item{sep_mark}{\emph{Thousands separator}

\verb{scalar<character>} // \emph{default:} \code{","}

The character to use as the thousands separator. Overridden if \code{locale} is
provided.}

\item{dec_mark}{\emph{Decimal mark}

\verb{scalar<character>} // \emph{default:} \code{"."}

The character to use as the decimal point. Overridden if \code{locale} is
provided.}

\item{force_sign}{\emph{Force positive sign}

\verb{scalar<logical>} // \emph{default:} \code{FALSE}

Force the display of a plus sign for positive values.}

\item{incl_space}{\emph{Include a space between the value and the unit symbol}

\verb{scalar<logical>} // \emph{default:} \code{TRUE}

An option for whether to include a space between the numerical value and
the SI prefix + unit (e.g., \code{TRUE} for "1.5 kW", \code{FALSE} for "1.5kW"). Per
SI convention, there should be a space between the value and the unit
symbol.}

\item{locale}{\emph{Locale identifier}

\verb{scalar<character>} // \emph{default:} \code{NULL} (\code{optional})

An optional locale identifier for locale-specific number formatting.
When provided, overrides \code{sep_mark} and \code{dec_mark} with locale-appropriate
values.}
}
\value{
An object of class \code{gt_tbl}.
}
\description{
Format numeric values with SI (International System of Units) prefixes,
automatically selecting the appropriate prefix to keep the mantissa in a
readable range. SI prefixes range from quetta (Q, 10^30) to quecto
(q, 10^-30) and are commonly used in scientific and engineering contexts to
represent very large or very small quantities with units (e.g., "5.2 kW",
"3.8 ng", "1.2 GHz", etc.).

This function provides fine control over SI prefix formatting with the
following options:
\itemize{
\item unit specification: define a fixed unit or use per-row units from a column
\item prefix selection: choose between all SI prefixes or only engineering
prefixes (powers of 1000)
\item precision control: specify decimal places or significant figures
\item spacing: customize the separator between number, prefix, and unit
\item locale-based formatting: use locale-specific decimal and thousands separators
}
}
\section{SI Prefix Modes}{


The \code{prefix_mode} argument controls which SI prefixes are used:
\itemize{
\item \code{"engineering"}: Uses only prefixes for powers of 1000. This includes:
\itemize{
\item greater than 1: k (kilo), M (mega), G (giga), T (tera), P (peta),
E (exa), Z (zetta), Y (yotta), R (ronna), Q (quetta)
\item less than 1: m (milli), u (micro), n (nano), p (pico), f (femto),
a (atto), z (zepto), y (yocto), r (ronto), q (quecto)
\item this is the most common convention in scientific and engineering
contexts.
}
\item \code{"decimal"}: Uses all SI prefixes including those for powers of 10 and 100:
\itemize{
\item Additional prefixes for greater-than-1 values: da (deca), h (hecto)
\item Additional prefixes for less-than-1 values: d (deci), c (centi)
\item This mode is less commonly used but follows the complete SI standard.
}
}
}

\section{Compatibility of formatting function with data values}{


\code{fmt_number_si()} is compatible with body cells that are of the \code{"numeric"}
or \code{"integer"} types. Any other types of body cells are ignored during
formatting. This is to say that cells of incompatible data types may be
targeted, but there will be no attempt to format them.
}

\section{Compatibility of arguments with the \code{from_column()} helper function}{


\code{\link[=from_column]{from_column()}} can be used with certain arguments of \code{fmt_number_si()} to
obtain varying parameter values from a specified column within the table.
This means that each row could be formatted a little bit differently. These
arguments provide support for \code{\link[=from_column]{from_column()}}:
\itemize{
\item \code{unit}: The unit can be specified with a column name in quotes.
\item \code{decimals}: Each row's number formatting could use a different number of
decimal places.
\item \code{n_sigfig}: Each row could have a different number of significant figures.
\item \code{drop_trailing_zeros}: The option to drop trailing zeros can be controlled
per row.
\item \code{drop_trailing_dec_mark}: The option to drop trailing decimal marks can be
controlled per row.
\item \code{use_seps}: The use of digit separators can be enabled or disabled on a
per-row basis.
\item \code{scale_by}: The scale multiplier can be different for each row.
\item \code{pattern}: The formatting pattern can be specified per row.
\item \code{sep_mark}: The thousands separator mark can be set per row.
\item \code{dec_mark}: The decimal mark can be set per row.
\item \code{force_sign}: Whether to force a plus sign can be controlled per row.
\item \code{incl_space}: Whether to include a space between number and unit can vary
per row.
\item \code{locale}: The locale can be specified per row.
}

Please note that for all of the aforementioned arguments, a \code{\link[=from_column]{from_column()}}
call needs to reference a column that has data of the correct type (this is
different for each argument). Additional columns for parameter values can be
generated with \code{\link[=cols_add]{cols_add()}} (if not already present). Columns that contain
parameter data can also be hidden from final display with \code{\link[=cols_hide]{cols_hide()}}.
Finally, there is no limitation to how many arguments the \code{\link[=from_column]{from_column()}}
helper is applied so long as the arguments belong to this closed set.
}

\section{Adapting output to a specific \code{locale}}{


This formatting function can adapt outputs according to a provided \code{locale}
value. Examples include \code{"en"} for English (United States) and \code{"fr"} for
French (France). The use of a valid locale ID here means separator and
decimal marks will be correct for the given locale. Should any values be
provided in \code{sep_mark} or \code{dec_mark}, they will be overridden by the locale's
preferred values.

Note that a \code{locale} value provided here will override any global locale
setting performed in \code{\link[=gt]{gt()}}'s own \code{locale} argument (it is settable there as
a value received by all other functions that have a \code{locale} argument). As a
useful reference on which locales are supported, we can call \code{\link[=info_locales]{info_locales()}}
to view an info table.
}

\section{Examples}{


Create a table showing the masses of obelisks located in Rome. The masses are
initially in metric tons, which we'll convert to grams using
\code{\link[=unit_conversion]{unit_conversion()}} in the \code{scale_by} argument. The resulting values are
then formatted with SI prefixes, which are all here as \code{M} (\emph{mega}).

\if{html}{\out{<div class="sourceCode r">}}\preformatted{dplyr::tibble(
  obelisk = c(
    "Lateran Obelisk",
    "Vatican Obelisk",
    "Flaminio Obelisk",
    "Pantheon Obelisk"
  ),
  mass_ton = c(455, 331, 235, 30)
) |>
  gt() |>
  fmt_number_si(
    columns = mass_ton,
    unit = "g",
    decimals = 0,
    scale_by = unit_conversion(
      from = "mass.metric-ton",
      to = "mass.gram"
    )
  ) |>
  cols_label(
    obelisk = "Obelisk",
    mass_ton = "Mass"
  )
}\if{html}{\out{</div>}}

\if{html}{\out{
<img src="https://raw.githubusercontent.com/rstudio/gt/master/images/man_fmt_number_si_1.png" alt="This image of a table was generated from the first code example in the `fmt_number_si()` help file." style="width:100\%;">
}}

Create a table showing measurements of different substances with varying
units. The \code{unit} column contains different units per row (grams and liters),
which are used with \code{\link[=from_column]{from_column()}} to apply appropriate SI prefixes.

\if{html}{\out{<div class="sourceCode r">}}\preformatted{dplyr::tibble(
  substance = c("Glucose", "Vitamin C", "Caffeine", "Water"),
  amount = c(0.0051, 0.000075, 0.0002, 0.250),
  unit = c("g", "g", "g", "L")
) |>
  gt() |>
  fmt_number_si(
    columns = amount,
    unit = from_column("unit"),
    n_sigfig = 2
  ) |>
  cols_hide(columns = unit)
}\if{html}{\out{</div>}}

\if{html}{\out{
<img src="https://raw.githubusercontent.com/rstudio/gt/master/images/man_fmt_number_si_2.png" alt="This image of a table was generated from the second code example in the `fmt_number_si()` help file." style="width:100\%;">
}}

You can combine \code{fmt_number_si()} with \code{\link[=fmt_units]{fmt_units()}} and \code{\link[=cols_merge]{cols_merge()}} to
format measurements with SI prefixes on units that need special typesetting.
In this example, \code{fmt_number_si()} handles both the SI prefix and the unit
with proper spacing, while \code{\link[=fmt_units]{fmt_units()}} creates a separate column with
additional unit information (like "per hour") that gets merged in.

\if{html}{\out{<div class="sourceCode r">}}\preformatted{dplyr::tibble(
  measurement = c("Power", "Resistance", "Energy", "Fall Velocity"),
  value = c(1500, 2400000, 3600000,0.033),
  unit = c("W", ":ohm:", "J", "m /s")
) |>
  gt() |>
  fmt_number_si(columns = value, decimals = 1) |>
  fmt_units(columns = unit) |>
  cols_merge(columns = c(value, unit), pattern = "\{1\}\{2\}")
}\if{html}{\out{</div>}}

\if{html}{\out{
<img src="https://raw.githubusercontent.com/rstudio/gt/master/images/man_fmt_number_si_3.png" alt="This image of a table was generated from the third code example in the `fmt_number_si()` help file." style="width:100\%;">
}}
}

\section{Function ID}{

3-5
}

\section{Function Introduced}{

\code{v1.2.0} (December 16, 2025)
}

\seealso{
The vector-formatting version of this function:
\code{\link[=vec_fmt_number_si]{vec_fmt_number_si()}}.

Other data formatting functions: 
\code{\link{data_color}()},
\code{\link{fmt}()},
\code{\link{fmt_auto}()},
\code{\link{fmt_bins}()},
\code{\link{fmt_bytes}()},
\code{\link{fmt_chem}()},
\code{\link{fmt_country}()},
\code{\link{fmt_currency}()},
\code{\link{fmt_date}()},
\code{\link{fmt_datetime}()},
\code{\link{fmt_duration}()},
\code{\link{fmt_email}()},
\code{\link{fmt_engineering}()},
\code{\link{fmt_flag}()},
\code{\link{fmt_fraction}()},
\code{\link{fmt_icon}()},
\code{\link{fmt_image}()},
\code{\link{fmt_index}()},
\code{\link{fmt_integer}()},
\code{\link{fmt_markdown}()},
\code{\link{fmt_number}()},
\code{\link{fmt_partsper}()},
\code{\link{fmt_passthrough}()},
\code{\link{fmt_percent}()},
\code{\link{fmt_roman}()},
\code{\link{fmt_scientific}()},
\code{\link{fmt_spelled_num}()},
\code{\link{fmt_tf}()},
\code{\link{fmt_time}()},
\code{\link{fmt_units}()},
\code{\link{fmt_url}()},
\code{\link{sub_large_vals}()},
\code{\link{sub_missing}()},
\code{\link{sub_small_vals}()},
\code{\link{sub_values}()},
\code{\link{sub_zero}()}
}
\concept{data formatting functions}
