% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/hp1.R
\name{hp1}
\alias{hp1}
\title{One-Sided HP Filter}
\usage{
hp1(y, lambda = 1600, x_user = NA, P_user = NA, discard = 0)
}
\arguments{
\item{y}{a dataframe of size Txn, where "T" is the number of observations
for each variable (number of rows) and "n" - the number of variables in
the dataframe (number of columns).}

\item{lambda}{the smoothing parameter; a numeric scalar which takes the
default value of 1600, if unspecified by the user.}

\item{x_user}{user defined initial values of the state estimate for each
variable in y. Takes the form of a 2xn matrix. Since the underlying state
vector is 2x1, two values are needed for each variable in y. By default:
if no values are provided, backwards extrapolations based on the first two
observations are used.}

\item{P_user}{a structural array with n elements, each of which being a 2x2
matrix of initial MSE estimates for each variable in y. By default: if no
values are provided, a matrix with relatively large variances is used.}

\item{discard}{the number of discard periods, expressed as a numeric scalar.
The user specified amount of values will be discarded from the start of the
sample, resulting in output matrices of size (T-discard)xn. By default: if
no values are provided, is set to 0.}
}
\value{
a (T-discard)xn dataframe, containing the trend data
}
\description{
hp1 applies a one-sided Hodrick-Prescott filter derived using the Kalman
filter to separate a time series into trend and cyclical components. The
smoothing parameter should reflect the cyclical duration or frequency of
the data.
}
\details{
The length of the time series should be greater than four and the
  value of the smoothing parameter greater than zero for the code to function.
  Of course, having a sufficiently long time series is paramount to achieving
  meaningful results.
}
\examples{
# Generate the data and plot it
set.seed(10)
y <- as.data.frame(rev(diffinv(rnorm(100)))[1:100])+30
colnames(y) <- "gdp"
plot(y$gdp, type="l")

# Apply the HP filter to the data
ytrend = hp1(y)
ycycle = y - ytrend

# Plot the three resulting series
plot(y$gdp, type="l", col="black", lty=1, ylim=c(-10,30))
lines(ytrend$gdp, col="#066462")
polygon(c(1, seq(ycycle$gdp), length(ycycle$gdp)), c(0, ycycle$gdp, 0), col = "#E0F2F1")
legend("bottom", horiz=TRUE, cex=0.75, c("y", "ytrend", "ycycle"), lty = 1,
       col = c("black", "#066462", "#75bfbd"))

}
\references{
Balcilar, M. (2019). Miscellaneous Time Series Filters 'mFilter'. CRAN R Package Library.

Drehmann, M., and Yetman, J. (2018). Why You Should Use the Hodrick-Prescott Filter - at Least to Generate Credit Gaps. BIS Working Paper No. 744.

Eurostat (2023), Real Gross Domestic Product for European Union (28 countries) [CLVMNACSCAB1GQEU28], National Accounts - GDP.

Hamilton, J. D. (2017). Why You Should Never Use the Hodrick-Prescott Filter. Working Paper Series. National Bureau of Economic Research, May 2017.

Hodrick, R. J., and Prescott, E. C. (1997). Postwar U.S. Business Cycles: An Empirical Investigation. Journal of Money, Credit, and Banking 29: 1-16.

Hyeongwoo, K. (2004). "Hodrick-Prescott Filter". Notes, Auburn University.

Mcelroy, T. (2008). Exact formulas for the Hodrick-Prescott Filter. Econometrics Journal. 11. 209-217.

Meyer-Gohde, A. (2010). Matlab code for one-sided HP-filters. QM&RBC Codes 181, Quantitative Macroeconomics & Real Business Cycles.

Ravn, M., and Uhlig, H. (2002). On adjusting the Hodrick-Prescott filter for the frequency of observations, The Review of Economics and Statistics 2002; 84 (2): 371-376.

Shea, J. (2021). neverhpfilter: An Alternative to the Hodrick-Prescott Filter. CRAN R Package Library.
}
\seealso{
[hp2()]
}
\author{
Alexandru Monahov, <https://www.alexandrumonahov.eu.org/>
}
\keyword{basel}
\keyword{buffer}
\keyword{ccyb}
\keyword{filter}
\keyword{gap}
\keyword{hp}
\keyword{hp1}
\keyword{kalman}
\keyword{one-sided}
