# Wrapper function that takes arguments for the various indirect 
# effect tests and calls the desired/specified test. 
# Main point of this function is to allow input to the function
# of the parameters in a vector and the variance/covariance terms
# in a matrix. 

#' Testing the indirect effect.
#'
#' This function takes a vector estimates and a matrix of covariances 
#' or a vector of U values to be used in various indirect effect tests.
#' Estimate vectors with three parameters will default to single mediator
#' analysis (or one path of unordered mediation), and five parameters will
#' default to the ordered scenario. Values for the interaction term must
#' be provided for this wrapper. The user can then specify the distribution(s)
#' to be used as well as the test to be performed. 
#'
#' @param test Denotes the test to be performed. ("maxP", "ps-test", "asq-test", "sobel", "sobel unordered")
#' @param u A vector with the U values to be used in the test. Given priority over estimates, but all must be supplied. Order defined in note.
#' @param V A vector containing the estimates to be used in the test. Must follow same order as u.
#' @param cov A matrix of the covariance matrix of the estimates in V. Must be square with dimensions compatible with V. Must follow same order as u.
#' @param df A vector of the degrees of freedom for the effect estimates. Length of 2 for single mediator, 3 for ordered.  
#' @param V_0 A vector containing the null values of the estimates to be used in the test. Defaults to zero.
#' @param V1Dist String value specifying the distribution of the estimate of the independent variable on the mediator.
#' @param V2Dist String value specifying the distribution of the second estimate.
#' @param V3Dist String value specifying the distribution of the third estimate (only needed for ordered scenario).
#' @param numSquares The number of squares to be used in the asq-test. Always superseded by upLim.
#' @param upLim The allowed extension, between 0 and 1, for the asq and ps-tests. 
#' @param alpha A vector for the asq-test of significance levels to test. A value in the ps-test to control type I error.
#' @param interMult Integer indicating the level of the independent variable to use for the interaction terms.
#' @returns A p-value or p-value cutoff for the specified test for the indirect effect. 
#' @export
#' 
#' @note 
#' Order of parameters
#' Values must be in the correct order within u, V, and the cov matrix. 
#' 
#' \strong{Single mediator (or single path in unordered scenarios)}: \cr
#' Independent variable to mediator, mediator to response, 
#' interaction of independent and mediator.
#' 
#' \strong{Ordered mediators}: \cr
#' Independent variable to first mediator, 
#' first mediator to second mediator, interaction of independent and first mediator on second mediator, 
#' second mediator to response, interaction of independent and second mediator on response.
#' 
#' \strong{Combined Unordered Mediator}: \cr
#' Independent variable to first mediator, 
#' first mediator to response, interaction of independent and first mediator on response, 
#' Independent variable to second mediator, 
#' second mediator to response, interaction of independent and second mediator on response.
#' 
#' @examples
#' ieTest( test = "ps-test", u = c(.015, .02, .998), alpha = 0.05, upLim = 0.5)
#' 
#' @useDynLib ieTest, .registration=TRUE
#' @import MASS Rcpp RcppArmadillo RcppDist twosamples methods

ieTest = function(test, u = NULL, V = NULL, cov = NULL,
                df = NULL, V_0 = 0,
                V1Dist = NULL, V2Dist = NULL, V3Dist = NULL,
                numSquares = 10, upLim = 0.5, alpha = 0.05, 
                interMult = 1){

  # Determine if U values or estimates passed, and then call test accordingly;
  if(!is.null(u)){
    paramNum = length(u)
    
    
    # Single mediator or single path of unordered options
    if(paramNum == 2){
      
      # sobel test
      if(test %in% c("sobel", "Sobel", "delta", "Delta")){
        warning("The Sobel test requires estimate values\n")
        return(-1)
      }else if(test %in% c("maxP", "MaxP", "maxp", "Maxp")){
        return(maxp_one(u1 = u[1], u2 = u[2],
                          V1Dist = NULL, V1 = NULL, V1_VAR = NULL, V1_DF = NULL,
                          V2Dist = NULL, V2 = NULL, V2_VAR = NULL, V2_DF = NULL,
                          V2b = 0, V2b_VAR = 0, V2bmult = interMult,
                          V1_V2_cov = 0, V1_V2b_cov = 0, V2_V2b_cov = 0, 
                          V1_0 = 0, V2_0 = 0, V2b_0 = 0));
        
      }else if(test %in% c("ps", "PS", "ps-test","PS-test")){
        
        return(ps_one(u1 = u[1], u2 = u[2],
                      V1Dist = NULL, V1 = NULL, V1_VAR = NULL, V1_DF = NULL,
                      V2Dist = NULL, V2 = NULL, V2_VAR = NULL, V2_DF = NULL,
                      V2b = 0, V2b_VAR = 0, V2bmult = interMult,
                      V1_V2_cov = 0, V1_V2b_cov = 0, V2_V2b_cov = 0, 
                      V1_0 = 0, V2_0 = 0, V2b_0 = 0, upLim = upLim, alpha = alpha[1]));
        
      }else if(test %in% c("asq", "ASQ", "asq-test", "ASQ-test")){
       
        return(asq_one(u1 = u[1], u2 = u[2],
                       V1Dist = NULL, V1 = NULL, V1_VAR = NULL, V1_DF = NULL,
                       V2Dist = NULL, V2 = NULL, V2_VAR = NULL, V2_DF = NULL,
                       V2b = 0, V2b_VAR = 0, V2bmult = interMult,
                       V1_V2_cov = 0, V1_V2b_cov = 0, V2_V2b_cov = 0, 
                       V1_0 = 0, V2_0 = 0, V2b_0 = 0, 
                       numSquares = numSquares, upLim = upLim, alpha = alpha));
        
      }
# End 2 U value (single mediator) scenarios      
    }else if(paramNum == 3){         
      
      # sobel test
      if(test %in% c("sobel", "Sobel", "delta", "Delta")){
        warning("The Sobel test requires estimate values\n")
        return(-1)
      }else if(test %in% c("maxP", "MaxP", "maxp", "Maxp")){
        return(maxp_ord(u1 = u[1], u2 = u[2], u3 = u[3],
                        V1Dist = NULL, V1 = NULL, V1_VAR = NULL, V1_DF = NULL,
                        V2Dist = NULL, V2 = NULL, V2_VAR = NULL, V2_DF = NULL,
                        V2b = 0, V2b_VAR = 0, V2bmult = interMult,
                        V3Dist = NULL, V3 = NULL, V3_VAR = NULL, V3_DF = NULL,
                        V3b = 0, V3b_VAR = 0,
                        V1_V2_cov = 0, V1_V2b_cov = 0, V1_V3_cov = 0, V1_V3b_cov = 0, 
                        V2_V2b_cov = 0, V2_V3_cov = 0, V2_V3b_cov = 0, 
                        V2b_V3_cov = 0, V2b_V3b_cov = 0, V3_V3b_cov = 0, 
                        V1_0 = 0, V2_0 = 0, V2b_0 = 0, V3_0 = 0, V3b_0 = 0));
        
      }else if(test %in% c("ps", "PS", "ps-test","PS-test")){
        
        return(ps_ord(u1 = u[1], u2 = u[2], u3 = u[3],
                      V1Dist = NULL, V1 = NULL, V1_VAR = NULL, V1_DF = NULL,
                      V2Dist = NULL, V2 = NULL, V2_VAR = NULL, V2_DF = NULL,
                      V2b = 0, V2b_VAR = 0, V2bmult = interMult,
                      V3Dist = NULL, V3 = NULL, V3_VAR = NULL, V3_DF = NULL,
                      V3b = 0, V3b_VAR = 0,
                      V1_V2_cov = 0, V1_V2b_cov = 0, V1_V3_cov = 0, V1_V3b_cov = 0, 
                      V2_V2b_cov = 0, V2_V3_cov = 0, V2_V3b_cov = 0, 
                      V2b_V3_cov = 0, V2b_V3b_cov = 0, V3_V3b_cov = 0, 
                      V1_0 = 0, V2_0 = 0, V2b_0 = 0, V3_0 = 0, V3b_0 = 0, upLim = upLim, alpha = alpha[1]));
        
      }else if(test %in% c("asq", "ASQ", "asq-test", "ASQ-test")){
        return(asq_ord(u1 = u[1], u2 = u[2], u3 = u[3],
                       V1Dist = NULL, V1 = NULL, V1_VAR = NULL, V1_DF = NULL,
                       V2Dist = NULL, V2 = NULL, V2_VAR = NULL, V2_DF = NULL,
                       V2b = 0, V2b_VAR = 0, V2bmult = interMult,
                       V3Dist = NULL, V3 = NULL, V3_VAR = NULL, V3_DF = NULL,
                       V3b = 0, V3b_VAR = 0,
                       V1_V2_cov = 0, V1_V2b_cov = 0, V1_V3_cov = 0, V1_V3b_cov = 0, 
                       V2_V2b_cov = 0, V2_V3_cov = 0, V2_V3b_cov = 0, 
                       V2b_V3_cov = 0, V2b_V3b_cov = 0, V3_V3b_cov = 0, 
                       V1_0 = 0, V2_0 = 0, V2b_0 = 0, V3_0 = 0, V3b_0 = 0, 
                       numSquares = numSquares, upLim = upLim, alpha = alpha));
        
      }
# End 3 U value (single mediator) scenarios    
    }else{
      warning("U vector incorrectly specified\n")
      return(-1)
    }
# End provided U values
  }else if(!is.null(V)){    ### Else, must use provided estimates and covariances - errors should be caught by c functions
    paramNum = length(V)
    
    ### Fix null hypothesis vector if not provided
    if(length(V_0) == 1 & V_0 == 0){
      V_0 = rep(V_0, paramNum)
    }else if(length(V_0) < paramNum){
      warning("Null vector extended from given values. May be incorrect.\n")
      V_0 = rep(V_0, length = paramNum)
    }
    # Single mediator or single path of unordered options
    if(paramNum == 3){
      
      # sobel test
      if(test %in% c("sobel", "Sobel", "delta", "Delta")){
        return(sobelTest_one(mu1 = V[1], sig1 = cov[1, 1],
               mu2 = V[2], sig2 = cov[2, 2], sig12 = cov[1, 2],
               indL = interMult,
               mu3 = V[3], sig3 = cov[3, 3],
               sig13 = cov[1, 3], sig23 = cov[2, 3],
               mu1_0 = V_0[1], mu2_0 = V_0[2], mu3_0 = V_0[3]))
      }else if(test %in% c("maxP", "MaxP", "maxp", "Maxp")){
        return(maxp_one(u1 = NULL, u2 = NULL,
                        V1Dist = V1Dist, V1 = V[1], V1_VAR = cov[1, 1], V1_DF = df[1],
                        V2Dist = V2Dist, V2 = V[2], V2_VAR = cov[2, 2], V2_DF = df[2],
                        V2b = V[3], V2b_VAR = cov[3, 3], V2bmult = interMult,
                        V1_V2_cov = cov[1, 2], V1_V2b_cov = cov[1, 3], V2_V2b_cov = cov[2, 3], 
                        V1_0 = V_0[1], V2_0 = V_0[2], V2b_0 = V_0[3]));
        
      }else if(test %in% c("ps", "PS", "ps-test","PS-test")){
        
        return(ps_one(u1 = NULL, u2 = NULL,
                      V1Dist = V1Dist, V1 = V[1], V1_VAR = cov[1, 1], V1_DF = df[1],
                      V2Dist = V2Dist, V2 = V[2], V2_VAR = cov[2, 2], V2_DF = df[2],
                      V2b = V[3], V2b_VAR = cov[3, 3], V2bmult = interMult,
                      V1_V2_cov = cov[1, 2], V1_V2b_cov = cov[1, 3], V2_V2b_cov = cov[2, 3], 
                      V1_0 = V_0[1], V2_0 = V_0[2], V2b_0 = V_0[3], 
                      upLim = upLim, alpha = alpha[1]));
        
      }else if(test %in% c("asq", "ASQ", "asq-test", "ASQ-test")){
#        browser()
        return(asq_one(u1 = NULL, u2 = NULL,
                       V1Dist = V1Dist, V1 = V[1], V1_VAR = cov[1, 1], V1_DF = df[1],
                       V2Dist = V2Dist, V2 = V[2], V2_VAR = cov[2, 2], V2_DF = df[2],
                       V2b = V[3], V2b_VAR = cov[3, 3], V2bmult = interMult,
                       V1_V2_cov = cov[1, 2], V1_V2b_cov = cov[1, 3], V2_V2b_cov = cov[2, 3], 
                       V1_0 = V_0[1], V2_0 = V_0[2], V2b_0 = V_0[3], 
                       numSquares = numSquares, upLim = upLim, alpha = alpha));
        
      }
      # End 2 U value (single mediator) scenarios      
    } else if(paramNum == 5){         
    
      # sobel test
      if(test %in% c("sobel", "Sobel", "delta", "Delta")){
        return(sobelTest_ord(mu1 = V[1], sig1 = cov[1, 1],
                             mu2 = V[2], sig2 = cov[2, 2], 
                             mu2b = V[3], sig2b = cov[3, 3], 
                             mu3 = V[4], sig3 = cov[4, 4],
                             mu3b = V[5], sig3b = cov[5, 5],
                             sig12 = cov[1, 2], sig12b = cov[1, 3], sig13 = cov[1, 4], sig13b = cov[1, 5],
                             sig22b = cov[2, 3], sig23 = cov[2, 4], sig23b = cov[2, 5],
                             sig2b3 = cov[3, 4], sig2b3b = cov[3, 5], sig33b = cov[4, 5],
                             indL = interMult,
                             mu1_0 = V_0[1], mu2_0 = V_0[2], mu2b_0 = V_0[3], 
                             mu3_0 = V_0[4], mu3b_0 = V_0[5]))
      }else if(test %in% c("maxP", "MaxP", "maxp", "Maxp")){
        return(maxp_ord(u1 = NULL, u2 = NULL, u3 = NULL,
                        V1Dist = V1Dist, V1 = V[1], V1_VAR = cov[1, 1], V1_DF = df[1],
                        V2Dist = V2Dist, V2 = V[2], V2_VAR = cov[2, 2], V2_DF = df[2],
                        V2b = V[3], V2b_VAR = cov[3, 3], V2bmult = interMult,
                        V3Dist = V3Dist, V3 = V[4], V3_VAR = cov[4, 4], V3_DF = df[3],
                        V3b = V[5], V3b_VAR = cov[5, 5],
                        V1_V2_cov = cov[1, 2], V1_V2b_cov = cov[1, 3], 
                          V1_V3_cov = cov[1, 4], V1_V3b_cov = cov[1, 5], 
                        V2_V2b_cov = cov[2, 3], V2_V3_cov = cov[2, 4], 
                          V2_V3b_cov = cov[2, 5], 
                        V2b_V3_cov = cov[3, 4], V2b_V3b_cov = cov[3, 5], 
                          V3_V3b_cov = cov[4, 5], 
                        V1_0 = V_0[1], V2_0 = V_0[2], V2b_0 = V_0[3], V3_0 = V_0[4], V3b_0 = V_0[5]));
        
      }else if(test %in% c("ps", "PS", "ps-test","PS-test")){
        
        return(ps_ord(u1 = NULL, u2 = NULL, u3 = NULL,
                      V1Dist = V1Dist, V1 = V[1], V1_VAR = cov[1, 1], V1_DF = df[1],
                      V2Dist = V2Dist, V2 = V[2], V2_VAR = cov[2, 2], V2_DF = df[2],
                      V2b = V[3], V2b_VAR = cov[3, 3], V2bmult = interMult,
                      V3Dist = V3Dist, V3 = V[4], V3_VAR = cov[4, 4], V3_DF = df[3],
                      V3b = V[5], V3b_VAR = cov[5, 5],
                      V1_V2_cov = cov[1, 2], V1_V2b_cov = cov[1, 3], 
                        V1_V3_cov = cov[1, 4], V1_V3b_cov = cov[1, 5], 
                      V2_V2b_cov = cov[2, 3], V2_V3_cov = cov[2, 4], 
                        V2_V3b_cov = cov[2, 5], 
                      V2b_V3_cov = cov[3, 4], V2b_V3b_cov = cov[3, 5], 
                        V3_V3b_cov = cov[4, 5], 
                      V1_0 = V_0[1], V2_0 = V_0[2], V2b_0 = V_0[3], V3_0 = V_0[4], V3b_0 = V_0[5], 
                      upLim = upLim, alpha = alpha[1]));
        
      }else if(test %in% c("asq", "ASQ", "asq-test", "ASQ-test")){
        return(asq_ord(u1 = NULL, u2 = NULL, u3 = NULL,
                       V1Dist = V1Dist, V1 = V[1], V1_VAR = cov[1, 1], V1_DF = df[1],
                       V2Dist = V2Dist, V2 = V[2], V2_VAR = cov[2, 2], V2_DF = df[2],
                       V2b = V[3], V2b_VAR = cov[3, 3], V2bmult = interMult,
                       V3Dist = V3Dist, V3 = V[4], V3_VAR = cov[4, 4], V3_DF = df[3],
                       V3b = V[5], V3b_VAR = cov[5, 5],
                       V1_V2_cov = cov[1, 2], V1_V2b_cov = cov[1, 3], 
                        V1_V3_cov = cov[1, 4], V1_V3b_cov = cov[1, 5], 
                       V2_V2b_cov = cov[2, 3], V2_V3_cov = cov[2, 4], 
                        V2_V3b_cov = cov[2, 5], 
                       V2b_V3_cov = cov[3, 4], V2b_V3b_cov = cov[3, 5], 
                        V3_V3b_cov = cov[4, 5], 
                       V1_0 = V_0[1], V2_0 = V_0[2], V2b_0 = V_0[3], V3_0 = V_0[4], V3b_0 = V_0[5], 
                       numSquares = numSquares, upLim = upLim, alpha = alpha));
        
      }
      # End ordered scenario  
    }else if(paramNum == 6){
      #cat("Performing Sobel test for combined mediation effect with two unordered mediators.\n")
        return(sobelTest_unord(mu1 = V[1], sig1 = cov[1, 1],
                             mu2 = V[2], sig2 = cov[2, 2], 
                             mu2b = V[3], sig2b = cov[3, 3], 
                             mu3 = V[4], sig3 = cov[4, 4],
                             mu4 = V[5], sig4 = cov[5, 5],
                             mu4b = V[6], sig4b = cov[6, 6],
                             sig12 = cov[1, 2], sig12b = cov[1, 3], sig13 = cov[1, 4], 
                              sig14 = cov[1, 5], sig14b = cov[1, 6],
                             sig22b = cov[2, 3], sig23 = cov[2, 4], sig24 = cov[2, 5],
                              sig24b = cov[2, 6],
                             sig2b3 = cov[3, 4], sig2b4 = cov[3, 5], sig2b4b = cov[3, 6], 
                             sig34 = cov[4, 5], sig34b = cov[4, 6],
                             sig44b = cov[5, 6],
                             indL = interMult,
                             mu1_0 = V_0[1], mu2_0 = V_0[2], mu2b_0 = V_0[3], 
                             mu3_0 = V_0[4], mu4_0 = V_0[5], mu4b_0 = V_0[6]))
    }else{
      warning("V vector incorrectly specified") 
      return(-1)
    }
# End provided V values
  }else{
    warning("Error in provided values\n")
    return(-1)
  }
}
