#include <Rcpp.h>
using namespace Rcpp;

// Function that accepts parameter values for 2 main effects and one interaction
// and performs the sobel test. Main effects and covariance terms required.

//' Sobel test for the indirect effect - Single mediator path
//'
//' This function takes the parameter estimates and covariances
//' and performs the Sobel test for one mediator, or a single
//' mediator path for multiple unordered mediators.
//'
//' @param mu1 Value of the estimate of the independent variable on the mediator.
//' @param sig1 Value of the variance of the estimate of the independent variable on the mediator.
//' @param mu2 Value of the estimate of the mediator on the response.  
//' @param sig2 Value of the variance of the estimate of the mediator on the response. 
//' @param sig12 Value of the covariance between mu1 and mu2.
//' @param indL Value indicating the value of the independent variable used for the interaction. Typically 1.
//' @param mu3 Value of the estimate of the effect of the interaction of the independent and mediator variable on the response.
//' @param sig3 Value of the variance of the estimate of the effect of the interaction of the independent and mediator variable on the response.
//' @param sig13 Value of the covariance between mu1 and mu3.
//' @param sig23 Value of the covariance between mu2 and mu3.
//' @param mu1_0 Null value for mu1.
//' @param mu2_0 Null value for mu2.
//' @param mu3_0 Null value for mu3.
//' @returns A p-value for the test for the indirect effect.
//' @export
//' @examples
//' sobelTest_one(1, .1, .25, .01, .05)
// [[Rcpp::export]]
double sobelTest_one(const long double & mu1, const long double & sig1,
                     const long double & mu2, const long double & sig2, const long double & sig12,
                     const double & indL = 1,
                     const long double & mu3 = 0, const long double & sig3 = 0,
                     const long double & sig13 = 0, const long double & sig23 = 0,
                     double mu1_0 = 0, double mu2_0 = 0, double mu3_0 = 0){


  double deltaEst = mu1*(mu2 + indL*mu3);
  double deltaNull = mu1_0*(mu2_0 + indL*mu3_0);
  double deltaSE = std::sqrt(
                              std::pow(mu2 + indL*mu3, 2.0)*sig1 +
                              std::pow(mu1, 2.0)*(sig2 + std::pow(indL, 2.0)*sig3) +
                              2.0*( deltaEst*sig12 +
                                      std::pow(mu1, 2.0)*indL*sig23 +
                                      deltaEst*indL*sig13 )
                            );

  return(R::pnorm( -fabs( (deltaEst - deltaNull) / deltaSE), 0, 1, 1, 0)*2.0);
}
