% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/v1_stats_airr.R
\name{airr_stats}
\alias{airr_stats}
\alias{airr_stats_chains}
\alias{airr_stats_lengths}
\alias{airr_stats_genes}
\title{Compute key immune repertoire statistics}
\usage{
airr_stats_chains(
  idata,
  locus_col = NA,
  autojoin = getOption("immundata.autojoin", TRUE),
  format = c("long", "wide")
)

airr_stats_lengths(
  idata,
  seq_col = "cdr3_aa",
  autojoin = getOption("immundata.autojoin", TRUE),
  format = c("long", "wide")
)

airr_stats_genes(
  idata,
  gene_col = "v_call",
  level = c("receptor", "barcode"),
  by = c(NA, "locus"),
  autojoin = getOption("immundata.autojoin", TRUE),
  format = c("long", "wide")
)
}
\arguments{
\item{idata}{An \code{ImmunData} object.}

\item{locus_col}{Column in \code{idata$annotations} that stores the locus (e.g.
\code{"locus"}). If \code{NULL} or missing, the result is not split by locus.}

\item{autojoin}{Logical. If TRUE, join repertoire metadata by the schema repertoire id.
Change the default behaviour by calling \code{options(immunarch.autojoin = FALSE)}.}

\item{format}{String. One of \code{"long"} ("long" tibble with \code{imd_repertoire_id}, facet
columns, and \code{value}; useful for visualizations) or \code{"wide"} (wide/unmelted table of features,
with each row corresponding to a specific repertoire / pair of repertoires; useful for Machine Learning).}

\item{seq_col}{Character vector with names of the columns containing
sequences.}

\item{gene_col}{A single column name in \code{idata$annotations} with gene segment
calls (e.g., \code{"v_call"}, \code{"d_call"}, \code{"j_call"}, \code{"c_call"}). Default is
\code{"v_call"}.}

\item{level}{One of \code{"receptor"} or \code{"barcode"}. If \code{"receptor"} (default),
the function counts \strong{unique receptors} (one per receptor ID) that carry a
given gene segment. If \code{"barcode"}, the function \strong{sums counts} (e.g.,
cells/UMIs) per gene segment using the column defined by
\code{immundata::imd_schema("count")}.}

\item{by}{Either \code{NULL} (no split) or \code{"locus"}. When \code{"locus"}, the result
is further split by the locus column if present (as given by
\code{immundata::imd_schema("locus")}); otherwise a warning is emitted and the
split is ignored.}
}
\value{
\subsection{\code{airr_stats_chains} Returns a tibble with columns:}{
\itemize{
\item \code{repertoire_id} -- repertoire identifier
\item \code{locus} -- TRA, TRB, IGH, ... (present only if \code{locus_col} is supplied)
\item \code{n_chains} -- number of chains
}
}

\subsection{\code{airr_stats_lengths} Returns a tibble with columns:}{
\itemize{
\item \code{repertoire_id} -- repertoire identifier
\item \code{seq_len} -- lengths of sequences
\item \code{n} -- number of receptors
}
}

\subsection{\code{airr_stats_genes} A tibble with columns:}{
\itemize{
\item \code{repertoire_id} - repertoire identifier
\item \emph{(optional)} \code{locus} - TRA, TRB, IGH, ... (present only when \code{by = "locus"}
and the locus column exists)
\item \verb{<gene_col>} - the gene segment value (e.g., V gene)
\item \code{n} - the measure:
\itemize{
\item if \code{level = "receptor"}: number of receptors carrying the gene segment
\item if \code{level = "barcode"}: sum of counts across receptors for the segment
}
}
}
}
\description{
\ifelse{html}{\href{https://lifecycle.r-lib.org/articles/stages.html#experimental}{\figure{lifecycle-experimental.svg}{options: alt='[Experimental]'}}}{\strong{[Experimental]}}

A family of functions that extract \strong{core descriptive statistics} from an \code{ImmunData} object.
\subsection{Available functions}{

Supported methods are the following.
}

\code{airr_stats_chains} --- count V(D)J \emph{chains} per repertoire
(optionally split by locus). Quickly gauges capture depth per repertoire
and, when split by locus, reveals TRA/TRB/IGH balance. Use it for QC,
library-size checks, and to spot locus-specific dropouts or
over-representation.

\code{airr_stats_lengths} --- count the number of sequence lengths
per repertoire. Summarizes the CDR3 length distribution, a sensitive QC
fingerprint of repertoire prep and selection. Helpful for detecting
primer/UMI biases, comparing cohorts, and deriving length-based features for
models.

\code{airr_stats_genes} - count V(D)J gene segments per repertoire,
optionally split by locus and using either receptor counts or barcode/UMI
counts as the measure. Profiles V/D/J gene usage to characterize repertoire
composition and germline biases, with optional locus split. Useful for
cohort comparisons, flagging clonal expansions, and producing ML-ready
features for repertoire-level ML tasks.
}
\examples{
# Limit the number of threads used by the underlying DB for this session.
# Change this only if you know what you're doing (e.g., multi-user machines, shared CI/servers).
db_exec("SET threads TO 2")

# Load data
\dontrun{
immdata <- get_test_idata() |> agg_repertoires("Therapy")
}

#
# airr_stats_chains
#

\dontrun{
airr_stats_chains(immdata)
}

#
# airr_stats_lengths
#

\dontrun{
airr_stats_lengths(immdata)
}

#
# airr_stats_genes
#

\dontrun{
# V gene usage by receptor count
airr_stats_genes(immdata, gene_col = "v_call", level = "receptor")

# V gene usage by summed cell/UMI counts (if a count column is present)
airr_stats_genes(immdata, gene_col = "v_call", level = "barcode")

# Split by locus (TRA/TRB/... if locus column exists)
airr_stats_genes(immdata, gene_col = "v_call", level = "receptor", by = "locus")
}

}
\seealso{
\link[immundata:ImmunData]{immundata::ImmunData}
}
\concept{Key AIRR statistics}
