% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/read.emeld.R
\name{read.emeld}
\alias{read.emeld}
\title{Read an EMELD XML document containing an interlinearized corpus.}
\usage{
read.emeld(file, vernacular.languages, analysis.languages = "en",
  get.morphemes = TRUE, get.words = TRUE, get.sentences = TRUE,
  get.texts = TRUE, text.fields = c("title", "title-abbreviation", "source",
  "comment"), sentence.fields = c("segnum", "gls", "lit", "note"),
  words.vernacular.fields = "txt", words.analysis.fields = c("gls", "pos"),
  morphemes.vernacular.fields = c("txt", "cf"),
  morphemes.analysis.fields = c("gls", "msa", "hn"), sep = ";")
}
\arguments{
\item{file}{the path (or url) to a document in ELMED vocabulary}

\item{vernacular.languages}{character vector: one or more codes of languages analysed in the document.}

\item{analysis.languages}{character vector: one or more codes of languages used for the analyses (in glosses, translations, notes) in the document.}

\item{get.morphemes}{logical vector: should the returned list include a slot for the description of morphemes?}

\item{get.words}{logical vector: should the returned list include a slot for the description of words?}

\item{get.sentences}{logical vector: should the returned list include a slot for the description of sentences?}

\item{get.texts}{logical vector: should the returned list include a slot for the description of texts?}

\item{text.fields}{character vector: information to be extracted for the texts
(and turned into corresponding column in the data.frame describing texts)
The default are:
\itemize{
\item "title"
\item "title-abbreviation"
\item "source"
\item "comment"
}}

\item{sentence.fields}{character vector: information to be extracted for the sentences
(and turned into corresponding column in the data.frame describing sentences)
The default are:
\itemize{
\item "segnum" : an ID of the sentende
\item "gls": a translation (possibly in all analysis languages)
\item "lit": a litteral translation (possibly in all analysis languages)
\item "note": note (possibly in all analysis languages)
}}

\item{words.vernacular.fields}{character vector: information (in vernacular language(s))
to be extracted for the words (and turned into corresponding columns in the data.frame describing words)
The default are:
\itemize{
\item "txt" : the original text
}}

\item{words.analysis.fields}{character vector: information (in analysis language(s))
to be extracted for the words (and turned into corresponding columns in the data.frame describing words)
The default are:
\itemize{
\item "gls" : a gloss of the word
\item "pos" : the part of speech of the word
}}

\item{morphemes.vernacular.fields}{character vector: information (in vernacular language(s)) to be extracted for the morphemes (and turned into corresponding columns in the data.frame describing morphemes). May be null or empty.
\itemize{
\item "txt" : the text of the morpheme
\item "cf" : the canonical form of the morpheme
}}

\item{morphemes.analysis.fields}{character vector: information (in analysis language(s)) to be extracted for the morphemes (and turned into corresponding columns in the data.frame describing morphemes). May be null or empty.
\itemize{
\item "gls" : the gloss of the morpheme
\item "msa" : the part of speech of the morpheme
\item "hn" : a number for the identifiation of the morpheme amongst its homophone.
}}

\item{sep}{character vector: the character used to join multiple notes in the same language.}
}
\value{
a list with slots named "morphemes", "words", "sentences", "texts" 
(some slot may have been excluded throuth the "get.*" arguments, see above).
Each slot is a data.frame containing the information on the corresponding unit.
In each data.frame, each row describe an occurrence (the first row of the result$morphemes data.frame describe the first morpheme of the corpus).
In each data.frame, the first columns give ids refering to the line in other data.frame (so that we can link the first morpheme to the text, the sentence or the word it belongs to).
The following columns give information about the corresponding occurrence of the unit. Which information are extracted from the document and included in the data frame depends upton the *.fields parameters (see above).
Columns made are coined using the field name and the language code. For instance, if read.emeld is called with the parameters vernacular.languages="tww" and morphemes.vernacular.fields=c("txt", "cf"), then the column txt.tww and cf.tww will be created in the morphemes slot data frame.
}
\description{
The EMELD XML vocabulary has been proposed for the encoding of interlinear glosses.
It is used by the FieldWorks software (SIL FLEX) as an export format.
}
\details{
If several 'note' fields in the same language are present in a sentence,
they will be concatenated (see the "sep" argument)
}
\examples{
path <- system.file("exampleData", "tuwariInterlinear.xml", package="interlineaR")
corpus <- read.emeld(path, vernacular="tww", analysis="en")
head(corpus$morphemes)

# In some cases, one may have to combine information coming from various data.frame.
# Lets imagine one needs to have in the same data.frame the morphemes data 
# plus the "note" field attached to sentences:
# - The easy way is to combine all the columns of the two data frame 'morphemes' and 'sentence' :
combined <- merge(corpus$morphemes, corpus$sentences, by.x="sentence_id", by.y="sentence_id")
head(combined)

# - Alternatively, one may use vector extraction in order to add only the desired column
# to the morphemes data frame:
corpus$morphemes$note = corpus$sentences$note.en[ corpus$morphemes$sentence_id ]
head(corpus$morphemes)
}
\references{
Baden Hughes, Steven Bird and Catherine Bow \emph{Encoding and Presenting Interlinear Text Using XML Technologies}, http://www.aclweb.org/anthology/U03-1008

SIL FieldWorks: https://software.sil.org/fieldworks/
}
