% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/aiAssistant.R
\name{aiAssistant}
\alias{aiAssistant}
\title{aiAssistant: AI Assistant module server}
\usage{
aiAssistant(
  input,
  output,
  session,
  data,
  data_label,
  data_varStruct = NULL,
  api_key = NULL,
  stats_guide = NULL,
  show_api_config = TRUE,
  analysis_context = NULL
)
}
\arguments{
\item{input}{input}

\item{output}{output}

\item{session}{session}

\item{data}{Data (reactive). Should return the current data set each time it is called.}

\item{data_label}{Data label (reactive). Typically created with `jstable::mk.lev()`.}

\item{data_varStruct}{Variable structure list of data (reactive or NULL). If NULL, automatically generates `list(variable = names(data()))`. Can also be a reactive returning a named list with elements like `Base`, `Event`, `Time`, etc. Default: NULL}

\item{api_key}{API key for AI service. If NULL, reads from provider-specific environment variables (`ANTHROPIC_API_KEY`, `OPENAI_API_KEY`, `GOOGLE_API_KEY`) configured in `.Renviron` file}

\item{stats_guide}{Optional custom statistical guide text to override default guidelines. Can be a character string or reactive. If NULL, uses built-in statistical best practices guide. Useful for adding domain-specific statistical conventions or organizational standards.}

\item{show_api_config}{If TRUE, shows API config UI. If FALSE, uses only env vars. Default: TRUE}

\item{analysis_context}{Optional character string, list, or reactive returning that information. Used to pass prior analysis context that the AI can reference in follow-up questions.}
}
\value{
Server module (no explicit return value). Creates reactive outputs and observers for chat conversation history, generated code execution, analysis results (tables, plots, text), and export functionality.
}
\description{
AI-powered statistical analysis assistant module server
}
\details{
\itemize{
  \item `data` and `data_label` must be reactives; their values are re-evaluated every time the module needs data.
  \item `data_varStruct` can be NULL (auto-generated) or a reactive returning a named list with elements like `variable`, `Base`, `Event`, `Time`, etc. This mirrors the structure used by other *jsmodule* components.
  \item Generated code runs in a sandbox that only exposes the supplied data and allows the following packages: \code{jstable}, \code{jskm}, \code{jsmodule}, \code{survival}, \code{ggplot2}, \code{ggpubr}, \code{pROC}, \code{data.table}, \code{DT}, \code{gridExtra}, \code{GGally}, \code{forestploter}, \code{MatchIt}, \code{timeROC}.
  \item API keys are resolved in the order: explicit `api_key` argument, UI input (if `show_api_config = TRUE`), provider-specific environment variables (`ANTHROPIC_API_KEY`, `OPENAI_API_KEY`, `GOOGLE_API_KEY`).
  \item To use environment variables for API keys, add them to your `.Renviron` file (use `usethis::edit_r_environ()` to open it) with lines like: \cr
    `ANTHROPIC_API_KEY=your_key_here` \cr
    `OPENAI_API_KEY=your_key_here` \cr
    `GOOGLE_API_KEY=your_key_here` \cr
    Then restart R session for changes to take effect.
  \item `analysis_context` can be a static string/list or a reactive that returns a description of prior analyses (tables, plots, code snippets). The text is appended to the system prompt so the AI can reference earlier steps.
}
}
\examples{
\dontrun{
# Setup: Add API key to .Renviron file
# usethis::edit_r_environ()
# Add line: ANTHROPIC_API_KEY=your_actual_key_here
# Save and restart R

library(shiny)
library(DT)
library(survival)

# Example 1: Basic usage with auto-generated variable structure
ui <- fluidPage(
  titlePanel("AI Statistical Assistant"),
  aiAssistantUI("ai")
)

server <- function(input, output, session) {
  data <- reactive(colon)
  data.label <- reactive(jstable::mk.lev(colon))

  callModule(aiAssistant, "ai",
    data = data,
    data_label = data.label,
    data_varStruct = NULL  # Auto-generates variable structure
  )
}

shinyApp(ui, server)

# Example 2: With custom variable structure and analysis context
ui2 <- fluidPage(
  titlePanel("Survival Analysis Assistant"),
  aiAssistantUI("ai")
)

server2 <- function(input, output, session) {
  data <- reactive(colon)
  data.label <- reactive(jstable::mk.lev(colon))

  # Custom variable structure for survival analysis
  var_struct <- reactive({
    list(
      variable = names(colon),
      Base = c("rx", "sex", "age", "obstruct", "nodes"),
      Event = "status",
      Time = "time"
    )
  })

  callModule(aiAssistant, "ai",
    data = data,
    data_label = data.label,
    data_varStruct = var_struct,
    analysis_context = reactive({
      "Colon cancer adjuvant chemotherapy trial (survival::colon).
       Primary outcome: time to recurrence or death (status/time).
       Treatment groups: Observation, Levamisole, Levamisole+5-FU."
    })
  )
}

shinyApp(ui2, server2)

# Example 3: Production deployment without API config UI
ui_prod <- fluidPage(
  aiAssistantUI("ai", show_api_config = FALSE)
)

server_prod <- function(input, output, session) {
  # Relies entirely on .Renviron configuration
  callModule(aiAssistant, "ai",
    data = reactive(mtcars),
    data_label = reactive(jstable::mk.lev(mtcars)),
    show_api_config = FALSE
  )
}

shinyApp(ui_prod, server_prod)
}
}
