% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/minimizeMSE.R
\name{minimize_MSEHat}
\alias{minimize_MSEHat}
\title{Minimization of Estimated MSE}
\usage{
minimize_MSEHat(VarHat.scaled, BiasHat.squared, sigma, Ai, Bj, h, K, fnx,
  ticker = FALSE, plot = FALSE, ...)
}
\arguments{
\item{VarHat.scaled}{Vector of estimates of the scaled variance
(for values of \eqn{\sigma} in \code{sigma}).}

\item{BiasHat.squared}{Vector of estimates of the squared bias
(for values of \eqn{\sigma} in \code{sigma}).}

\item{sigma}{Numeric vector \eqn{(\sigma_1, \ldots, \sigma_s)} with
\eqn{s \ge 1}.}

\item{Ai}{Numeric vector expecting \eqn{(x_0 - X_1, \ldots, x_0 - X_n) / h},
where (usually) \eqn{x_0} is the point at which the density is to
be estimated for the data \eqn{X_1, \ldots, X_n} with
\eqn{h = n^{-1/5}}.}

\item{Bj}{Numeric vector expecting \eqn{(-J(1/n), \ldots, -J(n/n))} in case
of the rank transformation method, but \eqn{(\hat{\theta} - X_1,
\ldots, \hat{\theta} - X_n)} in case of the non-robust
Srihera-Stute-method. (Note that this the same as argument
\code{Bj} of \code{\link{adaptive_fnhat}}!)}

\item{h}{Numeric scalar, where (usually) \eqn{h = n^{-1/5}}.}

\item{K}{Kernel function with vectorized in- & output.}

\item{fnx}{\eqn{f_n(x_0) =} \code{mean(K(Ai))/h}, where here typically
\eqn{h = n^{-1/5}}.}

\item{ticker}{Logical; determines if a 'ticker' documents the iteration
progress through \code{sigma}. Defaults to FALSE.}

\item{plot}{Should graphical output be produced? Defaults to \code{FALSE}.}

\item{\ldots}{Currently ignored.}
}
\value{
A list with components \code{sigma.adap}, \code{msehat.min} and
        \code{discr.min.smaller} whose meanings are as follows:
  \tabular{ll}{
   \code{sigma.adap} \tab Found minimizer of MSE estimator. \cr
   \code{msehat.min} \tab Found minimum of MSE estimator. \cr
   \code{discr.min.smaller} \tab TRUE iff the numerically found minimum was
                            smaller than the discrete one.
   }
}
\description{
Minimization of the estimated MSE as function of \eqn{\sigma} in four steps.
}
\details{
Step 1: determine first (= smallest) maximizer of \code{VarHat.scaled} (!)
on the grid in \code{sigma}. Step 2: determine first (= smallest) minimizer
of estimated MSE on the \eqn{\sigma}-grid LEFT OF the first maximizer of
\code{VarHat.scaled}. Step 3: determine a range around the yet-found
(discrete) minimizer of estimated MSE within which a finer search for the
``true'' minimum is continued using numerical minimization. Step 4: check if
the numerically determined minimum is indeed better, i.e., smaller than the
discrete one; if not keep the first.
}
\examples{
require(stats)

set.seed(2017);     n <- 100;     Xdata <- sort(rnorm(n))
x0 <- 1;      Sigma <- seq(0.01, 10, length = 11)

h <- n^(-1/5)
Ai <- (x0 - Xdata)/h
fnx0 <- mean(dnorm(Ai)) / h   # Parzen-Rosenblatt estimator at x0.

 # For non-robust method:
Bj <- mean(Xdata) - Xdata
#  # For rank transformation-based method (requires sorted data):
# Bj <- -J_admissible(1:n / n)   # rank trafo

BV <- kader:::bias_AND_scaledvar(sigma = Sigma, Ai = Ai, Bj = Bj,
  h = h, K = dnorm, fnx = fnx0, ticker = TRUE)

kader:::minimize_MSEHat(VarHat.scaled = BV$VarHat.scaled,
  BiasHat.squared = (BV$BiasHat)^2, sigma = Sigma, Ai = Ai, Bj = Bj,
  h = h, K = dnorm, fnx = fnx0, ticker = TRUE, plot = FALSE)

}
