% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/lavPredict_parallel.R
\name{lavPredict_parallel}
\alias{lavPredict_parallel}
\title{Fast & robust parallel wrapper for lavaan::lavPredict()}
\usage{
lavPredict_parallel(
  fit,
  method = "ml",
  correct_extremes = TRUE,
  extreme_rule = c("auto", "abs", "z_by_se", "mad"),
  extreme_by = c("auto", "group", "global"),
  extreme_k = 4,
  extreme_eps = 1e-08,
  fallback_method = c("EMB", "EBM"),
  flag_column = FALSE,
  diagnostics = FALSE,
  workers = NULL,
  plan = c("auto", "multisession", "multicore", "sequential"),
  chunk_size = NULL,
  return_type = c("list", "data"),
  progress = FALSE,
  se = FALSE,
  prefix_se_fs = ".se_",
  distinct = c("never", "auto", "always"),
  distinct_threshold = 50000,
  ...
)
}
\arguments{
\item{fit}{lavaan model object.}

\item{method}{Character; estimation method passed to \code{lavaan::lavPredict()}
(e.g., \code{"ml"}, \code{"regression"}, \code{"EBM"}, \code{"EMB"}).}

\item{correct_extremes}{Logical; if \code{TRUE}, re-score only flagged rows via \code{fallback_method}.}

\item{extreme_rule}{One of \code{"auto"}, \code{"abs"}, \code{"z_by_se"}, \code{"mad"}.  See details above.}

\item{extreme_by}{One of \code{"auto"}, \code{"group"}, \code{"global"}.  If \code{"group"}, thresholds
are computed within groups.}

\item{extreme_k}{Numeric; rule-dependent threshold (default = 3.5).}

\item{extreme_eps}{Small numeric to guard division by zero in z-scores (default = 1e-8).}

\item{fallback_method}{Character; fallback method(s) for corrected rows.
Default tries \code{"EMB"} first, then \code{"EBM"} if that fails.}

\item{flag_column}{Logical; if \code{TRUE}, add column \code{.fs_corrected} marking corrected rows.}

\item{diagnostics}{Logical; if \code{TRUE}, attach diagnostic columns and attribute \code{fs_n_corrected}.}

\item{workers}{Integer; number of parallel workers
(default = \code{max(1, parallel::detectCores() - 1)}).}

\item{plan}{One of \code{"auto"}, \code{"multisession"}, \code{"multicore"}, \code{"sequential"}.
\code{"auto"} selects \code{"multisession"} cross-platform.}

\item{chunk_size}{Optional integer; number of rows per chunk (default computed adaptively).}

\item{return_type}{\code{"list"} or \code{"data"}; determines multi-group return format.}

\item{progress}{Logical; show \code{furrr} progress bar.}

\item{se}{Logical; if \code{TRUE} and model is continuous-only, attach standard-error
columns for factor scores.}

\item{prefix_se_fs}{Character; prefix for SE columns (default = \code{".se_"}).}

\item{distinct}{One of \code{"never"}, \code{"auto"}, \code{"always"}; controls use of \code{distinct()}
to shrink duplicated rows.}

\item{distinct_threshold}{Numeric; when \code{distinct = "auto"}, apply \code{distinct()}
only if \code{nrow(data) >= distinct_threshold} (default = 5e4). Ignored for mixed models.}

\item{...}{Additional arguments passed to \code{lavaan::lavPredict()}.}
}
\value{
A tibble (single-group) or a list/tibble (multi-group, depending on \code{return_type}),
containing predicted factor scores and optionally SEs and diagnostics.
}
\description{
Parallel, ordinal-aware, and mixed-safe implementation of \code{lavaan::lavPredict()}.
}
\details{
\subsection{Model-type behavior}{
\itemize{
\item \strong{Purely ordinal models:}
Optionally reduces duplicated work via \code{dplyr::distinct()} on full rows.
Prepends “all-categories” dummy rows per group to stabilize chunk predictions.
\item \strong{Mixed models (ordinal + continuous):}
No deduplication (predictions depend on continuous values).
Additionally prepends “variance-insurance” dummy rows to ensure non-zero variance
of all continuous variables within each group.
}
}

\subsection{Robust extreme-score handling}{

If \code{correct_extremes = TRUE}, only flagged rows are re-scored using a fallback method.
The flagging rule is controlled by \code{extreme_rule}:
\itemize{
\item \code{"auto"} (default): per-latent mixed metric — uses \code{"z_by_se"} where SE columns exist
for that latent, otherwise robust \code{"mad"}. A row is flagged if \strong{any} latent exceeds
the threshold; the maximum across latents is used.
\item \code{"z_by_se"}: flags rows where \code{abs(FS) / pmax(SE, eps) > extreme_k} for any latent.
\item \code{"mad"}: robust Z: \verb{abs(FS - median)/(1.4826 × MAD) > extreme_k}.
\item \code{"abs"}: simple absolute threshold: \code{abs(FS) > extreme_k}.
}

The re-scoring uses the first \code{fallback_method} supplied (default \code{"EMB"}),
and then automatically retries the other (e.g. \code{"EBM"}) if needed.

If \code{flag_column = TRUE}, a logical column \code{.fs_corrected} marks corrected rows.
If \code{diagnostics = TRUE}, columns \code{.fs_rule} and \code{.fs_metric} are attached and an
attribute \code{fs_n_corrected} is added.  The internal \code{"mixed"} rule is always
reported as \code{"auto"} for user clarity.
}
}
\examples{
\donttest{
# Convert selected indicators to ordinal
ord_items <- paste0("x", 1:9)
HS_ord <- lavaan::HolzingerSwineford1939
for (v in ord_items) {
  q <- stats::quantile(HS_ord[[v]], probs = seq(0, 1, length.out = 6), na.rm = TRUE)
  q <- unique(q)  # guard against duplicate cut points
  HS_ord[[v]] <- as.ordered(cut(HS_ord[[v]], breaks = q, include.lowest = TRUE))
}

HS.model <- '
  visual  =~ x1 + x2 + x3
  textual =~ x4 + x5 + x6
  speed   =~ x7 + x8 + x9
'

# Fit ordinal CFA model
fit_ord <- lavaan::cfa(
  HS.model,
  data             = HS_ord,
  ordered          = ord_items,
  estimator        = "WLSMV",
  parameterization = "delta",
  meanstructure    = TRUE
)

# Parallel prediction with automatic extreme-score handling
lavPredict_parallel(fit_ord, correct_extremes = TRUE)
}
}
