% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/prepare.R
\name{prepare}
\alias{prepare}
\title{Prepare smooth latent grids + model-based item curves (continuous, ordinal, or mixed)}
\usage{
prepare(
  fit,
  data = NULL,
  info = NULL,
  plan = c("auto", "none", "multisession", "multicore", "sequential", "cluster"),
  workers = NULL,
  cluster = NULL,
  ...
)
}
\arguments{
\item{fit}{A fitted \code{lavaan}/\code{blavaan} object.}

\item{data}{Optional factor-score table to reuse (either a single data frame
or a per-group list) as typically returned by \code{lavPredict_parallel()}.
If \code{NULL}, factor scores are obtained on demand by the sub-functions.}

\item{info}{Optional list from \code{model_info(fit)}. If \code{NULL}, it is computed once
and forwarded to both branches.}

\item{plan}{Parallelization backend for the \strong{ordinal} branch; one of
\code{c("auto", "none", "multisession", "multicore", "sequential", "cluster")}.
The continuous branch is fast and does not use futures.}

\item{workers}{Optional integer; number of workers used by future backends
(ignored unless \code{plan} uses a parallel backend). Forwarded to the ordinal branch.}

\item{cluster}{Optional external cluster object created by
\code{parallel::makeCluster()}; only used when \code{plan = "cluster"}. If supplied
with a different \code{plan}, it is ignored.}

\item{...}{Additional arguments passed unchanged to both sub-functions (e.g.,
\code{level}, \code{vcov_type}, \code{length.out}, \code{other_latents}, \code{latent_var_as_factor},
\code{se}, \code{se_summary}).}
}
\value{
A tibble. For mixed models, it is the full join of the two branches
using shared ID columns (see \strong{Merging}). For single-type models, it is the
single applicable branch.
}
\description{
Unified, robust wrapper that constructs synthetic latent-score grids and
model-based item curves for fitted \code{lavaan}/\code{blavaan} models. It tries both
internal branches and returns whichever applies:
\itemize{
\item \strong{Continuous branch} (for continuous indicators) and
\item \strong{Ordinal branch} (for ordinal indicators).
}

If both succeed (mixed models), their outputs are joined on shared ID columns.
The function is designed to avoid manual handling of factor-score data or group
columns; that logic is delegated to sub-functions.
}
\details{
\strong{Routing}
\itemize{
\item Calls internal functions \code{.prepare_continuous()} and \code{.prepare_ordinal()} for continuous and ordinal items, respectively.
\item Reuses \code{info} (from \code{model_info()}) and \code{data} (from \code{lavPredict_parallel()}) when provided, so each is computed at most once.
}

\strong{Merging}
\itemize{
\item If only one branch applies, returns that result.
\item If both apply, performs a \code{dplyr::full_join()} using shared keys among
\code{c(".rid", ".gid", ".group", ".latent_var")} when available, otherwise on the
intersection of column names.
\item To prevent duplicated factor-score columns (e.g., \code{A.x}, \code{E.y}), the latent
columns are kept from the \strong{continuous} branch and dropped from the ordinal
branch before the join.
}

\strong{Column semantics}
\itemize{
\item Leading ID columns: \code{.rid}, \code{.gid}, \code{.group}, \code{.latent_var}.
\item Latent columns follow directly after \code{.latent_var}.
\item Continuous/ordinal model-based outputs use the common
\verb{m_est_*}, \verb{m_lwr_*}, \verb{m_upr_*} naming convention.
}
}
\section{Parallelization}{

\itemize{
\item The function may set a safe fallback to \code{sequential} when executed inside a
worker to prevent nested futures. External clusters are respected only with
\code{plan = "cluster"}. \code{workers} is a hint for future backends and is ignored
otherwise.
}
}

\section{Errors and warnings}{

\itemize{
\item If neither branch succeeds, the function aborts with a diagnostic message.
\item When continuous and ordinal outputs have no shared columns, a row bind is
returned with a \code{.source} column and a warning is issued.
}
}

\examples{
# --- Continuous example --------------------------------------------------
HS.model <- 'visual  =~ x1 + x2 + x3
             textual =~ x4 + x5 + x6
             speed   =~ x7 + x8 + x9'
fit <- lavaan::cfa(HS.model,
                   data = lavaan::HolzingerSwineford1939,
                   meanstructure = TRUE)
prepare(fit)

\donttest{
# --- Ordinal example (discretize by quantiles; 5 ordered categories) -----
ord_items <- paste0("x", 1:9)
HS_ord <- lavaan::HolzingerSwineford1939
for (v in ord_items) {
  q <- stats::quantile(HS_ord[[v]], probs = seq(0, 1, length.out = 6), na.rm = TRUE)
  q <- unique(q)  # guard against duplicate cut points
  HS_ord[[v]] <- as.ordered(cut(HS_ord[[v]], breaks = q, include.lowest = TRUE))
}

fit_ord <- lavaan::cfa(
  HS.model,
  data             = HS_ord,
  ordered          = ord_items,
  estimator        = "WLSMV",
  parameterization = "delta",
  meanstructure    = TRUE
)
prepare(fit_ord)

# --- Mixed example (x1–x3 ordinal, others continuous) --------------------
mix_ord <- c("x1","x2","x3")
HS_mix  <- lavaan::HolzingerSwineford1939
for (v in mix_ord) {
  q <- stats::quantile(HS_mix[[v]], probs = seq(0, 1, length.out = 6), na.rm = TRUE)
  q <- unique(q)
  HS_mix[[v]] <- as.ordered(cut(HS_mix[[v]], breaks = q, include.lowest = TRUE))
}

fit_mix <- lavaan::cfa(
  HS.model,
  data             = HS_mix,
  ordered          = mix_ord,
  estimator        = "WLSMV",
  parameterization = "delta",
  meanstructure    = TRUE
)
prepare(fit_mix)
}

}
\seealso{
Internal helpers .prepare_continuous() and .prepare_ordinal() (not exported);
see also \code{model_info()} and \code{lavPredict_parallel()}.

Other lavDiag-augmenters: 
\code{\link{augment}()},
\code{\link{item_data}()}
}
\concept{lavDiag-augmenters}
