library(shiny)
library(leaflet)
library(leaflet.extras2)

ui <- fluidPage(
  tags$head(tags$style(".easyPrintHolder .customCssClass {
                            background-image: url(data:image/svg+xml;utf8;base64,PD94bWwgdmVyc2lvbj0iMS4wIiBlbmNvZGluZz0iaXNvLTg4NTktMSI/Pgo8IS0tIEdlbmVyYXRvcjogQWRvYmUgSWxsdXN0cmF0b3IgMTguMS4xLCBTVkcgRXhwb3J0IFBsdWctSW4gLiBTVkcgVmVyc2lvbjogNi4wMCBCdWlsZCAwKSAgLS0+CjxzdmcgeG1sbnM9Imh0dHA6Ly93d3cudzMub3JnLzIwMDAvc3ZnIiB4bWxuczp4bGluaz0iaHR0cDovL3d3dy53My5vcmcvMTk5OS94bGluayIgdmVyc2lvbj0iMS4xIiBpZD0iQ2FwYV8xIiB4PSIwcHgiIHk9IjBweCIgdmlld0JveD0iMCAwIDQ0NC44MzMgNDQ0LjgzMyIgc3R5bGU9ImVuYWJsZS1iYWNrZ3JvdW5kOm5ldyAwIDAgNDQ0LjgzMyA0NDQuODMzOyIgeG1sOnNwYWNlPSJwcmVzZXJ2ZSIgd2lkdGg9IjUxMnB4IiBoZWlnaHQ9IjUxMnB4Ij4KPGc+Cgk8Zz4KCQk8cGF0aCBkPSJNNTUuMjUsNDQ0LjgzM2gzMzQuMzMzYzkuMzUsMCwxNy03LjY1LDE3LTE3VjEzOS4xMTdjMC00LjgxNy0xLjk4My05LjM1LTUuMzgzLTEyLjQ2N0wyNjkuNzMzLDQuNTMzICAgIEMyNjYuNjE3LDEuNywyNjIuMzY3LDAsMjU4LjExNywwSDU1LjI1Yy05LjM1LDAtMTcsNy42NS0xNywxN3Y0MTAuODMzQzM4LjI1LDQzNy4xODMsNDUuOSw0NDQuODMzLDU1LjI1LDQ0NC44MzN6ICAgICBNMzcyLjU4MywxNDYuNDgzdjAuODVIMjU2LjQxN3YtMTA4LjhMMzcyLjU4MywxNDYuNDgzeiBNNzIuMjUsMzRoMTUwLjE2N3YxMzAuMzMzYzAsOS4zNSw3LjY1LDE3LDE3LDE3aDEzMy4xNjd2MjI5LjVINzIuMjVWMzR6ICAgICIgZmlsbD0iIzAwMDAwMCIvPgoJPC9nPgo8L2c+CjxnPgo8L2c+CjxnPgo8L2c+CjxnPgo8L2c+CjxnPgo8L2c+CjxnPgo8L2c+CjxnPgo8L2c+CjxnPgo8L2c+CjxnPgo8L2c+CjxnPgo8L2c+CjxnPgo8L2c+CjxnPgo8L2c+CjxnPgo8L2c+CjxnPgo8L2c+CjxnPgo8L2c+CjxnPgo8L2c+Cjwvc3ZnPgo=);
                            transform: rotate(90deg);
                        }
                       .easyPrintHolder .customCssClass1 {
                            background-image: url(data:image/svg+xml;utf8;base64,PD94bWwgdmVyc2lvbj0iMS4wIiBlbmNvZGluZz0iaXNvLTg4NTktMSI/Pgo8IS0tIEdlbmVyYXRvcjogQWRvYmUgSWxsdXN0cmF0b3IgMTguMS4xLCBTVkcgRXhwb3J0IFBsdWctSW4gLiBTVkcgVmVyc2lvbjogNi4wMCBCdWlsZCAwKSAgLS0+CjxzdmcgeG1sbnM9Imh0dHA6Ly93d3cudzMub3JnLzIwMDAvc3ZnIiB4bWxuczp4bGluaz0iaHR0cDovL3d3dy53My5vcmcvMTk5OS94bGluayIgdmVyc2lvbj0iMS4xIiBpZD0iQ2FwYV8xIiB4PSIwcHgiIHk9IjBweCIgdmlld0JveD0iMCAwIDQ0NC44MzMgNDQ0LjgzMyIgc3R5bGU9ImVuYWJsZS1iYWNrZ3JvdW5kOm5ldyAwIDAgNDQ0LjgzMyA0NDQuODMzOyIgeG1sOnNwYWNlPSJwcmVzZXJ2ZSIgd2lkdGg9IjUxMnB4IiBoZWlnaHQ9IjUxMnB4Ij4KPGc+Cgk8Zz4KCQk8cGF0aCBkPSJNNTUuMjUsNDQ0LjgzM2gzMzQuMzMzYzkuMzUsMCwxNy03LjY1LDE3LTE3VjEzOS4xMTdjMC00LjgxNy0xLjk4My05LjM1LTUuMzgzLTEyLjQ2N0wyNjkuNzMzLDQuNTMzICAgIEMyNjYuNjE3LDEuNywyNjIuMzY3LDAsMjU4LjExNywwSDU1LjI1Yy05LjM1LDAtMTcsNy42NS0xNywxN3Y0MTAuODMzQzM4LjI1LDQzNy4xODMsNDUuOSw0NDQuODMzLDU1LjI1LDQ0NC44MzN6ICAgICBNMzcyLjU4MywxNDYuNDgzdjAuODVIMjU2LjQxN3YtMTA4LjhMMzcyLjU4MywxNDYuNDgzeiBNNzIuMjUsMzRoMTUwLjE2N3YxMzAuMzMzYzAsOS4zNSw3LjY1LDE3LDE3LDE3aDEzMy4xNjd2MjI5LjVINzIuMjVWMzR6ICAgICIgZmlsbD0iIzAwMDAwMCIvPgoJPC9nPgo8L2c+CjxnPgo8L2c+CjxnPgo8L2c+CjxnPgo8L2c+CjxnPgo8L2c+CjxnPgo8L2c+CjxnPgo8L2c+CjxnPgo8L2c+CjxnPgo8L2c+CjxnPgo8L2c+CjxnPgo8L2c+CjxnPgo8L2c+CjxnPgo8L2c+CjxnPgo8L2c+CjxnPgo8L2c+CjxnPgo8L2c+Cjwvc3ZnPgo=);
                            transform: rotate(180deg);
                        }")
  ),
  leafletOutput("map", height = 700, width = "100%"),
  selectInput("scene", "Select Scene",
              choices = c("CurrentSize" = "CurrentSize",
                          "A4Landscape" = "A4Landscape",
                          "A4Portrait" = "A4Portrait",
                          "Custom Landscape" = "customCssClass",
                          "Custom Portrait" = "customCssClass1")),
  textInput("fn", "Filename", value = "map"),
  actionButton("print", "Print Map"),
  actionButton("rem", "removeEasyprint"),
  actionButton("cle", "clearControls")
)

server <- function(input, output, session) {
  output$map <- renderLeaflet({
    leaflet()  %>%
      addTiles(group = "basemap") %>%
      addCircleMarkers(data = leaflet::breweries91,
                       group = "markers", popup = ~brewery, label = ~brewery) %>%
      addPopups(data = leaflet::breweries91[1:5, ],
                group = "popups", popup = ~brewery) %>%
      addEasyprint(options = easyprintOptions(
        title = "Give me that map",
        position = "bottomleft",
        exportOnly = TRUE,
        # hideClasses = list("leaflet-overlay-pane", "leaflet-popup"),
        # hidden = FALSE, hideControlContainer = TRUE,
        filename = "mapit",
        tileLayer = "basemap",
        tileWait = 5000,
        defaultSizeTitles = list(
          "CurrentSize" = "The current map extent",
          "A4Landscape" = "A4 (Landscape) extent with w:1045, h:715",
          "A4Portrait" = "A4  (Portrait) extent with w:715, h:1045"
        ),
        # sizeModes = c("A4Portrait","A4Landscape"),
        sizeModes = list("CurrentSize" = "CurrentSize",
                         "A4Landscape" = "A4Landscape",
                         "A4Portrait" = "A4Portrait",
                         "Custom Landscape" = list(
                           width = 1800,
                           height = 700,
                           name = "A custom landscape size tooltip",
                           className = "customCssClass"),
                         "Custom Portrait" = list(
                           width = 700,
                           height = 1800,
                           name = "A custom portrait size tooltip",
                           className = "customCssClass1")
        ),
        customWindowTitle = "Some Fancy Title",
        customSpinnerClass = "shiny-spinner-placeholder",
        spinnerBgColor = "#b48484"))
  })
  observeEvent(input$print, {
    scene <- input$scene
    leafletProxy("map") %>%
      easyprintMap(sizeModes = scene, filename = input$fn)
  })
  observeEvent(input$rem, {
    leafletProxy("map") %>%
      removeEasyprint()
  })
  observeEvent(input$cle, {
    leafletProxy("map") %>%
      clearControls()
  })
}

shinyApp(ui, server)
