% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/coefLognorm.R
\name{getParmsLognormForMedianAndUpper}
\alias{getParmsLognormForMedianAndUpper}
\alias{getParmsLognormForMeanAndUpper}
\alias{getParmsLognormForLowerAndUpper}
\alias{getParmsLognormForLowerAndUpperLog}
\alias{getParmsLognormForModeAndUpper}
\alias{getParmsLognormForMoments}
\alias{getParmsLognormForExpval}
\title{Calculate mu and sigma of lognormal from summary statistics.}
\usage{
getParmsLognormForMedianAndUpper(median, upper, sigmaFac = qnorm(0.99))

getParmsLognormForMeanAndUpper(mean, upper, sigmaFac = qnorm(0.99))

getParmsLognormForLowerAndUpper(lower, upper, sigmaFac = qnorm(0.99))

getParmsLognormForLowerAndUpperLog(lowerLog, upperLog, sigmaFac = qnorm(0.99))

getParmsLognormForModeAndUpper(mle, upper, sigmaFac = qnorm(0.99))

getParmsLognormForMoments(mean, var, sigmaOrig = sqrt(var))

getParmsLognormForExpval(mean, sigmaStar)
}
\arguments{
\item{median}{geometric mu (median at the original exponential scale)}

\item{upper}{numeric vector: value at the upper quantile,
i.e. practical maximum}

\item{sigmaFac}{sigmaFac=2 is 95\% sigmaFac=2.6 is 99\% interval.}

\item{mean}{expected value at original scale}

\item{lower}{value at the lower quantile, i.e. practical minimum}

\item{lowerLog}{value at the lower quantile, i.e. practical minimum at log scale}

\item{upperLog}{value at the upper quantile, i.e. practical maximum at log scale}

\item{mle}{numeric vector: mode at the original scale}

\item{var}{variance at original scale}

\item{sigmaOrig}{standard deviation at original scale}

\item{sigmaStar}{multiplicative standard deviation}
}
\value{
numeric matrix with columns `mu` and `sigma`, the parameter of the
  lognormal distribution. Rows correspond to rows of inputs.
}
\description{
Calculate mu and sigma of lognormal from summary statistics.
}
\details{
For \code{getParmsLognormForMeanAndUpper}
there are two valid solutions, and the one with lower sigma
, i.e. the not so strongly skewed solution is returned.
}
\section{Functions}{
\itemize{
\item \code{getParmsLognormForMedianAndUpper}: Calculates mu and sigma of lognormal from median and upper quantile.

\item \code{getParmsLognormForMeanAndUpper}: Calculates mu and sigma of lognormal from mean and upper quantile.

\item \code{getParmsLognormForLowerAndUpper}: Calculates mu and sigma of lognormal from lower and upper quantile.

\item \code{getParmsLognormForLowerAndUpperLog}: Calculates mu and sigma of lognormal from lower and upper quantile at log scale.

\item \code{getParmsLognormForModeAndUpper}: Calculates mu and sigma of lognormal from mode and upper quantile.

\item \code{getParmsLognormForMoments}: Calculate mu and sigma from moments (mean anc variance)

\item \code{getParmsLognormForExpval}: Calculate mu and sigma from expected value and geometric standard deviation
}}

\examples{
# example 1: a distribution with mode 1 and upper bound 5
(thetaEst <- getParmsLognormForModeAndUpper(1,5))
mle <- exp(thetaEst[1] - thetaEst[2]^2)
all.equal(mle, 1, check.attributes = FALSE)

# plot the distributions
xGrid = seq(0,8, length.out = 81)[-1]
dxEst <- dlnorm(xGrid, meanlog = thetaEst[1], sdlog = thetaEst[2])
plot( dxEst~xGrid, type = "l",xlab = "x",ylab = "density")
abline(v = c(1,5),col = "gray")

# example 2: true parameters, which should be rediscovered
theta0 <- c(mu = 1, sigma = 0.4)
mle <- exp(theta0[1] - theta0[2]^2)
perc <- 0.975		# some upper percentile, proxy for an upper bound
upper <- qlnorm(perc, meanlog = theta0[1], sdlog = theta0[2])
(thetaEst <- getParmsLognormForModeAndUpper(
  mle,upper = upper,sigmaFac = qnorm(perc)) )

#plot the true and the rediscovered distributions
xGrid = seq(0,10, length.out = 81)[-1]
dx <- dlnorm(xGrid, meanlog = theta0[1], sdlog = theta0[2])
dxEst <- dlnorm(xGrid, meanlog = thetaEst[1], sdlog = thetaEst[2])
plot( dx~xGrid, type = "l")
#plot( dx~xGrid, type = "n")
#overplots the original, coincide
lines( dxEst ~ xGrid, col = "red", lty = "dashed")

# example 3: explore varying the uncertainty (the upper quantile)
x <- seq(0.01,1.2,by = 0.01)
mle = 0.2
dx <- sapply(mle*2:8,function(q99){
  theta = getParmsLognormForModeAndUpper(mle,q99,qnorm(0.99))
  #dx <- dDistr(x,theta[,"mu"],theta[,"sigma"],trans = "lognorm")
  dx <- dlnorm(x,theta[,"mu"],theta[,"sigma"])
})
  matplot(x,dx,type = "l")
# Calculate mu and sigma from expected value and geometric standard deviation
.mean <- 1
.sigmaStar <- c(1.3,2)
(parms <- getParmsLognormForExpval(.mean, .sigmaStar))
# multiplicative standard deviation must equal the specified value
cbind(exp(parms[,"sigma"]), .sigmaStar)

}
\references{
\code{Limpert E, Stahel W & Abbt M (2001)
Log-normal Distributions across the Sciences: Keys and Clues.
Oxford University Press (OUP) 51, 341,
10.1641/0006-3568(2001)051[0341:lndats]2.0.co;2}
}
