#' Summarize Components of an mKBO Decomposition
#'
#' Provides a summary of the modified Kitagawa-Blinder-Oaxaca (mKBO) decomposition for selected model terms or grouped categories of terms.
#' This function is useful for inspecting how specific variables (or sets of variables) contribute to the overall decomposition across groups.
#' @param x An object of class \code{"mkbo"}, as returned by the \code{\link{mkbo}} function.
#' @param term A character vector specifying one or more model terms (e.g., variable names) for which to summarize decomposition results. Use this argument to inspect contributions from specific variables.
#' @param term.cat A character string specifying a common prefix for a group of terms (typically dummy variables from a factor). This will summarize the decomposition results for all terms that match this pattern (e.g., \code{"education"} will match \code{"education.Bachelor"}, \code{"education.Master"}, etc.).
#' @return A \code{data.frame} with one row per group and the following columns:
#' \describe{
#'   \item{\code{group}}{Group identifier (from the grouping variable used in \code{mkbo}).}
#'   \item{\code{M}}{Group-specific mean of the outcome variable.}
#'   \item{\code{D}}{Difference from the reference (sample mean), as used in the mKBO decomposition.}
#'   \item{\code{R}}{Total explained difference (sum of E + C + I components).}
#'   \item{\code{E}}{Component of the difference due to endowments (differences in covariates).}
#'   \item{\code{C}}{Component due to coefficients (differences in effects).}
#'   \item{\code{I}}{Interaction component (joint difference in covariates and coefficients, conditional on E and C).}
#'   }
#' @importFrom stringr str_detect
#' @importFrom rlang .data
#' @examples
#' mkbo_output <- mkbo("PERNP ~ BACHELOR", group = "RACE", data=pums_subset)
#' mkbo_summary(mkbo_output, term="BACHELORTRUE")
#' @export
##
# Summary function, showing the results for a single or multiple parameters, or for all categories of a factor.

mkbo_summary <- function(x,  term=NULL, term.cat=NULL) {

  if(is.null(term) & is.null(term.cat)) stop("Either 'term' or 'term.cat' need to be specified.")
  if(!is.null(term) & !is.null(term.cat)) stop("'term' or 'term.cat' cannot both be specified.")

  if(!is.null(term)) {
    M <-  x$RECI[, "M"]
    D <- x$RECI[, "D"]
    E <- as.data.frame(t(colSums(x$E_var[which(x$E_var$parameter %in% term), -1])))
    C <- as.data.frame(t(colSums(x$C_var[which(x$C_var$parameter %in% term), -1])))
    I <- as.data.frame(t(colSums(x$I_var[which(x$I_var$parameter %in% term), -1])))
  }

  if(!is.null(term.cat)) {
    M <-  x$RECI[, "M"]
    D <- x$RECI[, "D"]
    E <- x$E_var %>% mutate(mkbo_select = str_detect(.data$parameter, paste(term.cat, ".", sep=""))) %>%
      filter(.data$mkbo_select) %>%
      select(-all_of(c("parameter", "mkbo_select"))) %>%
      summarise(across(dplyr::everything(), sum))
    C <- x$C_var %>% mutate(mkbo_select = str_detect(.data$parameter, paste(term.cat, ".", sep=""))) %>%
      filter(.data$mkbo_select) %>%
      select(-all_of(c("parameter", "mkbo_select"))) %>%
      summarise(across(dplyr::everything(), sum))
    I <- x$I_var %>% mutate(mkbo_select = str_detect(.data$parameter, paste(term.cat, ".", sep=""))) %>%
      filter(.data$mkbo_select) %>%
      select(-all_of(c("parameter", "mkbo_select"))) %>%
      summarise(across(dplyr::everything(), sum))

  }

  output <- data.frame(
    M = M,
    D = D,
    R = t(E) + t(C) + t(I),
    E = t(E),
    C = t(C),
    I = t(I))

  colnames(output) <- c("M", "D", "R", "E", "C", "I")

  output <- rownames_to_column(output, "group")

  return(output)

}
