% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/cliKoppen.R
\name{cliKoppenPoints}
\alias{cliKoppenPoints}
\title{Vegetation Classifier Using the KGC System}
\usage{
cliKoppenPoints(temp, prec, verbose = FALSE)
}
\arguments{
\item{temp}{'numeric' R object with one-year time series of monthly mean air temperature (in °C)}

\item{prec}{'numeric' R object with one-year time series of monthly precipitation sum (in mm)}

\item{verbose}{'logical' scalar that indicates whether or not values of the bioclimatic indices used should be
added to the output.}
}
\value{
Depending on the setting, a data frame with one or more columns where the KGC types are stored in the
    last (character) column, while the additional columns contain the values of bioclimatic indices used. The
    abbreviations of KGC types can be found in the data frame \code{\link[macroBiome]{vegClsNumCodes}}. If
    \code{verbose = FALSE}, the return object is a one-column data frame with the KGC types.
}
\description{
Calculates the values of bioclimatic indices used in the Köppen-Geiger classification (KGC) system
    (Köppen 1936), and designates the KGC type using these values, by using the monthly time series of temperature
    and precipitation. The classification scheme is based on the procedure described by Köppen (1936) and follows
    the modifications described by Peel et al. (2007).
}
\details{
To classify vegetation, the KGC system developed by Köppen (1936) and fine-tuned by Peel. et al. (2007)
    uses the values of the following 13 bioclimatic indices:

    \itemize{
      \item{\code{tap}: Total Annual Precipitation (in mm)}
      \item{\code{mat}: Mean Annual Temperature (in °C)}
      \item{\code{tw}: Mean Temperature of the Warmest Month (in °C)}
      \item{\code{tc}: Mean Temperature of the Coldest Month (in °C)}
      \item{\code{tm10}: Number of Months with Mean Temperature above 10°C (dimensionless)}
      \item{\code{pdry}: Precipitation Sum of the Driest Month (in mm)}
      \item{\code{psdry}: Precipitation Sum of the Driest Month in the Summer Half-Year (in mm)}
      \item{\code{pwdry}: Precipitation Sum of the Driest Month in the Winter Half-Year (in mm)}
      \item{\code{pswet}: Precipitation Sum of the Wettest Month in the Summer Half-Year (in mm)}
      \item{\code{pwwet}: Precipitation Sum of the Wettest Month in the Winter Half-Year (in mm)}
      \item{\code{ps}: Precipitation Sum of the Summer Half-Year (in mm)}
      \item{\code{pw}: Precipitation Sum of the Winter Half-Year (in mm)}
      \item{\code{pth}: Dryness Threshold (in mm)}
    }

    For details about calculating bioclimatic indices, see the function
    \code{\link[macroBiome]{cliBioCliIdxPoints}}. Since \code{pth} is more of a technical measure, it is not
    calculated by the function \code{\link[macroBiome]{cliBioCliIdxPoints}}. The value of \code{pth} depends on
    mean annual temperature and annual cycle of precipitation: \code{pth = 2 * mat} if >70 percentage of
    precipitation falls in winter half-year, (b) \code{pth = 2 * mat + 28} if >70 percentage of precipitation
    falls in summer half-year, and otherwise \code{pth = 2 * mat + 14}. For this index, the same definitions are
    used for seasons as in the function \code{\link[macroBiome]{cliBioCliIdxPoints}}, i.e., summer (winter)
    half-year is defined as the warmer (cooler) six month period of AMJJAS (from April to September) and ONDJFM
    (from October to March). \cr
    Numerous variants of the Köppen classification system are known (e.g., Köppen-Geiger classification: Köppen
    1936; Köppen-Trewartha classification: Trewartha and Horn 1980). Here, one of the most widely used versions of
    the Köppen-Geiger classification system is implemented, in accordance with works of Peel et al. (2007) and
    Beck et al. (2018). This classification system is the same as that presented by Köppen (1936) with three
    differences. First, classes \code{'C'} (temperate) and \code{'D'} (cold) are distinguished using a 0°C
    threshold instead of a -3°C threshold, following Russell (1931). Second, the sub-classes of the class
    \code{'B'} (arid) are identified depending on whether 70 percentage of precipitation falls in the summer or
    winter half-year. Third, the sub-classes \code{'s'} (dry summer) and \code{'w'} (dry winter) within the classes
    \code{'C'} and \code{'D'} are made mutually exclusive by assigning \code{'s'} when more precipitation falls in
    winter than in summer and assigning \code{'w'} otherwise. In this version, a total of 30 KGC types are
    distinguished (see \code{\link[macroBiome]{vegClsNumCodes}}).
}
\note{
As with any function with a point mode, a set of basic input data is defined here. In this case, they are as
    follows: \code{'temp'} (one-year time series of monthly mean air temperature), and \code{'prec'} (one-year
    time series of monthly precipitation sum). The objects \code{'temp'} and \code{'pre'} must be either vectors
    of length 12 or 12-column matrices. The first dimensions of these matrices have to be the same length. The
    function automatically converts vectors into single-row matrices during the error handling, and then uses
    these matrices. The first dimensions of these matrices determines the number of rows in the result matrix.
}
\examples{
# Loading mandatory data for the Example 'Points'
data(inp_exPoints)
data(vegClsNumCodes)

# Designate the KGC type (using the related bioclimatic indices),
# at a grid cell near Szeged, Hungary (46.3N, 20.2E) (for the normal period 1981-2010)
with(inp_exPoints, {
KGC <- cliKoppenPoints(colMeans(temp), colMeans(prec), verbose = TRUE)
numCode <- which(sapply(vegClsNumCodes$Code.KGC, identical, KGC[, "vegCls"]))
cbind(KGC[,-c(14)], vegClsNumCodes[numCode, c("Name.KGC", "Code.KGC")])
})

}
\references{
\cite{Beck HE, Zimmermann NE, McVicar TR, Vergopolan N, Berg A, Wood EF (2018) Present and future Köppen-Geiger
    climate classification maps at 1-km resolution. Sci Data 5:180214. \doi{10.1038/sdata.2018.214}}

\cite{Köppen W (1936) Das geographische System der Klimate. In: Köppen W, Geiger R (eds) Handbuch der
    Klimatologie. Verlag von Gebrüder Borntraeger, Berlin, Germany, pp 1–44}

\cite{Peel MC, Finlayson BL, McMahon TA (2007) Updated world map of the Köppen-Geiger climate classification.
    Hydrol Earth Syst Sci 11(5):1633–1644. \doi{10.5194/hess-11-1633-2007}}

\cite{Russell RJ (1931) Dry Climates of the United States: I. Climatic Map. University of California, Publications
    in Geography 5:1–41}

\cite{Trewartha GT, Horn LH (1980) An Introduction to Climate. Fifth Edition. McGraw-Hill, New York, NY}
}
