#' Mark correlation functions for inhomogeneous point patterns on Euclidean spaces.
#'
#' Mark correlation functions for inhomogeneous point patterns on Euclidean spaces.
#'
#' @usage \method{mcorrinhom}{ppp}(X,
#' ftype = c("variogram", "stoyan", "rcorr", "shimatani",
#'  "beisbart", "isham", "stoyancov", "schlather"),
#' r = NULL,
#' lambda = NULL,
#' method_lambda = c("kernel", "Voronoi"),
#' bw = bw.scott,
#' f = NULL,
#' method = c("density", "loess"),
#' correction = c("Ripley", "translate", "none"),
#' normalise = TRUE,
#' tol = 0.01,
#' ...)
#'
#' @param X An object of class ppp.
#' @param ftype Type of the test function \eqn{t_f}. Currently any selection of \code{"variogram", "stoyan", "rcorr", "shimatani", "beisbart", "isham", "stoyancov", "schlather"}.
#' @param r Optional. The values of the argument \code{r} at which the mark correlation function should be evaluated.
#' @param lambda Estimated intensity at data points. If not given, it will be estimated internally. See details.
#' @param method_lambda The method to be used for estimating intensity at data points, if \code{lambda = NULL}.
#' @param bw Bandwidth method to be used for estimating intensity at data points if \code{lambda = NULL} and \code{method_lambda = "kernel"}.
#' @param f  Optional. Test function \eqn{t_f} used in the definition of the mark correlation function. If \code{ftype} is given, \eqn{t_f} should be \code{NULL}.
#' @param method Type of smoothing, either \code{density} or \code{loess}.
#' @param correction Type of edge correction to be applied, either of \code{"Ripley", "translate", "none"}.
#' @param normalise If \code{normalise=FALSE}, only the numerator of the expression for the mark correlation function will be computed.
#' @param tol Tolerance used in the calculation of the conditional mean of marks. This is used only if \code{ftype} is \code{schlather}.
#' @param ... Arguments passed to \code{\link[spatstat.univar]{unnormdensity}} or \code{\link[stats]{loess}}.
#' @details
#' For an inhomogeneous point process \eqn{X} in \eqn{R^2}, the \eqn{t_f}-correlation function \eqn{\kappa_{t_f}^{inhom}(r)} is given as
#' \deqn{
#' \kappa_{t_f}^{inhom}(r)
#'     =
#'     \frac{
#'     \mathbb{E}_w \left[
#'     t_f \left(
#'     m_x, m_y
#'     \right) \mid x, y \in X
#'     \right]
#'     }{
#'     c_{t_f}
#'     },
#'     \quad
#'     d(x,y)=r,
#' }
#' where \eqn{m_x, m_y} are the marks of \eqn{x, y \in X}, \eqn{\mathbb{E}_w} is a conditional expectation with respect to a weighted Palm distribution, \eqn{c_{t_f}} is a normalising factor, and \eqn{d(x,y)=r} is the Euclidean distance. Therefore, each mark correlation function is defined by a specific test function \eqn{t_f(m_x, m_y)} and its associated normalising factor \eqn{c_{t_f}}. Let \eqn{\mu_m} and \eqn{\sigma^2_m} be the mean and variance of marks, then, the list below gives different test functions \eqn{t_f} and their normalised factors \eqn{c_{t_f}}, following distinct available \code{ftype}.
#'
#' \describe{
#'   \item{variogram:}{
#'     \eqn{t_f(m_x, m_y) = \frac{1}{2}(m_x - m_y)^2},
#'     \eqn{c_{t_f} = \sigma^2_m}.
#'   }
#'   \item{stoyan:}{
#'     \eqn{t_f(m_x, m_y) = m_x m_y},
#'     \eqn{c_{t_f} = \mu^2_m}.
#'   }
#'   \item{rcorr:}{
#'     \eqn{t_f(m_x, m_y) = m_x},
#'     \eqn{c_{t_f} = \mu_m}.
#'   }
#'   \item{shimatani:}{
#'     \eqn{t_f(m_x, m_y) = (m_x - \mu_m)(m_y - \mu_m)},
#'     \eqn{c_{t_f} = \sigma^2_m}.
#'   }
#'   \item{beisbart:}{
#'     \eqn{t_f(m_x, m_y) = m_x + m_y},
#'     \eqn{c_{t_f} = 2 \mu_m}.
#'   }
#'   \item{isham:}{
#'     \eqn{t_f(m_x, m_y) = m_x m_y - \mu^2_m},
#'     \eqn{c_{t_f} = \sigma^2_m}.
#'   }
#'   \item{stoyancov:}{
#'     \eqn{t_f(m_x, m_y) = m_x m_y - \mu^2_m},
#'     \eqn{c_{t_f} = 1}.
#'   }
#'   \item{schlather:}{
#'     \eqn{t_f(m_x, m_y) = (m_x - \mu_m(r))(m_y - \mu_m(r))},
#'     \eqn{c_{t_f} = \sigma^2_m}.
#'   }
#' }
#' For \code{ftype="schlather"}, \eqn{\mu_m(r)} denotes the mean of the marks of all pairs of points whose pairwise distance lies within a tolerance \code{tol} of \eqn{r}.
#' We refer to Eckardt and Moradi (2024) for details of these mark correlation functions.
#'
#'
#' Regarding the smoothing functions, if \code{method="density"}, the functions \code{\link[spatstat.univar]{unnormdensity}} will be called, and if \code{method="loess"}, the function \code{\link[stats]{loess}} will be called.
#'
#' If your \code{ftype} is not one of the defaults, then you need to give your test function \eqn{t_f(m_1, m_2)} using the argument \code{f}. In this case, \code{normalise} should be set as \code{FALSE}, as only the unnormalised version will be calculated. Depending on the form of the test function \eqn{t_f(m_1, m_2)}, one can manually compute the normalisation factor.
#'
#'
#'
#'
#' If \code{lambda = NULL}, the function internally estimates the intensity function using the given method via \code{method_lambda}. If \code{method_lambda = "kernel"}, the function calls
#' \code{\link[spatstat.explore]{density.ppp}}, with the bandwidth chosen by the given method \code{bw}, and argument \code{diggle=TRUE}. If \code{method_lambda = "Voronoi"}, then the functions calls
#' \code{\link[spatstat.explore]{densityVoronoi.ppp}} with arguments \code{f=0.2, nrep = 400} which are recommended by Moradi et al. (2019).
#'
#' Regarding the smoothing functions, if \code{method="density"}, the functions \code{\link[spatstat.univar]{unnormdensity}} will be called, and if \code{method="loess"}, the function \code{\link[stats]{loess}} will be called.
#'
#'
#' Type of edge correction is chosen among \code{"Ripley", "translate", "none"}. See  details in \code{\link[spatstat.explore]{edge.Trans}} and \code{\link[spatstat.explore]{edge.Ripley}}.
#'
#' If the point patten \eqn{X} has multiple real-valued marks, the function estimates the mark correlation function for each mark individually. In such case, marks are given as a \code{data.frame} whose columns represents different marks. The functions checks which columns are numeric, and for those the mark correlation function will be computed.
#'
#' @examples
#'  library(spatstat.geom)
#'  library(spatstat.random)
#'  library(spatstat.explore)
#'  X <- rpoispp(function(x,y) {100 * exp(-3*x)}, 100)
#'  marks(X) <- runif(npoints(X), 1, 10)
#'  mcorrinhom.ppp(X, ftype = "stoyan",
#'                 method = "density", correction = "translate",
#'                 method_lambda = "kernel", bw = bw.scott)
#' @return a data.frame which gives the estimated mark correlation function and the distance vector \eqn{r} at which the mark correlation function is estimated. If the point patten \eqn{X} has multiple real-valued marks, the estimated mark correlation function will be given for each mark. Name of columns will be the name of marks.
#' @author Mehdi Moradi \email{m2.moradi@yahoo.com} and Matthias Eckardt
#' @references
#' Moradi, M., Cronie, O., Rubak, E., Lachieze-Rey, R., Mateu, J., & Baddeley, A. (2019). Resample-smoothing of Voronoi intensity estimators. Statistics and computing, 29(5), 995-1010.
#'
#' Moradi, M., & Eckardt, M. (2025). Inhomogeneous mark correlation functions for general marked point processes. arXiv e-prints, arXiv-2505.
#' @seealso \code{\link[markstat]{mcorr.ppp}}, \code{\link[markstat]{mcorrinhom.lpp}}.

#' @import spatstat.univar
#' @import spatstat.random
#' @import spatstat.explore
#' @import spatstat.geom
#' @import spatstat.utils
#' @import stats
#' @export

mcorrinhom.ppp <- function(X,
                           ftype = c("variogram", "stoyan", "rcorr", "shimatani", "beisbart", "isham", "stoyancov", "schlather"),
                           r = NULL,
                           lambda = NULL,
                           method_lambda = c("kernel", "Voronoi"),
                           bw = bw.scott,
                           f = NULL,
                           method = c("density", "loess"),
                           correction = c("Ripley", "translate", "none"),
                           normalise = TRUE,
                           tol = 0.01,
                           ...){

  if (all(class(X) != "ppp")) stop("object X should be of class ppp.")

  if (is.null(f) & missing(ftype)) stop("ftype must be provided if 'f' is NULL.")

  if (missing(method)) stop("smoothing method should be chosen.")
  
  lambda_given <- lambda
  
  correction <- match.arg(correction)

  m <- marks(X)

  if (any(class(m) == "hyperframe" | class(m) == "data.frame")){
    m <- as.data.frame(m)
    num_cols <- unlist(sapply(m, is.numeric))
    s <- which(num_cols)
    
    out <- list()
    for (i in 1:length(s)) {
      marks(X) <- as.numeric(m[,s[i]])
      out[[i]] <- mcorrinhom.ppp(X, ftype = ftype, r = r, lambda = lambda, method_lambda = method_lambda, bw = bw, f = f,
                                    method = method, correction = correction, normalise = normalise,  tol = tol, ...)
    }

    r <- out[[1]]$r
    emps <- sapply(out, function(df) df$est)
    colnames(emps) <- names(s)
    finalout <- data.frame(r = r, emps)
    
    class(finalout) <- "mc"
    attr(finalout, "mtype") <- "real-valued"
    attr(finalout, "type") <- "global"
    attr(finalout, "ftype") <- ftype
    attr(finalout, "method") <- method
    attr(finalout, "lambda") <- lambda_given
    attr(finalout, "normalise") <- normalise
    attr(finalout, "method_lambda") <- method_lambda
    attr(finalout, "correction") <- correction
    attr(finalout, "bw") <- bw
    return(finalout)
  }

  if (is.null(f)) {
    if (ftype == "variogram") {
      f <- function(m1, m2, mu = NULL) 0.5 * ((m1 - m2)^2)
    } else if (ftype == "stoyan") {
      f <- function(m1, m2, mu = NULL) m1 * m2
    } else if (ftype == "rcorr") {
      f <- function(m1, m2, mu = NULL) m1
    } else if (ftype == "shimatani") {
      f <- function(m1, m2, mu = NULL) (m1 - mean(m)) * (m2 - mean(m))
    } else if (ftype == "beisbart") {
      f <- function(m1, m2, mu = NULL) m1 + m2
    } else if (ftype == "isham") {
      f <- function(m1, m2, mu = NULL) m1 * m2 - (mean(m))^2
    } else if (ftype == "stoyancov") {
      f <- function(m1, m2, mu = NULL) m1 * m2 - (mean(m))^2
    } else if (ftype == "schlather") {
      f <- function(m1, m2, mu = NULL) m1 * m2 - mu * (m1 + m2) + mu^2
    } else {
      stop("Your ftype is not supported!")
    }
  } else {
    warning("Your given test function is not among the default ones; only unnormalised version will be calculated.")
  }
  

  n <- npoints(X)
  d <- pairdist(X)

  if(is.null(r)){
    W <- X$window
    rmaxdefault <- rmax.rule("K", W, n/area(W))
    if(length(rmaxdefault)==0) {rmaxdefault <- 0.5 * max(d)}
    breaks <- handle.r.b.args(r, NULL, W, rmaxdefault = rmaxdefault)
    r <- breaks$r
  }

  rmax <- max(r)


  if(is.null(lambda)){

    if(method_lambda=="kernel"){

      lambda <- as.numeric(density(unmark(X), sigma = bw(X), at="points", diggle=T))

    }else if(method_lambda=="Voronoi"){

      lambda <- as.numeric(densityVoronoi(X, f=0.2, nrep = 400)[X])

    }else{

      stop("You need to pick a method for intensity estimation!")

    }
  }else{
    lambda <- lambda
  }


  if(correction=="translate"){

    close <- closepairs(X, rmax)
    XI <- ppp(close$xi, close$yi, window = X$window, check = FALSE)
    XJ <- ppp(close$xj, close$yj, window = X$window, check = FALSE)
    edgewt <- edge.Trans(XI, XJ, paired = TRUE)

  }else if(correction=="Ripley"){

    close <- closepairs(X, rmax)
    XI <- ppp(close$xi, close$yi, window = X$window, check = FALSE)
    edgewt <- edge.Ripley(XI, matrix(close$d, ncol = 1))

  }else if(correction=="none"){

    close <- closepairs(X, rmax)
    edgewt <- rep(1, length(close$d))

  }

  df <- cbind(
    dist = as.vector(d),
    id.row = rep(c(1:n), each=n),
    id.col = rep(c(1:n), n),
    int_i = rep(lambda, each=n),
    int_j = rep(lambda, n)
  )

  df.filter <- df[df[,1]<= rmax & df[,1]>0,]
  m1 <- m[df.filter[,2]]
  m2 <- m[df.filter[,3]]

  if (ftype=="schlather"){
    df.filter <- cbind(df.filter,
                       mu = as.numeric(unlist(sapply(df.filter[,1], function(d) {
                         matched <- df.filter[,3][abs(df.filter[,1] - d) <= tol]
                         paste(mean(m[matched]), collapse = ",")
                       }))))
    mu <- df.filter[,6]
    dfvario <- data.frame(d = df.filter[,1],
                         ff = (f(m1, m2, mu)),
                        int = df.filter[,"int_i"]*df.filter[,"int_j"],
                          w = edgewt)
  }else{
    dfvario <- data.frame(d = df.filter[,1],
                         ff = (f(m1,m2)),
                        int = df.filter[,"int_i"]*df.filter[,"int_j"],
                          w = edgewt
    )
  }



  if(method=="density"){

    Kf <- unnormdensity(dfvario$d, weights = dfvario$w*dfvario$ff/dfvario$int,
                        from=min(r), to=max(r), n=length(r), ...)$y
    K1 <- unnormdensity(dfvario$d, weights=dfvario$w/dfvario$int,
                        from=min(r), to=max(r), n=length(r), ...)$y
    Eff <- Kf/K1

  }else if(method=="loess"){

    lo <- loess(ff~d,data = dfvario,...)
    Eff <- predict(lo, newdata=data.frame(d=r))

  }else{
    stop("method should currently be either loess or density!!!")
  }


  if(normalise){
    if(ftype=="stoyan"){
      out <- Eff/(mean(m)^2)
    } else if(ftype=="variogram" | ftype=="isham" | ftype=="schlather" | ftype=="shimatani"){
      out <- Eff/var(m)
    }else if(ftype=="rcorr"){
      out <- Eff/mean(m)
    }else if(ftype=="Beisbart"){
      out <- Eff/(2*mean(m))
    }else if(ftype=="stoyancov"){
      out <- Eff
    }else{
      stop("your ftype is not supported!!")
    }
  }else{
    out <- Eff
  }


  out <- as.data.frame(cbind(r = r, est = out))
 
  if(ncol(out) == npoints(X) + 1 ) type <- "local" else type <- "global"
  
  class(out) <- "mc"
  attr(out, "mtype") <- "real-valued"
  attr(out, "type") <- type
  attr(out, "ftype") <- ftype
  attr(out, "method") <- method
  attr(out, "lambda") <- lambda_given
  attr(out, "normalise") <- normalise
  attr(out, "method_lambda") <- method_lambda
  attr(out, "correction") <- correction
  attr(out, "bw") <- bw

  return(out)
}
