% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/mhp.R
\name{mhp_batch}
\alias{mhp_batch}
\title{Batch Modified HP Filter}
\usage{
mhp_batch(X, max_lambda = 100000L)
}
\arguments{
\item{X}{Matrix or data.frame. Each column is a separate time series.
Rows represent time periods, columns represent different series.}

\item{max_lambda}{Integer. Maximum lambda value to search. Default 100000.}
}
\value{
A data.table in long format with columns:
\describe{
  \item{series}{Series identifier (column name or V1, V2, ...)}
  \item{t}{Time index (1, 2, ..., T)}
  \item{original}{Original series values}
  \item{trend}{Estimated trend component}
  \item{cycle}{Cyclical component}
}

With attribute \code{"lambdas"} containing a data.table of optimal lambdas
and GCV values for each series:
\describe{
  \item{series}{Series identifier}
  \item{lambda}{Optimal lambda for the series}
  \item{gcv}{GCV value at optimal lambda}
}
}
\description{
Process multiple time series efficiently using the Modified HP Filter.
Optimized for processing large collections of time series (e.g., panel data).
}
\details{
This function efficiently processes multiple series by:
1. Pre-computing the \eqn{K'K} matrix once for all series
2. Performing parallelizable grid search for each series
3. Using optimized C++ routines via RcppArmadillo
}
\examples{
\donttest{
# Example 1: Multiple macroeconomic series
set.seed(456)
n_periods <- 60
n_countries <- 5
gdp_matrix <- matrix(nrow = n_periods, ncol = n_countries)
colnames(gdp_matrix) <- c("USA", "UK", "Germany", "France", "Japan")

# Generate series with different characteristics
for (i in 1:n_countries) {
  trend <- cumsum(rnorm(n_periods, mean = 0.5, sd = 0.3))
  cycle <- rnorm(n_periods, sd = 1 + i * 0.2) # Increasing volatility
  gdp_matrix[, i] <- trend + cycle
}

# Apply batch Modified HP filter
results <- mhp_batch(gdp_matrix, max_lambda = 10000)

# Extract optimal lambdas
lambdas <- attr(results, "lambdas")
print(lambdas)

# Example 2: Sectoral data
set.seed(789)
n_time <- 120
n_sectors <- 8
sector_names <- c(
  "Agriculture", "Mining", "Manufacturing", "Construction",
  "Trade", "Transport", "Finance", "Services"
)
sector_data <- matrix(rnorm(n_time * n_sectors), nrow = n_time)

# Add sector-specific trends and cycles
for (i in 1:n_sectors) {
  trend_growth <- runif(1, 0.2, 1.0)
  cycle_amplitude <- runif(1, 0.5, 3.0)
  sector_data[, i] <- cumsum(rnorm(n_time, mean = trend_growth / 4, sd = 0.3)) +
    cycle_amplitude * sin(2 * pi * (1:n_time) / (20 + i * 5))
}
colnames(sector_data) <- sector_names

sector_results <- mhp_batch(sector_data, max_lambda = 50000)

# View results for first few periods
head(sector_results)
}

}
