% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/lr_age.R
\name{lr_age}
\alias{lr_age}
\title{Likelihood Ratio for Age Variable}
\usage{
lr_age(
  MPa = 40,
  MPr = 6,
  UHRr = 1,
  gam = 0.07,
  numsims = 1000,
  epa = 0.05,
  erRa = epa,
  H = 1,
  modelA = c("uniform", "custom")[1],
  LR = FALSE,
  seed = 1234,
  nsims = NULL
)
}
\arguments{
\item{MPa}{Numeric. Missing person's estimated age in years. Default: 40.}

\item{MPr}{Numeric. Age range tolerance (plus/minus years). The matching
interval is (MPa - MPr) to (MPa + MPr). Default: 6.}

\item{UHRr}{Numeric. Additional uncertainty range for the unidentified
person's age estimation. Default: 1.}

\item{gam}{Numeric. Gamma parameter for age uncertainty scaling.
The uncertainty interval is age +/- (gam * age + UHRr). Default: 0.07.}

\item{numsims}{Integer. Number of simulations to perform. Default: 1000.}

\item{epa}{Numeric (0-1). Error rate for age categorization. Default: 0.05.}

\item{erRa}{Numeric (0-1). Error rate in the reference/database.
Defaults to \code{epa}.}

\item{H}{Integer (1 or 2). Hypothesis to simulate under:
\itemize{
\item 1: H1 (Related) - Unidentified person IS the missing person
\item 2: H2 (Unrelated) - Unidentified person is NOT the missing person
}
Default: 1.}

\item{modelA}{Character. Reference age distribution model:
\itemize{
\item "uniform": Assumes uniform age distribution (default)
\item "custom": Uses empirical frequencies from simulations
}}

\item{LR}{Logical. If TRUE, compute and return LR values. Default: FALSE.}

\item{seed}{Integer. Random seed for reproducibility. Default: 1234.}

\item{nsims}{Deprecated. Use \code{numsims} instead.}
}
\value{
A data.frame with columns:
\itemize{
\item \code{group}: Age group classification ("T1" or "T0")
\item \code{age}: Simulated age value
\item \code{UHRmin}: Lower bound of uncertainty interval
\item \code{UHRmax}: Upper bound of uncertainty interval
\item \code{LRa}: Likelihood ratio (only if LR = TRUE)
}
}
\description{
Simulates age observations and optionally computes likelihood ratios (LRs)
under either H1 (unidentified person is the missing person) or H2
(unidentified person is not the missing person).

Ages are categorized into two groups based on whether they fall within
the missing person's estimated age range:
\itemize{
\item T1: Age within range (MPa - MPr) to (MPa + MPr)
\item T0: Age outside range
}
}
\details{
\strong{Under H1 (Related):}
Age is sampled to fall within the MP's range with probability (1 - erRa),
outside with probability erRa.

\strong{Under H2 (Unrelated):}
Age is sampled uniformly from 1-80, then categorized.

\strong{LR Calculation (uniform model):}
\itemize{
\item LR(T1) = (1 - epa) / P(T1), where P(T1) = 2*MPr/80
\item LR(T0) = epa / P(T0), where P(T0) = 1 - P(T1)
}
}
\examples{
# Simulate under H1 (related)
sim_h1 <- lr_age(MPa = 40, MPr = 6, H = 1, numsims = 100)
table(sim_h1$group)

# Simulate under H2 with LR values
sim_h2 <- lr_age(MPa = 40, MPr = 6, H = 2, numsims = 100, LR = TRUE)
head(sim_h2)

# Narrower age range (more discriminating)
sim_narrow <- lr_age(MPa = 35, MPr = 3, numsims = 500, LR = TRUE)
summary(sim_narrow$LRa)
}
\references{
Marsico FL, et al. (2023). "Likelihood ratios for non-genetic evidence
in missing person cases." \emph{Forensic Science International: Genetics},
66, 102891. \doi{10.1016/j.fsigen.2023.102891}
}
\seealso{
\code{\link{sim_lr_prelim}} for unified preliminary LR simulations,
\code{\link{lr_sex}}, \code{\link{lr_hair_color}} for other variables.
}
