#' Generate and Analyze Mixed-Level Blocked Factorial Designs
#'
#' Constructs blocked designs for mixed-level factorial experiments for a given
#' block size using finite-field based, collapsed, and heuristic methods. The
#' procedure creates the full treatment combination table, partitions runs into
#' blocks, and computes detailed confounding diagnostics for main effects and
#' two-factor interactions. The analyzer normalizes blocks into canonical labels,
#' checks balance and Orthogonal Factorial Structure (OFS), and computes
#' efficiencies of factorial effects. When OFS does not hold but the design has
#' equal treatment replications and equal block sizes, a general method based on
#' the C-matrix and custom contrast vectors is used to compute efficiencies. The
#' output includes GF-related metadata (when applicable), confounding summaries,
#' OFS diagnostics, and efficiency results.
#'
#' Internally, the algorithm:
#' \itemize{
#'   \item Generates candidate block structures (GF-based, collapsed, or heuristic).
#'   \item Computes confounding summaries for main effects and two-factor interactions.
#'   \item Normalizes blocks and checks balance and OFS.
#'   \item Computes efficiency factors for main and interaction effects.
#' }
#'
#' @param levels_vec Integer vector of factor levels
#'   (e.g., \code{c(2, 3, 4)} for a \eqn{2 \times 3 \times 4} design).
#' @param block_size Integer giving the number of runs per block.
#'   Must divide the total number of treatment combinations.
#' @param method Character string specifying the generator method:
#'   \itemize{
#'     \item \code{"auto"} (default): try GF, then collapsed, then heuristic.
#'     \item \code{"gf"}: finite-field based optimized generator.
#'     \item \code{"collapsed"}: random collapsed blocks.
#'     \item \code{"heuristic"}: heuristic ordering and blocking.
#'   }
#' @param verbose Logical; if \code{TRUE}, prints progress, summaries, and
#'   efficiency output.
#'
#' @return A list with components:
#' \describe{
#'   \item{code1}{Output from the generator, including \code{blocks},
#'     \code{confounding}, and (if applicable) \code{gf_info}.}
#'   \item{code2}{Output from the analyzer, including OFS and efficiency
#'     results.}
#'   \item{factor_levels}{The vector \code{levels_vec} supplied.}
#'   \item{block_size}{The block size used.}
#'   \item{blocks_numeric}{List of blocks with numeric factor values
#'     \code{F1}, \code{F2}, …}
#'   \item{blocks_labels}{List of blocks as character labels
#'     (e.g., \code{"012"}).}
#' }
#'
#' @examples
#' \donttest{
#'   out <- mixedfact(c(2, 3, 4), block_size = 12)
#'   str(out$code1)
#'   str(out$code2)
#' }
#'
#' @references
#' K. R. Nair and C. R. Rao (1948).
#' \emph{Confounding in Asymmetrical Factorial Experiments.}
#' Journal of the Royal Statistical Society: Series B (Methodological),
#' \strong{10}(1), 109-131.
#'
#' Gupta, S. and Mukerjee, R. (1989).
#' \emph{A Calculus for Factorial Arrangements.}
#' Lecture Notes in Statistics, Volume 59. Springer-Verlag.
#'
#' @importFrom stats as.formula contr.sum model.matrix runif
#' @importFrom utils head tail
#' @export

mixedfact <- function(levels_vec, block_size, method  = "auto", verbose = TRUE) {

  is_prime <- function(n) {
    if (n < 2) return(FALSE)
    if (n == 2) return(TRUE)
    if (n %% 2 == 0) return(FALSE)
    r <- floor(sqrt(n))
    for (i in seq(3, r, by = 2)) if (n %% i == 0) return(FALSE)
    TRUE
  }

  factorize_int <- function(n) {
    if (n <= 1) return(integer(0))
    out <- integer(0); d <- 2L
    while (n > 1L && d * d <= n) {
      while (n %% d == 0L) { out <- c(out, d); n <- n / d }
      d <- d + 1L
    }
    if (n > 1L) out <- c(out, as.integer(n))
    out
  }

  find_valid_GF_size <- function(levels_vec) {
    L <- max(levels_vec)
    primes <- c(2,3,5,7,11,13)
    powers <- c(1,2,3)
    pp <- integer(0)
    for (p in primes) for (m in powers) pp <- c(pp, p^m)
    pp <- sort(unique(pp))
    gf <- pp[pp >= L][1]
    if (is.na(gf)) stop("No GF(p^m) available for given levels (increase supported range).")
    found <- FALSE
    for (p in primes) for (m in powers) if (p^m == gf) { found <- TRUE; prime <- p; power <- m }
    if (!found) stop("Internal GF selection error.")
    list(gf = gf, p = prime, m = power)
  }

  get_irreducible_poly <- function(p, m) {
    if (m == 1) return(NULL)
    if (p == 2 && m == 2) return(c(1L,1L,1L))      # 1 + x + x^2
    if (p == 2 && m == 3) return(c(1L,1L,0L,1L))  # 1 + x + x^3
    if (p == 3 && m == 2) return(c(1L,0L,1L))     # 1 + x^2
    stop(sprintf("No built-in irreducible polynomial for p=%d m=%d in this script", p, m))
  }

  int_to_coeffs <- function(x, p, m) {
    x <- as.integer(x)
    coeffs <- integer(m)
    for (i in seq_len(m)) { coeffs[i] <- x %% p; x <- x %/% p }
    coeffs
  }

  coeffs_to_int <- function(coeffs, p) {
    res <- 0L; pow <- 1L
    for (c in coeffs) { res <- res + as.integer(c) * pow; pow <- pow * p }
    as.integer(res)
  }

  poly_mul_no_reduce <- function(a, b, p) {
    la <- length(a); lb <- length(b); res <- rep(0L, la + lb - 1L)
    for (i in seq_len(la)) for (j in seq_len(lb)) res[i + j - 1L] <- (res[i + j - 1L] + a[i] * b[j]) %% p
    res
  }

  poly_mod_reduce <- function(poly, mod_poly, p) {
    poly <- as.integer(poly %% p); mod_poly <- as.integer(mod_poly %% p)
    deg_mod <- length(mod_poly) - 1L
    while (length(poly) > 1 && tail(poly,1L) == 0L) poly <- poly[-length(poly)]
    while (length(poly) - 1L >= deg_mod) {
      deg_diff <- (length(poly) - 1L) - deg_mod
      lead_coef <- tail(poly,1L) %% p
      if (lead_coef != 0L) {
        for (i in seq_len(length(mod_poly))) {
          idx <- i + deg_diff
          poly[idx] <- (poly[idx] - lead_coef * mod_poly[i]) %% p
        }
      }
      while (length(poly) > 1 && tail(poly,1L) == 0L) poly <- poly[-length(poly)]
    }
    if (length(poly) < deg_mod) poly <- c(poly, rep(0L, deg_mod - length(poly)))
    as.integer(poly)
  }

  gf_add <- function(a, b, p, m) {
    va <- int_to_coeffs(a, p, m); vb <- int_to_coeffs(b, p, m)
    coeffs <- (va + vb) %% p; coeffs_to_int(coeffs, p)
  }

  gf_mul <- function(a, b, p, m, mod_poly) {
    if (m == 1) return(as.integer((as.integer(a) * as.integer(b)) %% p))
    va <- int_to_coeffs(a, p, m); vb <- int_to_coeffs(b, p, m)
    prod_poly <- poly_mul_no_reduce(va, vb, p)
    reduced <- poly_mod_reduce(prod_poly, mod_poly, p)
    coeffs_to_int(reduced, p)
  }

  make_gf_evaluator <- function(gf_size, p, m, mod_poly = NULL) {
    if (m == 1) {
      eval_fun <- function(tbl, coeffs) {
        vals <- as.matrix(tbl) %*% matrix(as.integer(coeffs) %% p, ncol = 1)
        ((vals %% p) + p) %% p
      }
      list(mode = "prime", p = p, m = m, eval = eval_fun)
    } else {
      eval_fun <- function(tbl, coeffs) {
        nr <- nrow(tbl); out <- integer(nr)
        for (i in seq_len(nr)) {
          s <- 0L
          for (j in seq_len(ncol(tbl))) {
            a <- as.integer(coeffs[j]) %% gf_size
            x <- as.integer(tbl[i,j]) %% gf_size
            prod <- gf_mul(a, x, p, m, mod_poly)
            s <- gf_add(s, prod, p, m)
          }
          out[i] <- as.integer(s %% gf_size)
        }
        out
      }
      list(mode = "prime_power", p = p, m = m, mod_poly = mod_poly, eval = eval_fun)
    }
  }

  generate_treatments <- function(levels_vec) {
    lv <- as.integer(levels_vec)
    grids <- lapply(lv, function(L) 0:(L-1))
    tr <- do.call(expand.grid, grids)
    names(tr) <- paste0("F", seq_along(lv))
    as.data.frame(tr, stringsAsFactors = FALSE)
  }

  compute_confounding_details <- function(treatments, blocks) {
    tr <- as.data.frame(treatments, stringsAsFactors = FALSE)
    n <- nrow(tr); p <- ncol(tr)

    # normalize blocks -> list of integer indices
    normalize_block_to_indices <- function(bl, tr) {
      if (is.numeric(bl) && all(bl %% 1 == 0) && all(bl >= 1) && all(bl <= nrow(tr))) return(as.integer(bl))
      if (is.data.frame(bl) || is.matrix(bl)) {
        blk_df <- as.data.frame(bl, stringsAsFactors = FALSE)
        if (ncol(blk_df) != ncol(tr)) stop("Block data.frame has different number of columns than treatments.")
        treat_keys <- apply(tr, 1, function(r) paste(as.character(r), collapse = "_"))
        blk_keys <- apply(blk_df, 1, function(r) paste(as.character(r), collapse = "_"))
        idxs <- match(blk_keys, treat_keys)
        if (any(is.na(idxs))) stop("Block contains treatments not found in full table.")
        return(as.integer(idxs))
      }
      if (is.character(bl)) {
        treat_keys <- apply(tr, 1, function(r) paste(as.character(r), collapse = "_"))
        idxs <- match(bl, treat_keys)
        if (any(is.na(idxs))) stop("Block contains treatments not found in full table.")
        return(as.integer(idxs))
      }
      stop("Unrecognized block format.")
    }

    block_indices <- lapply(blocks, function(b) normalize_block_to_indices(b, tr))
    blockid <- rep(NA_integer_, n)
    for (bi in seq_along(block_indices)) {
      idxs <- block_indices[[bi]]
      if (any(is.na(idxs))) stop("Block indices contain NA.")
      blockid[idxs] <- bi
    }
    if (any(is.na(blockid))) stop("Some treatments not assigned to a block.")

    B <- model.matrix(~ factor(blockid) - 1)
    Bc <- scale(B, center = TRUE, scale = FALSE)
    B_norms <- sqrt(colSums(Bc^2))
    eps <- .Machine$double.eps

    # orthogonal, centered contrasts for mains
    orth_contrast_centered <- function(f) {
      f <- factor(f, levels = sort(unique(as.character(f))))
      con <- contr.sum(length(levels(f)))
      M <- model.matrix(~ f, contrasts = list(f = con))[ , -1, drop = FALSE]
      if (is.null(dim(M))) M <- matrix(M, ncol = 1)
      colnames(M) <- paste0("C", seq_len(ncol(M)))
      scale(M, center = TRUE, scale = FALSE)
    }

    main_list <- lapply(seq_len(p), function(j) {
      fcol <- as.character(tr[[j]])
      cmat <- orth_contrast_centered(fcol)
      colnames(cmat) <- paste0("F", j, "_C", seq_len(ncol(cmat)))
      cmat
    })

    mm_main <- if (length(main_list) > 0) do.call(cbind, main_list) else matrix(0, nrow = n, ncol = 0)

    # two-factor interactions
    inter_list <- list()
    for (i in 1:(p-1)) for (j in (i+1):p) {
      f1 <- as.character(tr[[i]]); f2 <- as.character(tr[[j]])
      c1 <- orth_contrast_centered(f1); c2 <- orth_contrast_centered(f2)
      prod_list <- list()
      for (a in seq_len(ncol(c1))) for (b in seq_len(ncol(c2))) {
        prod_list[[length(prod_list) + 1]] <- c1[, a] * c2[, b]
      }
      if (length(prod_list) > 0) {
        tmp <- do.call(cbind, prod_list)
        colnames(tmp) <- paste0("F", i, "xF", j, "_C", seq_len(ncol(tmp)))
        inter_list[[paste0("F", i, "xF", j)]] <- scale(tmp, center = TRUE, scale = FALSE)
      }
    }
    mm_inter <- if (length(inter_list) > 0) do.call(cbind, inter_list) else matrix(0, nrow = n, ncol = 0)

    mm <- cbind(mm_main, mm_inter)

    # RawScore
    M_raw <- crossprod(mm, Bc)
    RawScore <- rowSums(abs(M_raw))

    # NormCovScore
    col_norms <- sqrt(colSums(mm^2)); mm_norm <- sweep(mm, 2, col_norms, "/"); mm_norm[is.nan(mm_norm)] <- 0
    M_norm <- crossprod(mm_norm, Bc); NormCovScore <- rowSums(abs(M_norm))

    # MaxAbsCorr
    MaxAbsCorr <- numeric(ncol(mm))
    if (ncol(mm) > 0) {
      for (r in seq_len(ncol(mm))) {
        x <- mm[, r]
        xn <- x / (sqrt(sum(x^2)) + eps)
        corrs <- as.numeric(crossprod(xn, Bc) / (B_norms + eps))
        MaxAbsCorr[r] <- max(abs(corrs))
      }
    }

    effect_names <- if (ncol(mm) > 0) colnames(mm) else character(0)
    is_main <- grepl("^F\\d+_C", effect_names)
    is_2fi  <- grepl("^F\\d+xF\\d+_C", effect_names)

    df <- data.frame(Effect = effect_names,
                     MaxAbsCorr = round(MaxAbsCorr, 2),
                     NormCovScore = round(NormCovScore, 2),
                     RawScore = round(RawScore, 2),
                     Type = ifelse(is_main, "Main", ifelse(is_2fi, "TwoFI", "Other")),
                     stringsAsFactors = FALSE)
    df <- df[order(-df$MaxAbsCorr, -df$NormCovScore, -df$RawScore), ]
    rownames(df) <- NULL

    list(df = df, mm = mm, Bc = Bc, blockid = blockid)
  }

  optimize_gf_generator <- function(levels_vec, block_size,
                                    initial_candidates = 120, random_restarts = 6, local_iters = 150,
                                    gf_size = NULL, p_base = NULL, m_pow = NULL, mod_poly = NULL,
                                    seed = 12345, verbose = FALSE) {
    set.seed(seed)
    if (is.null(gf_size)) {
      gf_choice <- find_valid_GF_size(levels_vec)
      gf_size <- gf_choice$gf; p_base <- gf_choice$p; m_pow <- gf_choice$m
    } else {
      if (is.null(p_base) || is.null(m_pow)) {
        facs <- factorize_int(gf_size)
        if (length(facs)==1 && facs[1]==gf_size) { p_base <- gf_size; m_pow <- 1 } else {
          primes <- c(2,3,5,7,11,13); found <- FALSE
          for (p in primes) for (m in 1:4) if (p^m == gf_size) { p_base <- p; m_pow <- m; found <- TRUE }
          if (!found) stop("Cannot infer p_base,m_pow for provided gf_size")
        }
      }
    }
    if (verbose) message(sprintf("Optimizer: using GF(%d) = %d^%d", gf_size, p_base, m_pow))
    if (is.null(mod_poly) && m_pow > 1) mod_poly <- get_irreducible_poly(p_base, m_pow)
    evaluator <- make_gf_evaluator(gf_size, p_base, m_pow, mod_poly)
    treatments <- generate_treatments(levels_vec); n <- nrow(treatments); p <- ncol(treatments)

    # helper to score blocks given last_vals
    score_blocks_two <- function(vals) {
      groups <- split(seq_len(n), as.character(vals)); uniq_vals <- sort(unique(vals))
      flat <- unlist(groups[as.character(uniq_vals)]); blocks <- split(flat, ceiling(seq_along(flat)/block_size))
      cs <- compute_confounding_details(treatments, lapply(blocks, function(idxs) treatments[idxs, , drop = FALSE]))$df
      main_sum <- sum(cs$MaxAbsCorr[cs$Type == "Main"])
      two_sum  <- sum(cs$MaxAbsCorr[cs$Type == "TwoFI"])
      c(main = main_sum, twofi = two_sum)
    }

    # initial candidates
    candidates <- list()
    for (j in 1:p) { v <- rep(0L,p); v[j] <- 1L; candidates <- c(candidates, list(as.integer(v))) }
    candidates <- c(candidates, list(as.integer((seq_len(p)-1L) %% gf_size)))
    if (gf_size > 2) for (k in 1:min(5,gf_size-1)) candidates <- c(candidates, list(as.integer((k * (seq_len(p)-1L)) %% gf_size)))
    for (i in 1:initial_candidates) candidates <- c(candidates, list(as.integer(sample(0:(gf_size-1), p, replace = TRUE))))

    cand_scores <- matrix(NA_real_, nrow = length(candidates), ncol = 2)
    for (i in seq_along(candidates)) {
      vals <- evaluator$eval(treatments, candidates[[i]])
      ss <- score_blocks_two(vals)
      cand_scores[i, ] <- ss
    }

    lexi_order <- order(cand_scores[,1], cand_scores[,2])
    Kbest <- unique(lexi_order[1:min(10, length(candidates))])
    best_pool <- candidates[Kbest]; best_scores <- cand_scores[Kbest,,drop=FALSE]
    overall_best <- list(coeffs = best_pool[[1]], score = best_scores[1,])

    neighbor <- function(vec) {
      v <- as.integer(vec); k <- sample(1:ceiling(max(1, p/3)), 1); idx <- sample.int(p, k)
      for (i in idx) {
        if (runif(1) < 0.6) {
          shift <- sample(c(-1,1),1); v[i] <- as.integer((v[i] + shift) %% gf_size)
        } else v[i] <- sample.int(gf_size,1) - 1L
      }
      v
    }

    for (restart in seq_len(random_restarts)) {
      if (restart <= length(best_pool)) curr <- best_pool[[restart]] else curr <- as.integer(sample(0:(gf_size-1), p, replace = TRUE))
      curr_vals <- evaluator$eval(treatments, curr); curr_score <- score_blocks_two(curr_vals)
      for (iter in seq_len(local_iters)) {
        prop <- neighbor(curr)
        prop_vals <- evaluator$eval(treatments, prop); prop_score <- score_blocks_two(prop_vals)
        if ((prop_score[1] < curr_score[1]) || (prop_score[1] == curr_score[1] && prop_score[2] < curr_score[2])) {
          curr <- prop; curr_score <- prop_score
          if ((curr_score[1] < overall_best$score[1]) || (curr_score[1] == overall_best$score[1] && curr_score[2] < overall_best$score[2])) {
            overall_best <- list(coeffs = curr, score = curr_score)
            if (verbose) message(sprintf("New best: main=%.6g, 2FI=%.6g", curr_score[1], curr_score[2]))
          }
        }
      }
    }

    final_vals <- evaluator$eval(treatments, overall_best$coeffs)
    groups <- split(seq_len(n), as.character(final_vals)); uniq_vals <- sort(unique(final_vals))
    flat <- unlist(groups[as.character(uniq_vals)]); final_blocks_idx <- split(flat, ceiling(seq_along(flat)/block_size))
    block_list <- lapply(final_blocks_idx, function(idx) treatments[idx, , drop = FALSE])

    scalar_score <- overall_best$score[1] + 0.001 * overall_best$score[2]

    list(gf_size = gf_size, p_base = p_base, m_pow = m_pow, coeffs = overall_best$coeffs,
         blocks = block_list, confounding = scalar_score, treatments = treatments, last_vals = final_vals)
  }

  generate_collapsed_blocks <- function(levels_vec, block_size) {
    trt <- generate_treatments(levels_vec); n <- nrow(trt)
    if (n %% block_size != 0) stop("block_size must divide total treatments")
    idx <- sample.int(n)
    blocks <- lapply(split(idx, ceiling(seq_along(idx)/block_size)), function(idxs) trt[idxs, , drop = FALSE])
    list(method = "collapsed", blocks = blocks, treatments = trt, confounding = NA_real_, coeffs = NULL)
  }

  generate_heuristic_blocks <- function(levels_vec, block_size) {
    trt <- generate_treatments(levels_vec); n <- nrow(trt)
    if (n %% block_size != 0) stop("block_size must divide total treatments")
    scv <- rowSums(trt); ord <- order(scv, runif(n)*0.1); idx <- ord
    blocks <- lapply(split(idx, ceiling(seq_along(idx)/block_size)), function(idxs) trt[idxs, , drop = FALSE])
    list(method = "heuristic", blocks = blocks, treatments = trt, confounding = NA_real_, coeffs = NULL)
  }

  generate_design_report <- function(levels_vec, block_size, method = c("auto","gf","collapsed","heuristic"), verbose = TRUE) {
    method <- match.arg(method)
    res <- NULL
    if (method == "gf") {
      res <- optimize_gf_generator(levels_vec, block_size, verbose = verbose, seed = as.integer(Sys.time()) %% 1e6)
    } else if (method == "collapsed") {
      res <- generate_collapsed_blocks(levels_vec, block_size)
    } else if (method == "heuristic") {
      res <- generate_heuristic_blocks(levels_vec, block_size)
    } else { # auto
      if (verbose) cat("--- Auto: evaluating 'gf', 'collapsed', 'heuristic' ---\n")
      gf_try <- try(optimize_gf_generator(levels_vec, block_size, verbose = verbose, seed = as.integer(Sys.time()) %% 1e6), silent = TRUE)
      if (!inherits(gf_try, "try-error")) {
        res <- gf_try
      } else {
        if (verbose) message("GF failed: ", attr(gf_try, "condition")$message)
        col_try <- try(generate_collapsed_blocks(levels_vec, block_size), silent = TRUE)
        if (!inherits(col_try, "try-error")) { res <- col_try } else {
          heu_try <- try(generate_heuristic_blocks(levels_vec, block_size), silent = TRUE)
          if (!inherits(heu_try, "try-error")) { res <- heu_try } else stop("All methods failed.")
        }
      }
    }

    blocks <- res$blocks
    treatments <- if (!is.null(res$treatments)) res$treatments else generate_treatments(levels_vec)
    nblocks <- length(blocks)

    block_rows <- do.call(rbind, lapply(seq_along(blocks), function(i) {
      df <- as.data.frame(blocks[[i]], stringsAsFactors = FALSE)
      df$block <- i
      df[, c("block", setdiff(names(df), "block")), drop = FALSE]
    }))

    pooled_det <- compute_confounding_details(treatments, lapply(blocks, function(df) df))
    conf_df <- pooled_det$df

    if (verbose) {
      cat("Design generated.\n")
      if (!is.null(res$gf_size)) {
        cat(sprintf("GF size: %d (= %d^%d)\n", res$gf_size, res$p_base, res$m_pow))
      }
      if (!is.null(res$coeffs)) {
        cat("Coefficients:", paste(res$coeffs, collapse = " "), "\n")
      } else {
        cat("Coefficients: NA\n")
      }
      cat("Number of blocks:", nblocks, "\n")
      if (!is.null(res$confounding)) {
        cat(sprintf("Confounding score (optimizer): %.4f\n\n", res$confounding))
      } else {
        cat("Confounding score (optimizer): NA\n\n")
      }

      cat("Block\tTreatments\n")
      for (i in seq_len(nblocks)) {
        df <- blocks[[i]]
        cols <- grep("^F\\d+$", names(df), value = TRUE)
        rows_str <- apply(df[, cols, drop = FALSE], 1, function(r) paste(r, collapse = ","))
        cat(i, "\t", paste(rows_str, collapse = " | "), "\n", sep = "")
      }

      cat("\nConfounding summary\n")
      toprint <- conf_df[, c("Effect", "MaxAbsCorr", "NormCovScore", "RawScore")]
      print(toprint, row.names = FALSE)
    }

    invisible(list(blocks = blocks,
                   confounding = conf_df,
                   gf_info = if (!is.null(res$gf_size))
                     list(gf_size = res$gf_size, p = res$p_base, m = res$m_pow, coeffs = res$coeffs)
                   else NULL))
  }

  normalize_blocks_for_analyzer <- function(levels, raw_blocks) {
    m <- length(levels)
    make_treatments <- function(levels) {
      grids <- do.call(expand.grid, lapply(rev(levels), function(s) 0:(s - 1)))
      grids <- grids[, rev(seq_len(ncol(grids))), drop = FALSE]
      apply(grids, 1, paste0, collapse = "")
    }
    canonical <- make_treatments(levels)

    normalize_one_entry <- function(entry) {
      if (is.numeric(entry)) {
        if (length(entry) == 0) return(NA_character_)
        if (length(entry) == 1 && length(levels) > 1) {
          return(NA_character_)
        }
        vals <- as.integer(entry)
        if (length(vals) != m) return(NA_character_)
        if (any(is.na(vals))) return(NA_character_)
        ok <- all(mapply(function(x, s) (x >= 0 && x <= s - 1), vals, levels))
        if (!ok) return(NA_character_)
        return(paste0(vals, collapse = ""))
      }
      if (is.character(entry)) {
        if (length(entry) > 1) {
          tokens <- entry
        } else {
          s <- entry
          s <- trimws(s)
          if (grepl("[,\\s]", s)) {
            tokens <- unlist(strsplit(s, "[,\\s]+"))
            tokens <- tokens[nzchar(tokens)]
          } else {
            digits <- unlist(regmatches(s, gregexpr("\\d+", s)))
            if (length(digits) == 0) {
              return(NA_character_)
            }
            if (length(digits) == m && all(nchar(digits) == 1)) {
              tokens <- digits
            } else if (length(s) == m) {
              tokens <- strsplit(s, "")[[1]]
            } else if (length(digits) == 1 && nchar(digits) == m) {
              tokens <- strsplit(digits, "")[[1]]
            } else {
              if (nchar(s) == m) tokens <- strsplit(s, "")[[1]] else return(NA_character_)
            }
          }
        }
        nums <- suppressWarnings(as.integer(tokens))
        if (any(is.na(nums)) || length(nums) != m) return(NA_character_)
        ok <- all(mapply(function(x, s) (x >= 0 && x <= s - 1), nums, levels))
        if (!ok) return(NA_character_)
        return(paste0(nums, collapse = ""))
      }
      if (is.factor(entry)) return(normalize_one_entry(as.character(entry)))
      if (is.list(entry) && length(entry) == m && all(sapply(entry, function(u) is.atomic(u)))) {
        tokens <- unlist(lapply(entry, function(u) {
          if (is.numeric(u)) as.character(as.integer(u)) else as.character(u)
        }))
        nums <- suppressWarnings(as.integer(tokens))
        if (any(is.na(nums))) return(NA_character_)
        ok <- all(mapply(function(x, s) (x >= 0 && x <= s - 1), nums, levels))
        if (!ok) return(NA_character_)
        return(paste0(nums, collapse = ""))
      }
      if (is.vector(entry) && length(entry) == m) {
        return(normalize_one_entry(as.character(entry)))
      }
      NA_character_
    }

    out_blocks <- vector("list", length(raw_blocks))
    bads <- list()
    for (bi in seq_along(raw_blocks)) {
      blk <- raw_blocks[[bi]]
      entries <- NULL
      if (is.data.frame(blk) || is.matrix(blk)) {
        entries <- lapply(seq_len(nrow(blk)), function(i) as.character(blk[i, ]))
      } else if (is.list(blk) && !is.character(blk)) {
        entries <- blk
      } else if (is.atomic(blk) && length(blk) > 1) {
        entries <- as.list(as.character(blk))
      } else if (is.atomic(blk) && length(blk) == 1) {
        s <- as.character(blk)
        if (grepl("[\\t;,]", s)) {
          entries <- list(s)
        } else {
          entries <- list(s)
        }
      } else {
        entries <- as.list(blk)
      }
      norm_entries <- character(0)
      for (ei in seq_along(entries)) {
        ent <- entries[[ei]]
        normalized <- normalize_one_entry(ent)
        if (is.na(normalized)) {
          bads[[length(bads) + 1]] <- list(block = bi, index = ei, raw = ent)
        } else {
          norm_entries <- c(norm_entries, normalized)
        }
      }
      out_blocks[[bi]] <- norm_entries
    }

    invalid_list <- list()
    for (bi in seq_along(out_blocks)) {
      for (j in seq_along(out_blocks[[bi]])) {
        t <- out_blocks[[bi]][j]
        if (!(t %in% canonical)) {
          invalid_list[[length(invalid_list) + 1]] <- list(block = bi, idx = j, treatment = t)
        }
      }
    }

    if (length(bads) > 0 || length(invalid_list) > 0) {
      msg_parts <- character(0)
      if (length(bads) > 0) {
        bad_examples <- sapply(head(bads, 5), function(z) {
          rawtxt <- if (is.atomic(z$raw)) {
            paste0(as.character(z$raw), collapse = " ")
          } else {
            "<complex>"
          }
          paste0("[block=", z$block, ", idx=", z$index, ", raw='", rawtxt, "']")
        })
        msg_parts <- c(
          msg_parts,
          paste0(
            length(bads), " raw entries could not be parsed to treatments. Examples: ",
            paste(bad_examples, collapse = "; "),
            " ..."
          )
        )
      }
      if (length(invalid_list) > 0) {
        invalid_examples <- sapply(head(invalid_list, 5), function(z) {
          paste0("[block=", z$block, ", treatment=", paste(z$treatment, collapse = ","), "]")
        })
        msg_parts <- c(
          msg_parts,
          paste0(
            length(invalid_list), " treatments do not match valid treatment set. Examples: ",
            paste(invalid_examples, collapse = "; "),
            " ..."
          )
        )
      }
      stop(paste(msg_parts, collapse = "\n"))
    }

    out_blocks
  }

  analyze_design <- function(factor_levels, blocks, verbose = FALSE) {
    # normalize blocks first
    blocks_norm <- normalize_blocks_for_analyzer(factor_levels, blocks)
    levels <- factor_levels

    mpinv <- function(A, tol = 1e-8) {
      sv <- svd(A)
      d <- ifelse(sv$d > tol, 1/sv$d, 0)
      sv$v %*% diag(d, nrow = length(d)) %*% t(sv$u)
    }

    is_zero_vec <- function(x, tol = 1e-10) all(abs(x) < tol)

    orthonormal_contrast_full <- function(si) {
      if (si < 2) stop("si must be >= 2")
      P <- matrix(0, nrow = si - 1, ncol = si)
      for (r in 1:(si - 1)) {
        P[r, 1:r] <- 1 / sqrt(r * (r + 1))
        P[r, r + 1] <- -r / sqrt(r * (r + 1))
      }
      P
    }

    generate_y_tuples <- function(m) {
      tuples <- as.matrix(expand.grid(replicate(m, 0:1, simplify = FALSE)))
      tuples <- tuples[rowSums(tuples) != 0, , drop = FALSE]
      tuples
    }

    generate_labels_from_tuples <- function(tuples) {
      apply(tuples, 1, function(row) {
        idx <- which(row == 1)
        paste0(paste(paste0("F", idx), collapse = ""))
      })
    }

    compute_p_y <- function(y, s_levels) {
      m <- length(y)
      p_list <- list()
      for (i in 1:m) {
        if (y[i] == 1) {
          p_list[[i]] <- orthonormal_contrast_full(s_levels[i])
        } else {
          p_list[[i]] <- matrix(rep(1 / sqrt(s_levels[i]), s_levels[i]), nrow = 1)
        }
      }
      Reduce(kronecker, p_list)
    }

    build_incidence <- function(levels, blocks) {
      grids <- do.call(expand.grid, lapply(rev(levels), function(s) 0:(s - 1)))
      grids <- grids[, rev(seq_len(ncol(grids)))]
      trt_labels <- apply(grids, 1, paste0, collapse = "")
      v <- nrow(grids); b <- length(blocks)

      N <- matrix(0, nrow = v, ncol = b)
      rownames(N) <- trt_labels
      colnames(N) <- paste0("B", seq_len(b))

      for (h in seq_len(b)) {
        for (trt in blocks[[h]]) {
          pos <- match(trt, trt_labels)
          if (is.na(pos)) stop(sprintf("Block %d contains treatment '%s' not in treatment list", h, trt))
          N[pos, h] <- N[pos, h] + 1
        }
      }
      list(N = N, treatments = trt_labels, grids = as.matrix(grids))
    }

    compute_Cmatrix <- function(N) {
      N <- as.matrix(N)
      r <- as.vector(rowSums(N))
      k <- as.vector(colSums(N))
      if (length(k) == 1) {
        NkN <- (N %*% t(N)) / k
      } else {
        NkN <- N %*% diag(1 / k, nrow = length(k)) %*% t(N)
      }
      C <- diag(r, nrow = length(r)) - NkN
      list(C = C, r = r, k = k)
    }

    check_proper <- function(N, tol = 1e-8) {
      k <- colSums(N)
      proper <- (max(k) - min(k)) < tol
      list(proper = proper, block_sizes = k)
    }

    check_equireplicate <- function(N, tol = 1e-8) {
      r <- rowSums(N)
      equirep <- (max(r) - min(r)) < tol
      list(equireplicate = equirep, replicates = r)
    }

    check_design_balance_OFS <- function(levels, blocks, tol = 1e-8, verbose = TRUE) {
      decode_treatment <- function(trt) {
        as.numeric(strsplit(trt, "")[[1]])
      }
      code_levels <- function(s) {
        if (s %% 2 == 1) {
          seq(-(s - 1) / 2, (s - 1) / 2, by = 1)
        } else {
          seq(-(s - 1), (s - 1), by = 2)
        }
      }

      all_treatments <- unique(unlist(blocks))
      if (length(all_treatments) == 0) stop("No treatments found in blocks")
      m <- nchar(all_treatments[1])
      design_data <- data.frame()
      for (b in seq_along(blocks)) {
        block <- blocks[[b]]
        for (t in block) {
          factors <- decode_treatment(t)
          if (length(factors) != m) stop("Inconsistent treatment label length")
          row <- data.frame(Block = paste0("B", b), t(factors))
          design_data <- rbind(design_data, row)
        }
      }
      colnames(design_data)[-1] <- paste0("F", 1:m)

      if (verbose) cat("\n---------------- BALANCE CHECK ----------------\n")
      balance_ok <- TRUE
      for (f in paste0("F", 1:m)) {
        tbl <- table(design_data[[f]])
        if (verbose) {
          cat("\nFactor:", f, "\n")
          print(tbl)
        }
        if (length(unique(as.vector(tbl))) == 1) {
          if (verbose) cat("Balanced for", f, "\n")
        } else {
          if (verbose) cat("NOT balanced for", f, "\n")
          balance_ok <- FALSE
        }
      }

      coded_data <- design_data
      for (i in 1:m) {
        s <- levels[i]
        code_map <- code_levels(s)
        map_df <- data.frame(orig = 0:(s - 1), coded = code_map)
        coded_data[[paste0("F", i)]] <- sapply(design_data[[paste0("F", i)]],
                                               function(x) map_df$coded[map_df$orig == x])
      }

      form_str <- paste("~ (", paste(colnames(coded_data)[-1], collapse = " + "), ")^", m, sep = "")
      form <- as.formula(form_str)
      X <- tryCatch({
        mm <- model.matrix(form, data = coded_data[,-1])
        if ("(Intercept)" %in% colnames(mm)) mm <- mm[, setdiff(colnames(mm), "(Intercept)"), drop = FALSE]
        mm
      }, error = function(e) {
        if (verbose) cat("\n Model matrix could not be fully generated due to collinearity.\n")
        return(NULL)
      })

      if (is.null(X)) {
        if (verbose) cat("Cannot proceed with OFS check -insufficient unique treatments.\n")
        return(invisible(list(ok = FALSE, reason = "insufficient_unique_treatments")))
      }

      blk_levels <- unique(as.character(coded_data$Block))
      blk_factor <- factor(as.character(coded_data$Block), levels = blk_levels)
      B <- matrix(0, nrow = nrow(coded_data), ncol = length(blk_levels))
      for (i in seq_len(nrow(coded_data))) {
        B[i, as.integer(blk_factor[i])] <- 1
      }
      colnames(B) <- paste0("Block", seq_len(ncol(B)))
      rownames(B) <- rownames(coded_data)

      Q <- diag(nrow(X)) - B %*% solve(t(B) %*% B) %*% t(B)
      X_adj <- Q %*% X

      Cmat <- t(X_adj) %*% X_adj + diag(1e-12, ncol(X_adj))

      safe_cov2cor <- function(M) {
        d <- sqrt(diag(M))
        d[d == 0 | is.na(d)] <- 1
        corM <- M / (d %o% d)
        corM[is.na(corM)] <- 0
        return(corM)
      }

      corC <- safe_cov2cor(Cmat)

      effect_names <- colnames(X)
      effect_order <- sapply(strsplit(effect_names, ":"), length)

      effect_group_report <- list()
      for (ord in sort(unique(effect_order))) {
        idx <- which(effect_order == ord)
        subC <- corC[idx, idx, drop = FALSE]

        if (length(idx) <= 1) {
          if (verbose) cat(paste0("\n", ord, "-FACTOR INTERACTIONS: Only one effect, skipping correlation check.\n"))
          effect_group_report[[as.character(ord)]] <- list(mean_offdiag = 0, orthogonal = TRUE)
          next
        }

        offdiag_vals <- abs(subC[upper.tri(subC)])
        offdiag_mean <- ifelse(length(offdiag_vals) == 0, 0, mean(offdiag_vals, na.rm = TRUE))

        label <- switch(as.character(ord),
                        "1" = "MAIN EFFECTS",
                        "2" = "TWO-FACTOR INTERACTIONS",
                        paste0(ord, "-FACTOR INTERACTIONS"))

        if (verbose) {
          cat("\n", label, ":\n", sep = "")
          cat("Average off-diagonal correlation =", round(offdiag_mean, 3), "\n")
          if (!is.na(offdiag_mean) && offdiag_mean < tol) {
            cat("Orthogonal among themselves\n")
          } else {
            cat("Some correlation exists among same-order effects\n")
          }
        }
        effect_group_report[[as.character(ord)]] <- list(mean_offdiag = offdiag_mean,
                                                         orthogonal = (!is.na(offdiag_mean) && offdiag_mean < tol))
      }

      offdiag_all <- mean(abs(corC[upper.tri(corC)]), na.rm = TRUE)
      if (is.na(offdiag_all)) offdiag_all <- 0

      if (verbose) {
        cat("\n=========================================================\n")
        if (offdiag_all < tol) {
          cat("Overall Design has Orthogonal Factorial Structure (OFS)\n")
        } else {
          cat("Design deviates from perfect OFS (mean off-diagonal =", round(offdiag_all, 3), ")\n")
        }
        cat("=========================================================\n")
      }

      invisible(list(ok = (offdiag_all < tol),
                     offdiag_all = offdiag_all,
                     corC = corC,
                     balance_ok = balance_ok,
                     effect_report = effect_group_report,
                     design_data = design_data))
    }

    compute_divisors <- function(levels) {
      v <- prod(levels)
      sprod <- v
      sapply(seq_along(levels), function(i) {
        si <- levels[i]
        (sprod / si) * choose(si, 2)
      })
    }

    compute_f_sequences <- function(levels) {
      D <- compute_divisors(levels)
      seqs <- vector("list", length(levels))
      for (i in seq_along(levels)) {
        si <- levels[i]
        j <- 0:(si - 1)
        seqs[[i]] <- (si - (2 * j + 1)) / D[i]
      }
      seqs
    }

    build_L_mains <- function(levels, grids, f_seqs) {
      v <- nrow(grids)
      m <- length(levels)
      L_list <- vector("list", m)
      for (i in 1:m) {
        si <- levels[i]
        if (si < 2) {
          warning(sprintf("Factor %d has only one level; skipping contrast generation.", i))
          L_list[[i]] <- rep(0, v)
          next
        }
        vals <- sapply(seq_len(v), function(t) {
          lvl <- grids[t, i]
          f_seqs[[i]][lvl + 1]
        })
        L_list[[i]] <- vals
      }
      L_list
    }

    build_L_interaction <- function(L_list, idxs) {
      if (any(sapply(idxs, function(ii) all(abs(L_list[[ii]]) < 1e-12)))) {
        return(rep(0, length(L_list[[1]])))
      }
      v <- length(L_list[[1]])
      L <- numeric(v)
      for (t in 1:v) {
        vals <- sapply(idxs, function(ii) L_list[[ii]][t])
        if (any(abs(vals) < 1e-12)) {
          L[t] <- 0
        } else {
          sgn <- prod(sign(vals))
          mag <- min(abs(vals))
          L[t] <- sgn * mag
        }
      }
      L
    }

    compute_efficiency_from_L <- function(L, C, r_scalar) {
      num <- as.numeric(L %*% C %*% L)
      den <- as.numeric(r_scalar * (L %*% L))
      if (abs(den) < 1e-12) return(NA_real_)
      num / den
    }

    analyze_design_efficiency <- function(levels, blocks, verbose = TRUE, tol = 1e-6) {
      built <- build_incidence(levels, blocks)
      N <- built$N
      grids <- built$grids
      v <- nrow(grids); m <- length(levels)
      comp <- compute_Cmatrix(N)
      C <- comp$C
      r_vec <- comp$r
      k_vec <- comp$k

      if (verbose) {
        cat("\nBlock sizes (k):", paste(round(k_vec, 6), collapse = ", "), "\n")
        cat("Replications (r) [per treatment]:", paste(round(r_vec, 6), collapse = ", "), "\n\n")
      }

      v_ofs <- check_design_balance_OFS(levels, blocks, tol = tol, verbose = verbose)
      if (!is.null(v_ofs) && isTRUE(v_ofs$ok)) {
        if (verbose) cat("Design verified: OFS OK using standard p^(y) method\n\n")
        y_tuples <- generate_y_tuples(m)
        labels <- generate_labels_from_tuples(y_tuples)
        results <- data.frame(label = character(0), rho = numeric(0), efficiency = numeric(0), stringsAsFactors = FALSE)
        for (i in seq_len(nrow(y_tuples))) {
          y <- y_tuples[i, ]
          P_y <- compute_p_y(y, levels)
          trace_val <- sum(diag(P_y %*% C %*% t(P_y)))
          rank_val <- nrow(P_y)
          rho_y <- trace_val / rank_val
          eff_y <- rho_y / r_vec[1]
          results <- rbind(results, data.frame(label = labels[i], rho = round(rho_y,6), efficiency = round(eff_y,6), stringsAsFactors = FALSE))
          if (verbose) cat(sprintf("Efficiency(%s) = %g\n", labels[i], eff_y))
        }
        return(invisible(list(mode = "OFS", results = results)))
      }

      if (!is.null(v_ofs)) {
        if (verbose) cat("Design does NOT satisfy OFS (mean off-diagonal =",
                         ifelse(is.null(v_ofs$offdiag_all), "NA", format(v_ofs$offdiag_all, digits = 6)),
                         "). Attempting general method if design is proper & equireplicate...\n\n")
      } else {
        if (verbose) cat("OFS check returned NULL then Attempting general method if design is proper & equireplicate...\n\n")
      }

      pr <- check_proper(N)
      er <- check_equireplicate(N)
      if (!pr$proper || !er$equireplicate) {
        if (verbose) {
          cat("Design is not proper and/or not equireplicate:\n")
          cat("  proper:", pr$proper, "  equireplicate:", er$equireplicate, "\n")
          cat("Efficiency cannot be computed via general method\n")
        }
        return(invisible(list(mode = "NO_OFS", reason = "not_proper_or_not_equirep")))
      }

      if (verbose) cat("Design is proper and equireplicate. Proceeding with general method of computation.\n\n")
      r_scalar <- er$replicates[1]
      D <- compute_divisors(levels)
      names(D) <- paste0("F", seq_len(m))
      f_seqs <- compute_f_sequences(levels)
      L_mains <- build_L_mains(levels, grids, f_seqs)
      names(L_mains) <- paste0("F", seq_len(m))

      tuples <- generate_y_tuples(m)
      labels <- generate_labels_from_tuples(tuples)
      results <- data.frame(label = character(0), rho = numeric(0), efficiency = numeric(0), stringsAsFactors = FALSE)

      for (ti in seq_len(nrow(tuples))) {
        y <- tuples[ti, ]
        lbl <- labels[ti]
        idxs <- which(y == 1)
        if (length(idxs) == 1) {
          L_vec <- L_mains[[ idxs ]]
        } else {
          L_vec <- build_L_interaction(L_mains, idxs)
        }
        eff_val <- compute_efficiency_from_L(L_vec, C, r_scalar)
        rho_val <- as.numeric(L_vec %*% C %*% L_vec)
        results <- rbind(results, data.frame(label = lbl, rho = round(rho_val,6), efficiency = round(eff_val,6), stringsAsFactors = FALSE))
        if (verbose) cat(sprintf("Efficiency(%s) = %g\n", lbl, eff_val))
      }

      invisible(list( results = results, N = N, C = C, L_mains = L_mains))
    }

    res <- analyze_design_efficiency(levels, blocks_norm, verbose = verbose)
    invisible(res)
  }

  res1 <- generate_design_report(levels_vec, block_size,
                                 method  = method,
                                 verbose = verbose)

  blocks_from_code1 <- res1$blocks
  factor_levels     <- levels_vec

  blocks_for_code2 <- lapply(blocks_from_code1, function(df) {
    apply(df[, grep("^F\\d+$", colnames(df)), drop = FALSE],
          1,
          function(x) paste0(x, collapse=""))
  })

  res2 <- analyze_design(factor_levels, blocks_for_code2, verbose = verbose)

  invisible(list(
    code1          = res1,
    code2          = res2,
    factor_levels  = factor_levels,
    block_size     = block_size,
    blocks_numeric = blocks_from_code1,
    blocks_labels  = blocks_for_code2
  ))
}
