#' @name PredictSurvFit
#' @export
predict_SurvFitCstExp <- function(fit,
                              display.exposure = NULL,
                              hb_value = NULL,
                              interpolate_length = NULL, ...){
    df <- display.exposure
    # EXTRACT FROM FIT
    df.param <- extract_param(fit)
    # RUN PREDICTION
    if (fit$model_type == "SD") {
        df_mcmc <- predict_cstSD(display.exposure = df,
                              display.parameters = df.param,
                              hb_value = hb_value,
                              interpolate_length = interpolate_length)
    }
    if (fit$model_type == "IT") {
        df_mcmc <- predict_cstIT(display.exposure = df,
                              display.parameters = df.param,
                              hb_value = hb_value,
                              interpolate_length = interpolate_length)
    }
   
    Psurv.col.position = grep("Psurv", colnames(df_mcmc))
    if (length(Psurv.col.position) == 1) {
        df_quantile = df_mcmc[, c("conc", "time", "replicate")]
        df_quantile$q50 = df_mcmc$Psurv_1
        df_quantile$qinf95 = df_quantile$q50 
        df_quantile$qsup95 = df_quantile$q50 
    } else{
        df_Psurv = df_mcmc[, Psurv.col.position]
        dfquant <- t(apply(df_Psurv, 1, quantile, probs = c(0.5,0.025,0.975), names = TRUE, na.rm = TRUE))
        
        df_quantile = df_mcmc[,grep("Psurv",colnames(df_mcmc), invert = TRUE)]
        df_quantile$qinf95 = dfquant[, "2.5%"]
        df_quantile$q50 = dfquant[, "50%"]
        df_quantile$qsup95 = dfquant[, "97.5%"]
    }
    
    return_object <- list(df_quantile = df_quantile,
                          df_mcmc = df_mcmc)
    class(return_object) <- append("SurvPredict", class(return_object))
    return(return_object)
}


#' @name PredictSurvFit
#' @export
predict_cstSD <- function(display.exposure = NULL,
                       display.parameters = NULL,
                       hb_value = NULL,
                       interpolate_length = NULL){
    df <- display.exposure
    kd <- display.parameters$kd
    hb <- display.parameters$hb
    if (!is.null(hb_value)) {
        hb <- rep(hb_value, length(display.parameters$hb))
    }
    z <- display.parameters$z
    kk <- display.parameters$kk
    
    ls_df <- lapply(unique(df$replicate), function(r){
        df[df$replicate == r, ]
    })
    ls <- lapply(ls_df, function(d) {
        dout = SurvSD_cst(Cw = unique(d$conc),
                          time = d$time,
                          kd = kd,
                          hb = hb,
                          z = z,
                          kk = kk,
                          interpolate_length = interpolate_length)
        dout$replicate = unique(d$replicate)
        return(dout)
    })
    df_mcmc <- do.call("rbind", c(ls, make.row.names = FALSE))
    return(df_mcmc)
}

#' @name PredictSurvFit
#' @export
predict_cstIT <- function(display.exposure = NULL,
                       display.parameters = NULL,
                       hb_value = NULL,
                       interpolate_length = NULL){
    df <- display.exposure
    kd <- display.parameters$kd
    hb <- display.parameters$hb
    if (!is.null(hb_value)) {
        hb <- rep(hb_value, length(display.parameters$hb))
    }
    alpha <- display.parameters$alpha
    beta <- display.parameters$beta
    
    ls_df <- lapply(unique(df$replicate), function(r){
        df[df$replicate == r, ]
    })
    ls <- lapply(ls_df, function(d) {
        dout = SurvIT_cst(Cw = unique(d$conc),
                          time = d$time,
                          kd = kd,
                          hb = hb,
                          alpha = alpha,
                          beta = beta,
                          interpolate_length = interpolate_length)
        dout$replicate = unique(d$replicate)
        return(dout)
    })
    df_mcmc <- do.call("rbind", c(ls, make.row.names = FALSE))
    return(df_mcmc)
}


#' @title Internal predict function
#' 
#' @description
#' Survival function for "SD" model with external concentration changing
#' with time
#'
#' @param Cw A vector of external concentration
#' @param time A vector of time
#' @param kd a vector of parameter
#' @param hb a vector of parameter
#' @param z a vector of parameter
#' @param kk a vector of parameter
#' @param interpolate_length can be used to provide a sequence from 0 to maximum of
#' the time of exposure in original dataset (used for fitting).
#' 
#' @return A data.frame with exposure columns \code{time} and \code{conc} and
#' the resulting probabilisty of survival in \code{Psurv_XX} column where 
#' \code{XX} refer to an MCMC iteration
SurvSD_cst <- function(Cw, time, kd, hb, z, kk,
                       interpolate_length = NULL){
    if (!is.null(interpolate_length)) {
        time_seq <- seq(min(time), max(time), length.out = interpolate_length)
        time <- sort(unique(c(time, time_seq)))
    }
    S <- gutsredsd_cst(Cw, time, kd, hb, z, kk)
    df_Psurv <- as.data.frame(S)
    colnames(df_Psurv) = paste("Psurv", 1:ncol(df_Psurv), sep = "_")
    df_Psurv$conc = Cw
    df_Psurv$time = time
    return(df_Psurv)
}

gutsredsd_cst <- function(Cw, time, kd, hb, z, kk){
    S <- sapply(time, function(t) { 
        S_hb <- exp(-hb*t)
        r <- ifelse(Cw > z, z/Cw, 0)
        tz <- -(1/kd)*log(1 - r)
        y <- exp( kk/kd*Cw*(exp(-kd*tz) - exp(-kd*t)) - kk*(Cw - z)*(t - tz))
        S_t <- S_hb * ifelse(Cw > z & t > tz, y, 1)
        return(S_t)
    })
    if (is.matrix(S)) {
        S <- t(S)
    } else{
        S <- as.matrix(S)
    }
    return(S)
}


#' @title Internal predict function
#' 
#' @description
#' Survival function for "IT" model with external concentration changing
#' with time
#'
#' @param Cw A vector of external concentration
#' @param time A vector of time
#' @param kd a vector of parameter
#' @param hb a vector of parameter
#' @param alpha a vector of parameter
#' @param beta a vector of parameter
#' @param interpolate_length can be used to provide a sequence from 0 to maximum of
#' the time of exposure in original dataset (used for fitting).
#' 
#' @return A data.frame with exposure columns \code{time} and \code{conc} and
#' the resulting probabilisty of survival in \code{Psurv_XX} column where 
#' \code{XX} refer to an MCMC iteration
SurvIT_cst <- function(Cw, time, kd, hb, alpha, beta,
                       interpolate_length = NULL){
    if (!is.null(interpolate_length)) {
        time_seq <- seq(min(time), max(time), length.out = interpolate_length)
        time <- sort(unique(c(time, time_seq)))
    }
    S <- gutsredit_cst(Cw, time, kd, hb, alpha, beta)
    df_Psurv <- as.data.frame(S)
    colnames(df_Psurv) = paste("Psurv", 1:ncol(df_Psurv), sep = "_")
    df_Psurv$conc = Cw
    df_Psurv$time = time
    return(df_Psurv)
}

gutsredit_cst <- function(Cw, time, kd, hb, alpha, beta){
    D <- Cw*(1 - exp(-kd %*% t(time)))
    Dmax <- apply(D, 1, cummax)
    D.max <- ifelse(is.matrix(Dmax), t(Dmax), as.matrix(Dmax))
    S <- exp(-hb %*% t(time)) * 
        (1 - stats::plogis(log(D.max), location = log(alpha), scale = 1/beta))
    return(t(S))
}

