% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/export_json.R
\name{export_json}
\alias{export_json}
\alias{export_results_json}
\alias{export_results_dist_json}
\alias{export_code_json}
\alias{export_data_json}
\title{Exporting results from a multiverse analysis to JSON}
\usage{
export_results_json(x, term, mean, sd, dist, filename)

export_results_dist_json(x, term, dist, filename)

export_code_json(x, filename)

export_data_json(x, filename)
}
\arguments{
\item{x}{a tidy tibble or data frame which contains summary statistics or distributional information 
of each regression coefficient parameter}

\item{term}{column in the data frame, x, which contains the name of the coefficients}

\item{mean}{column in the data frame, x, which contains the mean estimate for each coefficient}

\item{sd}{column in the data frame, x, which contains the standard error estimate for each coefficient}

\item{dist}{column in the data frame, x, which contains vectorised distributions---an object of class 
`distribution` for each coefficient}

\item{filename}{filename on disk (as a character string)}
}
\value{
a JSON file or (if a filepath is not specified) a dataframe for the results file and a list for the code file
}
\description{
Exports the results of the multiverse analysis to JSON in a format which is compatible with the multiverse visualisation tool
}
\details{
## results JSON file schema
It consists of a list of objects (where each object corresponds to one analysis in the multiverse). 
Within this object, the results attribute contains a(nother) list of objects corresponding to each outcome variable. 
For e.g., here we have four coefficients (see the results of the regression model), and thus the results attribute will contain four objects. 
Each object has the following attributes:
- `term`: name of the outcome variable
- `estimate`: mean / median point estimate i.e., \eqn{\mathbb{E}(\mu)} for any parameter \eqn{\mu}.
- `std.error`: standard error for the point estimate i.e., \eqn{\sqrt{\text{var}(\mu)}}
- `cdf.x`: a list of quantiles
- `cdf.y`: a list of cumulative probability density estimates corresponding to the quantiles

In addition, it also contains the following attributes, but these are not currently used by Milliways:
- `statistic`
- `p.value`
- `conf.low`
- `conf.high`


## code JSON file schema
It consists of two attributes: `code` and `parameters`. 
`code` is a list of strings consisting of the R and multiverse syntax used to implement the analysis. For readability, we
use [styler] to break up the declared code.
`parameters` is an object listing the parameter names and the corresponding options for each of the parameters declared in the analysis.

## data JSON file schema
It consists of a list of objects, each with two attributes: `field` and `values`. 
`field` is the name of a column corresponding to a variable in the dataset. 
`values` are a list of values for that variable in the dataset.
}
\examples{
\donttest{
library(dplyr)
library(tidyr)

M = multiverse()

inside(M, {
  df = tibble(
    x = rnorm(100),
    y = x * 0.5 + rnorm(100, 0, 2)
 )
 
 # alternatives to remove outlier
 df.filtered = df \%>\%
   filter(
     branch(outlier_exclusion,
        "2SD" ~ abs(y - mean(y)) > 2*sd(y),
        "3SD" ~ abs(y - mean(y)) > 3*sd(y)
     )
 )
  
  fit = lm(y ~ x, data = df)
  res = broom::tidy(fit)
})

execute_multiverse(M)

multiverse::expand(M) \%>\%
  extract_variables(res) \%>\%
  unnest(res) \%>\%
  export_results_json(term, estimate, std.error) 
} 


}
