\name{mvsusy}
\alias{mvsusy}
\alias{mvSUSY}
\title{ Multivariate Surrogate Synchrony }
\description{
  Multivariate Surrogate Synchrony (mvSUSY) estimates the synchrony within datasets that contain more than two time series. mvSUSY was developed from Surrogate Synchrony (SUSY) with respect to implementing surrogate controls, and extends synchrony estimation to multivariate data.
}
\usage{
mvsusy(x, segment, Hz, method=c("lambda_max","omega"), max_pseudo=1000, seed=1)
}
\arguments{
  \item{x}{ A data.frame of numeric columns. }
  \item{segment}{ Integer, size in seconds. Must not be be larger than half the time series (\code{nrow(x)/2}). }
  \item{Hz}{ Integer, measures per second (sampling rate). }
  \item{method}{ Character, either \code{"lambda_max"} or \code{"omega"}. }
  \item{max_pseudo}{ Numeric, maximum number of surrogate ("psuedo") data, default 1000. }
  \item{seed}{ Numeric, same seed generates the same random surrogate data. }
}
\details{
  Data are entered as a file where multiple time series are in columns. First row with column names. For example, if the data represent the interaction of a group of five people with one time series from each group member, the file has five columns, and its first row contains the member names. The time series are divided in segments of appropriate length (parameter \code{segment}). Synchrony computation is done separately in each segment, then all segment synchronies are aggregated to yield the "real synchrony" of the multiple time series. Segments are non-overlapping, and the number of segments that fit into the time series may have a remainder (usually a few seconds at the end of the time series), which is not considered. Rows having missing values are removed.
  "Real synchrony" is controlled for spurious synchrony and non-stationarity using surrogate analysis. Surrogate ("pseudo") data are generated by random segment-shuffling of each column separately. From \emph{m} columns with \emph{s} segments, \code{s! / (s - m)!} surrogates can be generated. This often very high number of surrogates should be curbed by setting \code{max_pseudo} appropriately. For each of the \code{max_pseudo} surrogate datasets, (pseudo) synchrony is computed in the same way as real synchrony. The mean and standard deviation of surrogate synchronies are provided in the output as \emph{mean(synchrony-pseudo)} and \emph{sd(synchrony-pseudo)}.
  The \emph{effect size (ES)} of synchrony is computed by the difference between \emph{mean(synchrony-real)} and \emph{mean(synchrony-pseudo)} standardized by \emph{sd(synchrony-pseudo)}. Tests against the null-hypothesis \code{mean(synchrony-real) = mean(synchrony-pseudo)} are performed by a \emph{t-statistic} and a Wilcoxon test (\emph{statistic-nonpar}). Two methods are available to assess mvSUSY: \emph{"lambda_max"} and \emph{"omega"}. \emph{lambda_max} is computed by the eigendecomposition of the correlation matrix. The correlation matrix of the \emph{m} columns (time series) can be described by \emph{m} eigenvalues \emph{lambda}, the largest of which provides an assessment of multivariate synchrony, i.e. the coupling between the time series (columns) of the data. \emph{lambda} is computed in each segment, then aggregated across all segments. \emph{omega} is a measure of multivariate synchrony that makes use of the actually measured degree of entropy, a measure of disorder of a dataset, with its equivalent to \emph{Shannon information}. Landsberg suggested to normalize entropy \emph{S} by the potential entropy \emph{Spot} possible in a system, providing the measure of \emph{omega} ("Landsberg order") as \code{omega = 1 - S / Spot}. The entropies can be computed based on the variance-covariance matrix of the multiple time series (Shiner, Davison & Landsberg, 1999). Again, \emph{omega} is computed in each segment then aggregated. Tschacher, Scheier & Grawe (1998) applied these methods in psychotherapy research.
}
\value{
  Object of class \code{mvsusy} is returned.
}
\seealso{\code{\link{plot.mvsusy}}, \code{\link{as.data.frame.mvsusy}}, \code{\link{print.mvsusy}}}
\references{
  Meier D & Tschacher W (2021). \href{https://www.mdpi.com/1099-4300/23/11/1385}{"Beyond Dyadic Coupling: The Method of Multivariate Surrogate Synchrony (mv-SUSY)"}. Entropy, 23, 1385.\cr
  Shiner JS, Davison M, & Landsberg PT (1999). On measures for order and its relation to complexity. In Tschacher W & Dauwalder J-P (Eds). Dynamics, Synergetics, Autonomous Agents. Singapore: World Scientific:, pp. 49-63.\cr
  Tschacher W, Scheier C, & Grawe K (1998). Order and Pattern Formation in Psychotherapy. Nonlinear Dyn. Psychol. Life Sci., 2, 195-215.
}
\examples{
\donttest{
set.seed(1)
data = as.data.frame(replicate(5, sample(10, 5000, TRUE)))

## compute mvSUSY (lambda_max method)
res = mvsusy(data, segment=10, Hz=10)
res
plot(res, type="eigenvalue")

## omega method
res = mvsusy(data, segment=10, Hz=10, method="omega")
res
plot(res, type="density")

## export to flat file via data.frame and write.csv
df = as.data.frame(res)
df
}
}
\keyword{ data }
