% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/global.R
\name{trtSelThresh}
\alias{trtSelThresh}
\title{Estimation of the optimal threshold of a treatment selection marker}
\usage{
trtSelThresh(EvtRefDist = NULL, NoEvtRefDist = NULL,
  EvtInnovDist = NULL, NoEvtInnovDist = NULL, mRiskRef = NULL,
  mRiskInnov = NULL, lowRef = TRUE, toxRef = TRUE, r = 0,
  le.MCMC = 1000, hessTol = 10^(-6), plot = FALSE,
  progress.bar = NULL, seed = NULL)
}
\arguments{
\item{EvtRefDist}{an object of class allowedFitDist that summarizes the distribution fitted to the marker values of patients that developed the event of interest in the reference arm. This class of objects is obtained using the \code{\link[optimalThreshold]{fit}} function.}

\item{NoEvtRefDist}{an object of class allowedFitDist that summarizes the distribution fitted to the marker values of patients that did not develop the event of interest in the reference arm. This class of objects is obtained using the \code{\link[optimalThreshold]{fit}} function.}

\item{EvtInnovDist}{an object of class allowedFitDist that summarizes the distribution fitted to the marker values of patients that developed the event of interest in the innovative arm. This class of objects is obtained using the \code{\link[optimalThreshold]{fit}} function.}

\item{NoEvtInnovDist}{an object of class allowedFitDist that summarizes the distribution fitted to the marker values of patients that did not develop the event of interest in the innovative arm. This class of objects is obtained using the \code{\link[optimalThreshold]{fit}} function.}

\item{mRiskRef}{an object of class mcmc.list provided by the user. It must be a sample of the posterior distribution of the mean risk of event in the reference treatment arm. If NULL, the function samples values in the posterior distribution of the mean risk of event in the reference arm assuming Jeffrey's prior (Beta(0.5,0.5)), and estimating the mean risk using the number of marker values specified in each treatment arm.}

\item{mRiskInnov}{an object of class mcmc.list provided by the user. It must be a sample of the posterior distribution of the mean risk of event in the innovative treatment arm. If NULL, the function samples values in the posterior distribution of the mean risk of event in the innovative arm assuming Jeffrey's prior (Beta(0.5,0.5)), and estimating the mean risk using the number of marker values specified in each treatment arm..}

\item{lowRef}{a logical value indicating whether low values of the marker are associated with low (TRUE) or high (FALSE) risk under the reference treatment arm.}

\item{toxRef}{a logical value indicating whether the reference treatment arm (TRUE) or the innovative treatment arm (FALSE) must be preferred at equal efficacy taking into account toxicity.}

\item{r}{a numeric value indicating the cost ratio between the most harmful treatment and the event (see Details).}

\item{le.MCMC}{length of the desired MCMC chain.}

\item{hessTol}{tolerance for the hessian value of the utility function at the optimal threshold.}

\item{plot}{a logical value indicating whether routine graphics must be produced.}

\item{progress.bar}{a character string indicating whether the user wishes to print a progress bar during the function process.}

\item{seed}{a numerical value used to fix the random seed.}
}
\value{
Returns an object of class \code{trtSelOptThresh}.
}
\description{
This function produces a sample of the posterior distribution of the optimal threshold of a treatment selection marker. The optimal threshold is defined as the marker value that maximized the utility of using the marker to decide between two treatment options (innovative and reference one). The utility function takes into account the efficacy of the treatment options as well as treatment induced toxicities.
The estimation of the utility function needs data from a clinical trial about the two treatment options, in which the success of a treatment is defined by the absence of an event of interest in a given post-treatment interval (binary data). For the time being, the package cannot estimate the optimal threshold in case of censored data about the occurrence of the event in the given post-treatment interval.
To calculate the utility function, the user needs to specify: 
\itemize{
\item the distribution of the marker in the four groups defined by the treatment option and the outcome; in fact only three distributions need to be specified, the fourth one being derived from the three others and the mean risks of event in the two treatment arms through the randomization constraint (the distribution of the marker being the same in both treatment arms; see the \code{\link[optimalThreshold]{fit}} function for more details),
\item and the mean risks of the event in the two treatment arms. The user must also specify: the cost of the innovative treatment relative to the cost of the event (see Details). 
}
The optimal threshold and its credible interval are calculated using a Monte Carlo approach.
}
\details{
When \code{toxRef==FALSE} then Janes et al. (2014) defined the costs of event and treatment as:
\tabular{lcc}{
\tab Y=0 \tab Y=1 \cr
Z=0 \tab 0 \tab \eqn{C_Y} \cr
Z=1 \tab \eqn{C_Z} \tab \eqn{C_Z+C_Y} 
}
When \code{toxRef==TRUE} it is defined as :
\tabular{lcc}{
\tab Y=0 \tab Y=1 \cr
Z=0 \tab \eqn{C_Z} \tab \eqn{C_Z+C_Y} \cr
Z=1 \tab 0 \tab \eqn{C_Y} 
}
According to the value of \code{toxRef}, the r ratio is simply \eqn{r=C_Z/C_Y}. The r ratio can also be indirectly specified by the absolute difference in risk of event between the two treatments above which a physician would recommend the use of the most harmful treatment. The inverse of the r ratio can also be interpreted as the number of patients for whom the physician is ready to give the most harmful treatment to prevent one additional case compared with the less harmful treatment.
}
\section{References }{

Blangero, Y, Rabilloud, M, Ecochard, R, and Subtil, F. A Bayesian method to estimate the optimal threshold of a marker used to select patients' treatment. \emph{Statistical Methods in Medical Research}. 2019.
}

\examples{
#Simulating data from four gaussian distributions, 
#with mean risks equal to 0.5 in each arm:
x0E <- rnorm(250) # reference arm, event
x0Eb <- rnorm(250, 2) # reference arm, no event
x1E <- rnorm(250, 2) # innovative arm, event
x1Eb <- rnorm(250) # innovative arm, no event

#When working with real data. You can check the randomization constraint using the 
#densCurves function:
densCurves(x0 = c(x0E, x0Eb), x1 = c(x1E, x1Eb), type = "treatment selection")

#You can also use the riskCurves function to know if low values of the marker are associated
#with a better response under the reference treatment or not:
library(mgcv)
riskCurves(x0E, x0Eb, x1E, x1Eb)

#Fit normal distributions on three groups. And let the last one (1E) be undefined (derived 
#indirectly using the randomization constraint):
fit0E <- fit(x0E, "norm")
fit0Eb <- fit(x0Eb, "norm")
fit1E <- fit(x1E, "undefined")
fit1Eb <- fit(x1Eb, "norm")

#Apply the main function to estimate the optimal threshold:
# first case: the mean risks of event in the two treatment arms are left unspecified (are 
# determined by the number of marker measurements in the fit0E, fi0Eb, fit1E, fit1Eb) 
\donttest{
res <- trtSelThresh(fit0E, fit0Eb, fit1E, fit1Eb, 
                    lowRef = FALSE, toxRef = FALSE, r = 0.02, le.MCMC = 5000, plot = TRUE, 
                    progress.bar = "text")

# second case: the mean risks of event in the two treatment arms are given through mcmc.lists 
# that correspond to their posterior distributions (see the fit man page for examples on how
# to generate posterior distributions manually)

#You can summarize the results using the summary() function:
summary(res, method = "median")

#You can extract the estimates and CI bounds of each indicator presented in the summary:
estimates(res, method = "median")
credibleIntervals(res)

#Plot the decision curves (this function is time-consuming):
dCres <- decisionCurve(res, r = seq(0, 0.2, length.out = 6))

#You can plot again the decision curves by applying the plot method to dCres, 
#this function is a lot faster than the previous one. It also has more options
#to customize the plot:
plot(dCres)
plot(dCres, which = 1)
plot(dCres, which = 2)
}
}
