#' Fits Agresti's agreement model that includes kappa as a parameter.
#'
#' Agresti, A. (1989).  An agreement model with kappa as a parameter.
#' Statistics and Probability Letters, 7, 271-273.
#' @param n matrix of observed counts
#' @param verbose should cycle-by-cycle info be printed as messages? The default
#' is FALSE.
#' @returns a list containing
#'    kappa: value of kappa coefficient
#'    pi_margin: value of marginal p-values.  They apply
#'    to rows and columns
#'    chisq: Pearson X^2
#'    df: degrees of freedom
#'    expected: fitted frequencies
#' @export
Agresti_kappa_agreement <- function(n, verbose=FALSE) {
  r <- nrow(n)
  N <- sum(n)
  p <- n / N
  pi_margin <- Agresti_starting_values(p)
  if (verbose) {
    message("iter,   logL,        kappa")
  }
  for (iter in 1:5) {
    kappa <- Agresti_bisection(p, pi_margin, 0, 1)
    lambda <- Agresti_equation_3(p, pi_margin, kappa)
    pi_margin <- Agresti_equation_2(p, pi_margin, lambda, kappa)
    pi <- Agresti_compute_pi(pi_margin, kappa)
    log_l <- log_likelihood(n, pi)
    if (verbose) {
      message(paste(iter, log_l, kappa))
    }
  }
  chisq = pearson_chisq(n, pi)
  g_squared <- likelihood_ratio_chisq(n, pi)
  m <- pi * N
  df <- r * r - length(pi_margin) - 1
  list(kappa=kappa, pi_margin=pi_margin, chisq=chisq,
       g_squared=g_squared, df=df, expected=m)
}


#' Computes staring values for marginal pi.
#'
#' @param p matrix of observed proportions
#' @returns vector containing pi
Agresti_starting_values <- function(p) {
  r <- nrow(p)
  rows <- rowSums(p)
  cols <- colSums(p)

  pi_margin <- vector("double", r)
  for (i in 1:r) {
    pi_margin[i] <- (rows[i] + cols[i]) / 2.0
  }
  pi_margin
}


#' Computes value of lambda parameter
#'
#' @param p matrix of observed proportions
#' @param pi matrix of model-supplied proportions
#' @returns value of the lambda parameter
Agresti_compute_lambda <- function(p, pi) {
  r <- nrow(p)
  rows <- rowSums(p)
  cols <- colSums(p)
  pi_margin <- rowSums(pi)
  diagonal <- sum(diag(p))
  marginal_sum <- 0.0
  diagonal_sum <- sum(diag(pi))
  for (i in 1:r) {
    margin_sum <- margin_sum + pi_margin[i] * (rows[i] + cols[i])
  }
  lambda <- (1.0 - kappa) * margin_sum + kappa * (2.0 - diagonal) / diagonal_sum
  lambda
}


#' Computes the matrix pi of model-based proportions
#'
#' @param pi_margin current value of (row and column) marginal proportion
#' @param kappa current estimate of kappa coefficient
#' @returns matrix of model-based proportions
Agresti_compute_pi <- function(pi_margin, kappa) {
  pi <- matrix(0.0, nrow=length(pi_margin), ncol=length(pi_margin))
  for (i in 1:length(pi_margin)) {
    pi[i, i] <- pi_margin[i]^2 + kappa * pi_margin[i] * (1.0 - pi_margin[i])
    for (j in 1:length(pi_margin)) {
      if (j == i) {
        next
      }
      pi[i, j] = pi_margin[i] * pi_margin[j] * (1.0 - kappa)
    }
  }
  pi
}


#' Function value for first equation in section 3.
#'
#' Used by Agresti_bisection()
#' @param p matrix of observed proportions
#' @param pi_margin current value of (row and column) marginal proportion
#' @param kappa current estimate of kappa coefficient
Agresti_f <- function(p, pi_margin, kappa) {
  r <- nrow(p)
  result <- 1.0 - sum(diag(p))
  ratio <- kappa / (1.0 - kappa)
  for (i in 1:r) {
    numer <- p[i, i] * (1.0 - pi_margin[i])
    result <- result - numer / (pi_margin[i] + ratio)
  }
  result
}


#' Solves equation Agresti_f() = 0 for delta by method of bisection..
#'
#' @param p matrix of observed proportions
#' @param pi_margin current value of (row and column) marginal proportion
#' @param x_low lower bound for search. Default value is 0.0
#' @param x_high upper bound for search. Default value is 1.0
#' @returns value of kappa that makes the function 0.0
Agresti_bisection <- function(p, pi_margin, x_low=0, x_high=1) {
  f_low <- Agresti_f(p, pi_margin, x_low)
  f_high <- Agresti_f(p, pi_margin, x_high)
  if (f_low * f_high > 0.0) {
    stop("value is not within bound")
  }
  for (iter in 1:10) {
    x_mid <- (x_low + x_high) / 2.0
    f_mid <- Agresti_f(p, pi_margin, x_mid)
    if (f_low * f_mid < 0) {
      x_high <- x_mid
      f_high <- f_mid
    } else {
      x_low <- x_mid
      f_low <- f_mid
    }
  }
  x_mid <- (x_low + x_high) / 2.0
  x_mid
}


#' First equation in section 3. Solved for kappa.
#'
#' @param p matrix of observed proportions
#' @param pi_margin current value of (row and column) marginal proportion
#' @param kappa current value of coefficient kappa
Agresti_equation_1 <- function(p, pi_margin, kappa) {
  r <- nrow(p)
  result <- 1.0 - sum(diag(p))
  for (iter in 1:8) {
    deriv <- 0.0
    ratio <- kappa / (1.0 - kappa)
    for (i in 1:r) {
      numer <- p[i, i] * (1.0 - pi_margin[i])
      result <- result - numer / (pi_margin[i] + ratio)
      denom <- (1.0 - kappa) * pi_margin[i] + kappa
      # deriv <- deriv + numer / denom^2
    }
    result
  }
  kappa
}


#' Second equation in section 3. Solved for pi_margin.
#'
#' @param p matrix of observed proportions
#' @param pi_margin current value of (row and column) marginal proportion
#' @param lambda value of quantity lambda defined in third equation
#' @param kappa current value of coefficient kappa
Agresti_equation_2 <- function(p, pi_margin, lambda, kappa) {
  r <- nrow(p)
  p_i_plus <- rowSums(p)
  p_plus_j <- colSums(p)
  for (i in 1:r) {
    pi_ii <- pi_margin[i]^2 + kappa * pi_margin[i] * (1.0 - pi_margin[i])
    pi_margin[i] <- (((p_i_plus[i] + p_plus_j[i]) * pi_ii)
        / (lambda * pi_ii  + kappa * p[i, i]))
  }
  pi_margin
}


#' Third equation in section 3. Solved for lambda
#'
#' @param p matrix of observed proportions
#' @param pi_margin current value of (row and column) marginal proportion
#' @param kappa current valye of coefficient kappa
Agresti_equation_3 <- function(p, pi_margin, kappa) {
  r <- nrow(p)
  p_i_plus <- rowSums(p)
  p_plus_j <- colSums(p)
  sum_diag <- sum(diag(p))
  sum_pi_ii <- 0.0
  sum_product <- 0.0
  for (i in 1:r) {
    pi_ii <- pi_margin[i]^2 + kappa * pi_margin[i] * (1.0 - pi_margin[i])
    sum_pi_ii <- sum_pi_ii + pi_ii
    sum_product <- sum_product + pi_margin[i] * (p_i_plus[i] + p_plus_j[i])
  }
  numer <- (1.0 - kappa) * sum_product + kappa * (2.0 - sum_diag)
  lambda <- numer / sum_pi_ii
  lambda
}
