# Utility routines for ordinalTables

#' Determines if its argument is not a valid number.
#'
#' @param x Numeric. Number of be evaluated
#' @returns TRUE if is.na(), is.nan(), or is.infinite() returns TRUE. FALSE otherwise.
is_missing_or_infinite <- function(x) {
  is.na(x) || is.nan(x)  || is.infinite(x)
}


#' Computes the log-odds (logit) for the value provided
#'
#' @param p Numeric. Assumed to lie in interval(0, 1)
#' @return log(p / (1.0 - p))
logit <- function(p) {
  log(p / (1.0 - p))
}


#' Computes the "expit" function -- inverse of logit.
#'
#' @param z Numeric. Real valued argument to expit() function.
#' @returns exp(z) / (1.0 + exp(z))
expit <- function(z) {
  exp(z) / (1.0 + exp(z))
}


#' Computes the constant of integration of a multinomial sample.
#'
#' N! / product(n[i]!)
#' @param n Matrix of observed counts
#' @param exclude_diagonal logical. Should the diagonal cells of a
#' square matrix be excluded from the computation. Default is FALSE,
#' @returns value of constant of integration for observed matrix provided
constant_of_integration <- function(n, exclude_diagonal=FALSE) {

  # log(N!)
  N <- sum(n)
  if (exclude_diagonal) {
    N <- N - sum(diag(n))
  }
  numer <- sum(log(1:N))

  # log(product n[i][j]!)
  r <- nrow(n)
  denom <- 0.0
  for (i in 1:r) {
    for (j in 1:ncol(n)) {
      if (exclude_diagonal && j == i) {
        next
      }
      if (0 < n[i, j]) {
        denom <- denom + sum(log(1:n[i,j]))
      }
    }
  }

  numer - denom
}


#' Computes the multinomial log(likelihood).
#'
#' @param n Matrix of observed counts
#' @param pi Matrix of same dimensions as n. Model-based matrix of predicted proportions
#' @param exclude_diagonal logical. Should diagonal cells of square matrix
#' be excluded from the computation? Default is FALSE.  The effect of setting
#' it to TRUE for non-square matrices may be unintuitive and should he avoided.
#' @returns log(likelihood)
#' @export
log_likelihood <- function(n, pi, exclude_diagonal=FALSE) {
  const <- constant_of_integration(n, exclude_diagonal)
  N <- sum(n)
  if (exclude_diagonal) {
    N <- N - sum(diag(n))
  }
  logL <- sum(n * log(pi))
  sum_m <- N * sum(pi)

  if (exclude_diagonal) {
    diag_n <- diag(n)
    diag_pi <- diag(pi)
    logL <- logL - sum(diag_n * log(diag_pi))
    sum_m <- sum_m - N * sum(diag_pi)
  }
  const + logL - sum_m
}


#' Computes the Pearson X^2 statistic.
#'
#' @param n Matrix of observed counts
#' @param pi Matrix with same dimensions as n. Model-based matrix of predicted proportions
#' @param exclude_diagonal logical. Should diagonal cells of square matrix
#' be excluded from the computation? Default is FALSE. The effect of setting
#' it to TRUE for non-square matrices may be unintuitive and should he avoided.
#' @returns X^2
#' @export
pearson_chisq <- function(n, pi, exclude_diagonal=FALSE) {
  N <- sum(n)
  if (exclude_diagonal) {
    N <- N - sum(diag(n))
  }
  chisq <- 0.0
  for (i in 1:nrow(pi)) {
    for (j in 1:ncol(pi)) {
      if (exclude_diagonal && j == i) {
        next
      }
      # expect_true(pi[i, j] > 0.0, info="encountered non-positive pi")
      chisq <- chisq + (n[i, j] - N * pi[i, j])^2 / (N * pi[i, j])
    }
  }
  chisq
}


#' Computes the likelihood ratio G^2 measure of fit.
#'
#' @param n Matrix of observed counts
#' @param pi Matrix of same dimensions as n. Model-based matrix of predicted proportions
#' @param exclude_diagonal logical. Should the diagonal cells of a square
#' matrix be excluded from the computation. Default is FALSE. The effect of setting
#' it to TRUE for non-square matrices may be unintuitive and should he avoided.
#' @returns G^2
#' @export
likelihood_ratio_chisq <- function(n, pi, exclude_diagonal=FALSE) {
  N <- sum(n)
  if (exclude_diagonal) {
    N <- N - sum(diag(n))
  }
  m <- N * pi
  g_squared <- 0.0
  for (i in 1:nrow(m)) {
    for (j in 1:ncol(m)) {
      if (exclude_diagonal && j == i) {
        next
      }
      # expect_true(pi[i, j] > 0.0, info="encountered non-positive pi")
      if (0.0 < n[i, j]) {
        g_squared <- g_squared + n[i, j] * log(n[i, j] / m[i, j])
      }
    }
  }
  2.0 * g_squared
}


#' Tests whether a square matrix is invertible (non singular)
#'
#' from stackoverflow:
#' https://stackoverflow.com/questions/24961983/how-to-check-if-a-matrix-has-an-inverse-in-the-r-language
#' @param X Matrix to be tested.  It is assumed X is square
#' @returns logical: TRUE if inversion succeeds, FALSE otherwise
#' @export
is_invertible <- function(X) !inherits(try(solve(X), silent = TRUE), "try-error")


#' Function to load a data set written out using save().
#'
#' The first (should be the only) element read from the RData file is returned
#' From:
#' https://stackoverflow.com/questions/5577221/how-can-i-load-an-object-into-a-variable-name-that-i-specify-from-an-r-data-file
#'
#' usage x <- loadRData(file_name="")
#' @param file_name Character. Name of the file containing the RData
#' @returns the first object from the restored RData
#' @export
loadRData <- function(file_name) {
  load(file_name)
  get(ls()[ls() != "file_name"])
}


#' Converts weighted (x, w) pairs into unweighted data by replicating x[i] w[i] times
#'
#' Takes a set of (value, weight) pairs and converts into unweighted vector (w[i]) for each i
#' Weights are assumed to be integers
#' @param x Numeric vector of scores.
#' @param w Numeric vector of weights. These are assumed to be integers
#' @returns new unweighted vector of scores
#' @export
expand <- function(x, w) {
  if (length(x) != length(w)) {
    stop(paste("lengths of vectors must match, x:", length(x), "w:", length(w)))
  } else if (length(x) < 1) {
    stop(paste("x must be a vector, x:", x))
  }

  var1 <- vector("double", sum(w))
  index <- 1
  for (i in 1:length(x)) {
    if (w[i] > 0) {
      for (wi in 1:w[i]) {
        var1[index] <- x[i]
        index <- index + 1
      }
    }
  }
  var1
}


#' Computes the weighted variance
#'
#' Computes variance between x and y using case weights in w
#' @importFrom stats weighted.mean
#' @param x Numeric vector. First variable
#' @param w Numeric vector. Case weights
#' @param use_df Logical. Should the divisor be sum of weights - 1 (TRUE) or N - 1 (FALSE)
#' @returns the weighted covariance between x and y
#' @export
weighted_var <- function(x, w, use_df=TRUE) {
  if (length(x) != length(w)) stop("lengths must be the same")
  mu = weighted.mean(x, w)
  if (use_df) {
    answer <- sum(w * (x - mu)^2) / (sum(w) - 1)
  } else {
    answer <- sum(w * (x - mu)^2) / sum(w)
  }
  answer
}


#' Computes the weighted covariance
#'
#' Computes covariance between x and y using case weights in w
#' @importFrom stats weighted.mean
#' @param x Numeric vector. First variable
#' @param y Numeric vector. Second variable
#' @param w Numeric vector. case weights
#' @param use_df Logical. should the divisor be sum of weights - 1 (TRUE) or N - 1 (FALSE)
#' @returns the weighted covariance between x and y
#' @export
weighted_cov <- function(x, y, w, use_df=TRUE) {
  if (length(x) != length(w) || length(x) != length(y)) stop("lengths must be the same")
  mu_x = weighted.mean(x, w)
  mu_y = weighted.mean(y, w)
  if (use_df) {
    answer <- sum(w * (x - mu_x) * (y - mu_y)) / (sum(w) - 1)
  } else {
    answer <- sum(w * (x - mu_x) * (y - mu_y)) / sum(w)
  }
  answer
}


#' Computes Cohen's 1960 kappa coefficient
#'
#' @param n matrix of observed counts
#' @returns kappa coefficient
#' @export
kappa <- function(n) {
  if (nrow(n) != ncol(n)) {
    stop(paste("Input matrix must be square, not", nrow(n), "by", ncol(n)))
  }
  r <- nrow(n)
  N <- sum(n)
  p_rows <- rowSums(n) / N
  p_columns <- colSums(n) / N
  p_agree <- sum(diag(n)) / N
  p_chance <- 0.0
  for (i in 1:r) {
    p_chance <- p_chance + p_rows[i] * p_columns[i]
  }

  kappa <- (p_agree - p_chance) / (1.0 - p_chance)
  se <- p_chance + p_chance^2
  term1 <- 0.0
  for (i in 1:r) {
    term1 <- term1 + p_rows[i] * p_columns[i] * (p_rows[i] + p_columns[i])
  }

  se <- se - term1
  se <- sqrt(se / (N * (1.0 - p_chance)^2))
  # the above appears to be algebraically equal
  # to the value returned from v_kappa()
  variances <- var_kappa(n)
  list(kappa=kappa, se=sqrt(variances$var_kappa0))
}


#' Computes Cohen's 1968 weighted kappa coefficient
#'
#' @param n matrix of observed counts
#' @param w matrix of weights. Defaults to identity matrix
#' @param quadratic logical. Should quadratic weights be used? Default is FALSE.
#' If TRUE, quadratic weights are used. These override the values in w.
#' If FALSE, weights in w are used
#' @returns value of weighted kappa
#' @export
weighted_kappa <- function(n, w=diag(rep(1.0, nrow(n))),quadratic=FALSE) {
  if (nrow(n) != ncol(n)) {
    stop(paste("Input matrix must be square, not", nrow(n), "by", ncol(n)))
  }
  r <- nrow(n)
  N <- sum(n)
  p <- n / N
  p_expected <- 0.0
  p_observed <- 0.0
  if (quadratic) {
    w <- matrix(0.0, nrow=r, ncol=r)
    for (i in 1:r) {
      for (j in 1:r) {
        w[i, j] <- (i - j) ^2
      }
    }
  }
  p_rows <- rowSums(n) / N
  p_columns <- colSums(n) / N

  p_expected <- sum(p * w)
  p_observed <- sum(p_rows %*% t(p_columns) * w)
  kappa <- 1.0 - (p_expected / p_observed)
  variances <- var_weighted_kappa(n, w)
  list(kappa=kappa, se=sqrt(variances$var_kappa0))
}


#' Computes the sampling variance of kappa.
#'
#' Formulas are from the paper by Fleiss,J. L., Cohen, J., & Everitt, B. S.
#' (1969).  Large sample standard errors of kappa and weighted kappa.
#' Two results are returned in a list.
#' var_kappa0 is the null case and would be used for testing the
#' hypothesis that kappa = 0.  The second is var_kappa and is for
#' the non-null case, such as constructing CI for estimated kappa.
#' Not that both are in the variance metric.  Take the square root to
#' get the standard error.
#' @param n matrix of observe counts
#' @returns a list containing;
#'  var_kappa0: variance for the null case
#'  var_kappa: variance for the non-null case.
#' @export
var_kappa <- function(n) {
  N <- sum(n)
  p <- n / N
  r <- nrow(n)

  p_rows <- rowSums(p)
  p_cols <- colSums(p)
  w_bar_rows <- p_cols
  w_bar_cols <- p_rows

  w_sum <- matrix(0.0, nrow=r, ncol=r)
  for (i in 1:r) {
    for (j in 1:r) {
      w_sum[i, j] <- w_bar_rows[i] + w_bar_cols[j]
    }
  }

  po <- sum(diag(p))
  pc <- sum(diag(p_rows %*% t(p_cols)))
  kappa <- (po - pc) / (1.0 - pc)

  d_term <- vector("double", r)
  term1 <- 0.0
  term2 <- 0.0
  for (i in 1:r) {
    for (j in 1:r) {
      if (j == i) {
        next
      }
      term2 <- term2 + p[i, j] * w_sum[i, j]^2
    }
    d_term[i] <- ((1 - pc) - w_sum[i, i] * (1.0 - po))^2
    term1 <- term1 + p[i, i] * d_term[i]
  }
  term3 <- term2 * (1.0 - po)^2
  term4 <- (po * pc - 2.0 * pc + po)^2
  denom <- N * (1.0 - pc)^4
  var_kappa <- (term1 + term3 - term4) / denom

  termA <- 0.0
  termB <- 0.0
  for (i in 1:r) {
    termA <- termA + w_bar_rows[i] * w_bar_cols[i] * (1.0 - w_sum[i, i])^2
    for (j in 1:r) {
      if (j == i) {
        next
      }
      termB <- termB + w_bar_rows[j] * w_bar_cols[i] * w_sum[i, j]^2
    }
  }
  var_kappa0 <- (termA + termB - pc^2) / (N * (1.0 - pc)^2)
  list(var_kappa0=var_kappa0, var_kappa=var_kappa)
}


#' Computes the sampling variance of weighted kappa.
#'
#' Formulas are from the paper by Fleiss,J. L., Cohen, J., & Everitt, B. S.
#' (1969).  Large sample standard errors of kappa and weighted kappa.
#' Two results are returned in a list.
#' var_kappa0 is the null case and would be used for testing the
#' hypothesis that kappa = 0.  The second is var_kappa and is for
#' the non-null case, such as constructing CI for estimated kappa.
#' Not that both are in the variance metric.  Take the square root to
#' get the standard error.
#' @param n matrix of observe counts
#' @param w matrix of penalty weights
#' @returns a list containing;
#'  var_kappa0: variance for the null case
#'  var_kappa: variance for the non-null case.
#' @export
var_weighted_kappa <- function(n, w) {
  N <- sum(n)
  p <- n / N
  r <- nrow(n)
  p_rows <- rowSums(p)
  p_cols <- colSums(p)
  w_bar_rows <- vector("double", r)
  w_bar_cols <- vector("double", r)
  for (i in 1:r) {
    w_bar_rows[i] <- 0.0
    w_bar_cols[i] <- 0.0
    for (j in 1:r) {
      w_bar_rows[i] <- w_bar_rows[i] + w[i, j] * p_cols[j]
      w_bar_cols[i] <- w_bar_cols[i] + w[i, j] * p_rows[j]
    }
  }
  w_bar_rows <- w_bar_rows
  w_bar_cols <- w_bar_cols

  pij <- p_rows %*% t(p_cols)
  po <- sum(w * p)
  pc <- sum(w * pij)

  w_sums <- matrix(0.0, nrow=r, ncol=r)
  e_terms <- matrix(0.0, nrow=r, ncol=r)
  f_terms <- matrix(0.0, nrow=r, ncol=r)
  for (i in 1:r) {
    for (j in 1:r) {
      w_sums[i, j] <- w_bar_rows[i] + w_bar_cols[j]
      e_terms[i, j] <- (w[i, j] * (1.0 - pc)
                        - (w_bar_rows[i] + w_bar_cols[j]) * (1.0 - po))^2
      f_terms[i, j] <- (w[i, j] - w_sums[i, j])^2
    }
  }

  term1 <- sum(p * e_terms)
  term2 <- (po * pc - 2.0 * pc + po)^2
  denom <- N * (1.0 - pc)^4
  var_kappa <- (term1 - term2) / denom

  term <- sum(pij * f_terms)
  var_kappa0 <- (term - pc^2) / (N * (1.0 - pc)^2)
  list(var_kappa0=var_kappa0, var_kappa=var_kappa)
}
