#
# Functions for making axes a bit easier!

#' Auto Axis Tool
#'
#' Overlay base plot with a new axis and optional gridlines. The axis spacing can
#' be manually specified or automatically generated, including for date and time
#' axis. A default grid is drawn if called with just the `side` specified.
#'
#' Major and minor tick marks can be specified in a number of ways:
#'
#'  - As a character string if the axis is datetime, such as 'year' or 'hour'
#'    which are passed as `by` to `seq()`. These can be prefixed with an integer multiplier,
#'    for example '6 hour' or '10 year', as per `seq.POSIXt`
#'
#'  - As a tick interval using the default `spacing = TRUE`
#'
#'  - As an approximate number of tick marks to include, using `pretty()` to find
#'    the best interval, using `spacing = FALSE`. Use a character number if this
#'    is a Date or Time axis, such as `major = '100'` and `spacing` will be set
#'    FALSE automatically.
#'
#' Major adds labels and ticks, minor is just half-sized ticks marks. Both
#' tick sizes can be changed (or direction changed) using `tck`.
#'
#' Three different datetime axis are possible: year, day-offset, seconds-offset. Use
#' `format` to specify how the label should appear, such as '%b %Y' (see `?strptime`)
#'
#'  - Year should be treated as a conventional numeric axis, use `major=1/12` not `major='month'`
#'
#'  - day-offset is an axis of `class(x)=='Date'` and is identified if the axis range exists
#'    within +/-9e4, meaning within dates 1723 - 2216, and minimum interval is 'day'
#'
#'  - second-offset is an axis of `class(x)=='POSIXct'` and is identified by a range outside
#'    of +/-9e4. This will give very strange results if your entire POSIXct axis is within
#'    24 hours of 1970-01-01
#'
#' A grid can be added at the same time by setting `major_grid` or `minor_grid` to `TRUE`
#' or a colour string. If `TRUE`, a transparent black is used by default.
#'
#' Any other options can be passed through to `axis()` directly (see `?axis`), most
#' notably `las = 2` to rotate the labels, and `cex.axis` for label size.
#'
#' The function will exit with a warning if more than 1000 ticks or gridlines were
#' generated, as this is most likely a mistake with autogenerated date / time intervals
#' and can lead to very slow behaviour.
#'
#' This does NOT work well for `barplot()` categorical axis, for this continue to use
#' the basic `axis()` function with custom labels, see examples.
#'
#' @examples
#' plot(sunspots) # This time series is actually given in decimal years
#'   autoaxis(side=3, major=50, major_grid='coral', minor=10, minor_grid=TRUE, spacing=TRUE)
#'   autoaxis(side=4, major=11, minor=25, spacing=FALSE, las=2, cex.axis=0.5, tck=0.02)
#'
#' plot(seq(as.POSIXct('2020-01-01'),as.POSIXct('2020-01-03'),length.out=1e3),
#'     rnorm(1e3), xlab='POSIXct', xaxt='n')
#'   autoaxis(side=1, major='day', minor='3 hour', format='%x')
#'   # Shortcut method to make a default dense grid
#'   autoaxis(side='3')
#'   autoaxis(side=2)
#'   # You can always request a datetime axis (side='4' not 4L) but it will be nonsense
#'   autoaxis(side='4', col='red')
#'
#' plot(seq(as.Date('2013-02-01'),as.Date('2020-01-03'),length.out=1e3),
#'     rnorm(1e3), xlab='Date', xaxt='n')
#'   autoaxis(side=1, major='10', minor='50', format='%Y')
#'   autoaxis(side=3, minor='3 month', minor_grid=TRUE)
#'
#' # Guessing is ambiguous with small values, depends on smallest interval
#' plot(1:500,runif(500), type='l', xaxt='n', xlab='Time or Date?', main=
#'   'For small values (<1e5), use interval to guess format\n')
#' autoaxis(1, major='min', minor='10 sec', format='%M:%S')
#' autoaxis(3, major='quarter', minor='month', format='%b %Y')
#'
#' # For barplot() use base functions - remember to set width=1, space=0
#' # otherwise bars will not be plotted on integer x-coordinates
#' barplot(mtcars$mpg, width=1, space=0, ylab='mpg')
#'   # Adjust the x-axis down by 0.5 so that the tick is in centre of each bar
#'   axis(side=1, at=-0.5+1:length(mtcars$mpg), labels=rownames(mtcars), las=2 )
#'   # Often prettier, label each bar inside the bar itself using text()
#'   text(x=-1+1:length(mtcars$mpg), y=1, pos=4,
#'     labels=rownames(mtcars), srt=90, cex=0.7)
#'   # autoaxis can still be used for adjusting the numeric scale
#'   autoaxis(side=2, major=5, major_grid=TRUE, minor=1, minor_grid=TRUE)
#'
#' @param side  Side to add axis, 1 = bottom, 2 = left, 3 = top, 4 = right. If
#'              only this argument is given, a default dense grid is drawn. If
#'              this argument is given as a character, a date-time grid will be
#'              attempted, for example `side='1'`
#' @param major Spacing of major axis ticks and labels (or approx. number of
#'              intervals if `spacing = FALSE`). If the axis is date or time,
#'              use a interval specified in `?seq.POSIXt`, such as 'sec' or
#'              'week', or character value for spacing such as `='20'`
#' @param major_grid Add grid lines corresponding to major axis ticks, `TRUE`
#'              to get default translucent black, otherwise colour (name or hex)
#' @param minor Spacing (or number) of minor ticks (note, no label for minor).
#'              If given as a character string, it will pass to `seq.POSIXt`
#' @param minor_grid Add gridlines for minor ticks, `TRUE` uses transparent
#'              black, otherwise colour string
#' @param format Date or time format for major axis for example `'%Y %b'`. If left
#'              as the default `'auto'` an appropriate choice between seconds
#'              and years will be used. Note, `major` or `side` must be given as a
#'              character string to trigger datetime labels.
#' @param tck   Size of axis tick: minor axis will always take half this value
#' @param spacing Should `major` and `minor` be interpreted as tick spacing
#'              (default) or approximate number of ticks
#' @param ...   Additional arguments passed to `axis()`, for example `las=2`
#'              for perpendicular labels
#'
#' @return No return value (`NULL`)
#'
#' @import graphics
#' @import grDevices
#' @export
autoaxis = function(side, major = NA, major_grid = FALSE, minor = NA, minor_grid = FALSE,
                    format = 'auto', spacing = TRUE, tck=-0.03, ...){
  if(side %in% c(1,3))
    lims = par('usr')[1:2] # Drawing x-axis
  else
    lims = par('usr')[3:4] # Drawing y-axis

  if(is.na(major) & is.na(minor)){
    major = if(is.character(side)) '10' else 10
    minor = if(is.character(side)) '50' else 50
    major_grid = TRUE
    minor_grid = TRUE
    spacing = FALSE
  }

  date_axis = class(side)=='character' | class(major)=='character' | class(minor)=='character'
  date_format = FALSE
  #if(date_axis==TRUE & spacing==FALSE) stop('spacing must be TRUE for time-interval axes')

  # Is a date axis wanted here?
  if(date_axis){
    # Guess whether datetime is in seconds (POSIXct) or days (as.Date)
    # If plot range is 1723-2216 (Date) or within 24hr of 1970-01-01
    # ==> assume it's a date, not seconds. Pretty safe!
    # Unless you have small time, such as ITime or difftime of only a few hours
    # and have EXPLICITLY asked for a small unit eg 'min'

    smallest_interval = which(c('year', 'quarter', 'month', 'week', 'day', 'hour', 'min', 'sec') %in% gsub('[^a-z]','',c(major,minor)))
    if(length(smallest_interval)==0){
      spacing = FALSE # only way this is going to work
      smallest_interval = 0
    }
    smallest_interval = max(smallest_interval)

    # If this were POSIX we would be talking about +/-1 day
    # If asking for at least a day, then assume it must be date
    # Possible to ask for 'hour' if date range one week
    if(lims[1] > -9e4 & lims[2] < 9e4){
      if(smallest_interval<=5) date_format = TRUE
      if(smallest_interval==6 & diff(lims)<=7) date_format = TRUE
    }

    # For manipulation, we will convert EVERYTHING into seconds
    # This will make pretty() and seq() behave themselves and give sub-day intervals
    if(date_format)
      lims = as.POSIXct.Date(lims, origin = '1970-01-01')
    else
      lims = as.POSIXct.numeric(lims, origin = '1970-01-01')
  }

  # If format is not specified then try to guess it from units()
  # Note, this is way more simple than axis.POSIXct() and axis.Date()
  # but will need tweaking to make it look natural as often as possible
  if(format=='auto' & date_axis){
    seconds_diff = diff(as.numeric(lims))
    format = as.character(cut(seconds_diff,
                 breaks=c(0,
                          60,
                          3600, # Less than a day is easy -- give 'time' only
                          24*3600, # More than 1 day, give "Weds 12:00"
                          7*24*3600, # More than 7 days,
                          10*24*3600, # Default ISO datestamp for more than 10 days
                          365*24*3600, # more than 1 year, give "Jan 2020"
                          10*365*24*3600, # more than 10 years, give only year
                          +Inf),
                 labels=c('%S',
                          '%M:%S',
                          '%H:%M',
                          '%a %H:%M',
                          '%a %d',
                          '%Y-%m-%d',
                          '%b %Y',
                          '%Y') ))
  }

  # Start off by getting pretty start / finish -- used for spacing = T or F
  if(!is.na(major)) major_at = pretty(lims, 2)
  if(!is.na(minor)) minor_at = pretty(lims, 2)

  # Create the tick 'at'
  # If you want to get this SPACING rather than number-of-ticks, seq()
  # If something like '6 hour' this is automatically used by seq.POSIXt
  if(spacing){
    if(!is.na(major)) major_at = seq(major_at[1], major_at[length(major_at)], by=major)
    if(!is.na(minor)) minor_at = seq(minor_at[1], minor_at[length(minor_at)], by=minor)
  }
  # Other option is to give major as an approx number of ticks
  if(!spacing & !date_axis){
    if(!is.na(major)) major_at = pretty(lims, major)
    if(!is.na(minor)) minor_at = pretty(lims, minor)
  }
  if(!spacing & date_axis){
    if(!is.na(major)) major_at = pretty(lims, as.integer(major))
    if(!is.na(minor)) minor_at = pretty(lims, as.integer(minor))
  }

  if(!is.na(major)){
    if(date_axis)
      major_labs = format(major_at,format)
    else
      major_labs = major_at
  }

  if(!is.na(major) & date_axis & date_format)
    major_at = as.numeric(major_at) / 86400 # Do not use as.Date - gets rid of decimal hour
  if(!is.na(minor) & date_axis & date_format)
    minor_at = as.numeric(minor_at) / 86400

  # Check length - otherwise can accidentally crash
  if(!is.na(major)) if(length(major_at)>1e3) stop('Major axis has more than 1000 ticks')
  if(!is.na(minor)) if(length(minor_at)>1e3) stop('Minor axis has more than 1000 ticks')
  if(!is.na(major)) if(length(major_at)<2) stop('Major axis has one or fewer ticks - check interval', if(date_axis) ' - do not use anything less than "hour" if as.Date axis' )
  if(!is.na(minor)) if(length(minor_at)<2) stop('Minor axis has one or fewer ticks - check interval', if(date_axis) ' - do not use anything less than "hour" if as.Date axis' )

  # Add the axis
  if(!is.na(major)) axis(side=side, at=major_at, labels=major_labs, tck=tck, ...)
  if(!is.na(minor)) axis(side=side, at=minor_at, labels=FALSE, tck=tck/2, ...)

  if(major_grid==FALSE & minor_grid==FALSE)
    return(NULL)

  if(major_grid==TRUE) major_grid = '#00000030' # Default transparent grey overlay
  if(minor_grid==TRUE) minor_grid = '#00000010'
  if(major_grid==FALSE) major_grid = NA   # Invisible colour
  if(minor_grid==FALSE) minor_grid = NA
  # Add grid lines - maybe needs more fine tune options here!
  if(side %in% c(1,3)){
    if(!is.na(major)) abline(v = major_at, col = major_grid)
    if(!is.na(minor)) abline(v = minor_at, col = minor_grid)
  }
  if(side %in% c(2,4)){
    if(!is.na(major)) abline(h = major_at, col = major_grid)
    if(!is.na(minor)) abline(h = minor_at, col = minor_grid)
  }

  return(invisible(NULL))
}
