#' @title Spectral irradiance for diverse LEDs
#'
#' @description A collection of emission spectra of light-emitting-diodes
#'   from different suppliers.
#'
#' @details The \code{leds.mspct} and \code{led_arrays.mspct} objects contain
#'   \code{"source_spct"} objects with spectral irradiance data, for single
#'   channel and multichannel LEDs, respectively. As the exact distance from LED
#'   to cosine diffuser and/or the driving current vary among spectra for
#'   different LEDs, members of \code{leds.mspct} have been all normalized to
#'   the wavelength of maximum spectral energy irradiance. In contrast, spectra
#'   in \code{led_arrays.mspct} have not been normalised as comparisons among
#'   channels of each array are meaningful, even if approximate.
#'   
#'   When the details of the measurement conditions are know, these
#'   are stored as metadata attributes. In any case, it needs to be taken into
#'   account than even in these cases measurements have not been done in an
#'   optical bench, so values of spectral irradiance are subject to errors due
#'   to possible misalignment. The shape of the spectra, in contrast can be
#'   relied upon as measurements were done with well calibrated instruments.
#'   
#'   The output of LEDs at a given current decreases as their temperature
#'   increases. The wavelength at the peak of emission can depend on the
#'   temperature and current, but shifts tend to be only a couple of nanometres.
#'   In LED arrays with heterogeneous LED chips or white LEDs based on secondary
#'   emission from phosphor the shape of the spectrum can slightly change
#'   depending on the drive current and temperature.
#'   
#'   There is variation among LEDs of the same type, specially with respect
#'   wavelength and light output. The data included are for individual LEDs and
#'   can be expected to differ to some extent from the typical values in the
#'   manufacturers specifications. Some of the LEDs for which data are included
#'   are only of historical interest as their production has been discontinued,
#'   usually replaced by new types with enhanced performance. When known, the
#'   approximate "vintage" is provided in the metadata.
#'   
#'   Some LEDs are "generic" ones which lacked detailed data sheets at the time
#'   of purchase, or even lacked a manufacturer's type denomination code. For
#'   them, the metadata is necessarily less detailed.
#'   
#'   The spectral data are not expressed at constant wavelength intervals. Not
#'   only the intervals vary in the raw data from the array spectrometer, but in
#'   addition function \code{\link[photobiology]{thin_wl}} has been applied to
#'   reduce the storage space needed. In brief the wavelength interval has been
#'   increased as much as possible in those regions of the spectrum that lack
#'   detailed features (such as linear slopes and wavelength regions with zero
#'   light emission). The algorithm used preserved the location and height of 
#'   significant energy emission peaks, i.e., the global peak and clear local 
#'   peaks. 
#'
#' @note Please see the help page for \code{\link{led_brands}} for LED 
#' suppliers' contact information. 
#' Please see the metadata in each spectrum for other information. 
#' These metadata are
#' stored as attributes of the individual \code{source_spct} objects and can 
#' accessed with functions
#' \code{\link{comment}},
#' \code{\link[photobiology]{getWhatMeasured}},
#' \code{\link[photobiology]{getWhenMeasured}} and
#' \code{\link[photobiology]{getHowMeasured}}.
#' Many of the spectra also contain information on the measurement procedure accessible with 
#' \code{\link[photobiology]{getInstrDesc}} and
#' \code{\link[photobiology]{getInstrSettings}}.
#'
#' @seealso \code{\link{oo_maya_leds}}
#'
#' @docType data
#' @keywords datasets
#' @format A \code{"source_mspct"} object containing 74 
#'   \code{"source_spct"} objects.
#' 
#' In each of the member spectra, the variables are as follows:
#' \itemize{
#'   \item w.length (nm)  
#'   \item s.e.irrad (W m-2 nm-1)
#' }
#'
#' @examples
#' library(photobiology)
#' 
#' names(leds.mspct)
#' 
#' names(led_arrays.mspct)
#' 
#' leds.mspct$Nichia_NS6L183AT_H1_sw
#' 
#' cat(getWhatMeasured(leds.mspct$Nichia_NS6L183AT_H1_sw))
#' 
#' peaks(leds.mspct$Nichia_NS6L183AT_H1_sw, span = 100)
#' 
#' wl_range(leds.mspct$Nichia_NS6L183AT_H1_sw)
#' 
#' wl_stepsize(leds.mspct$Nichia_NS6L183AT_H1_sw)
#' 
#' intersect(LedEngin_leds, blue_leds)
#' 
#' leds.mspct[intersect(LedEngin_leds, blue_leds)]
#' 
"leds.mspct"

#' @rdname leds.mspct
#' 
"led_arrays.mspct"