#' @title One-Compartment Oral Pharmacokinetic Model (Nonlinear, First-Order Absorption)
#' @name one_compartment_oral_nl
#' @description
#' Fits plasma concentration-time data to a one-compartment pharmacokinetic model
#' following oral administration with first-order absorption and first-order
#' elimination. The model assumes that drug absorption from the gastrointestinal
#' tract and systemic elimination are both proportional to the amount of drug
#' present (linear kinetics), and that the body can be represented as a single,
#' well-mixed compartment.
#'
#' Model parameters are estimated by nonlinear least squares regression for each
#' group (if specified). The primary parameters include the absorption rate constant
#' (k_a), elimination rate constant (k_el), and apparent volume of distribution
#' (Vd/F). Model-consistent secondary pharmacokinetic parameters are derived,
#' including time to maximum concentration (Tmax), maximum concentration (Cmax),
#' area under the concentration-time curve (AUC), and apparent clearance (CL/F),
#' where AUC = Dose / (k_el * Vd/F) and CL/F = k_el * Vd/F.
#'
#' The function includes safeguards against numerical instability when k_a and
#' k_el are similar, enforces positive parameter bounds during fitting, and performs
#' validity checks for the computation of Tmax and Cmax. Model performance is
#' summarized using root mean squared error (RMSE) and information criteria
#' (AIC and BIC), which are more appropriate than R^2 for nonlinear pharmacokinetic
#' models.
#'
#' If grouping is specified (e.g., by subject or formulation), the model is fit
#' independently to each group. Groups with insufficient observations trigger a
#' warning, as parameter estimates may be unreliable.
#'
#' When plotting is enabled, the function generates publication-quality
#' concentration-time plots showing observed data and the corresponding
#' model-predicted curves based on the fitted parameters. Annotations summarizing
#' key pharmacokinetic parameters and model diagnostics are optionally added for
#' one group.
#'
#' An optional LOESS (locally estimated scatterplot smoothing) curve may be overlaid
#' on the observed data for exploratory visualization only. This smoother is purely
#' descriptive, does not represent any pharmacokinetic mechanism, and should not be
#' interpreted as part of the fitted model. For this reason, LOESS smoothing is
#' disabled by default.
#'
#' Model:
#' C(t) = ((F * Dose * k_a) / (Vd * (k_a - k_el))) * [exp(-k_el * t) - exp(-k_a * t)]
#'
#' @param data A data frame containing plasma concentration-time data.
#' @param time_col Character string specifying the column name for time.
#' @param conc_col Character string specifying the column name for plasma concentration.
#' @param dose Numeric value specifying the administered oral dose.
#' @param group_col Optional character string specifying a grouping variable
#'   (e.g., formulation, subject).
#' @param plot Logical; if TRUE, generates a concentration-time plot with fitted curves.
#' @param annotate Logical; if TRUE, annotates the plot with PK parameters
#'   (only if <= 2 groups).
#'
#' @import stats
#' @import ggplot2
#' @importFrom stats nls na.omit
#' @importFrom ggplot2 ggplot aes geom_point geom_line geom_smooth geom_text labs
#' theme theme_bw element_text element_blank
#'
#' @return A list containing:
#' \describe{
#'   \item{\code{fitted_parameters}}{Data frame with k_a, k_el, Tmax, Cmax, Vd/F, CL/F, and R^2.}
#'   \item{\code{data}}{Processed data used for fitting and plotting.}
#' }
#' @examples
#' # Example I: Single subject oral dosing
#' df <- data.frame(
#'   time = c(0.25, 0.5, 1, 2, 4, 6, 8, 12),
#'   concentration = c(1.2, 2.8, 5.1, 6.4, 5.2, 4.1, 3.0, 1.8)
#' )
#' one_compartment_oral_nl(
#'   data = df,
#'   time_col = "time",
#'   conc_col = "concentration",
#'   dose = 100
#' )
#'
#' # Example II: Condition-dependent oral pharmacokinetics (e.g., formulation or pH effect)
#' df_cond <- data.frame(
#'   time = rep(c(0.25, 0.5, 1, 2, 4, 6, 8), 2),
#'   concentration = c(
#'     1.4, 3.1, 5.6, 6.8, 5.9, 4.7, 3.6,   # Condition A
#'     0.9, 2.2, 4.1, 5.3, 4.8, 3.9, 3.0    # Condition B
#'   ),
#'   condition = rep(c("Condition A", "Condition B"), each = 7)
#' )
#' one_compartment_oral_nl(
#'   data = df_cond,
#'   time_col = "time",
#'   conc_col = "concentration",
#'   dose = 100,
#'   group_col = "condition"
#' )
#'
#' # Example III: Multiple subjects (population-style oral pharmacokinetics)
#' df_subjects <- data.frame(
#'   time = rep(c(0.25, 0.5, 1, 2, 4, 6, 8, 12), 6),
#'   concentration = c(
#'     1.3, 3.0, 5.4, 6.7, 5.8, 4.6, 3.5, 2.3,   # Subject 1
#'     1.1, 2.7, 5.0, 6.3, 5.5, 4.4, 3.3, 2.2,   # Subject 2
#'     1.0, 2.5, 4.7, 6.0, 5.3, 4.2, 3.2, 2.1,   # Subject 3
#'     0.9, 2.3, 4.4, 5.7, 5.0, 4.0, 3.0, 2.0,   # Subject 4
#'     0.8, 2.1, 4.1, 5.4, 4.8, 3.8, 2.9, 1.9,   # Subject 5
#'     0.7, 2.0, 3.9, 5.2, 4.6, 3.7, 2.8, 1.8    # Subject 6
#'   ),
#'   subject = rep(paste0("S", 1:6), each = 8)
#' )
#' one_compartment_oral_nl(
#'   data = df_subjects,
#'   time_col = "time",
#'   conc_col = "concentration",
#'   dose = 100,
#'   group_col = "subject"
#' )
#' @references Gibaldi, M. & Perrier, D. (1982) <isbn:9780824710422> Pharmacokinetics,
#' 2nd Edition. Marcel Dekker, New York.
#' @references Gabrielsson, J. & Weiner, D. (2000) <isbn:9186274929> Pharmacokinetic/Pharmacodynamic
#' Data Analysis: Concepts and Applications, 3rd Edition, Revised and Expanded.
#' Swedish Pharmaceutical Press, Stockholm.
#' @author Paul Angelo C. Manlapaz
#' @export

utils::globalVariables(c("time", "conc", "group", "ka", "kel", "Tmax", "Cmax",
                         "Vd_F", "CL_F", "AUC", "RMSE", "AIC", "BIC", "label",
                         "tail"))

one_compartment_oral_nl <- function(data,
                                    time_col = "time",
                                    conc_col = "concentration",
                                    dose,
                                    group_col = NULL,
                                    plot = TRUE,
                                    annotate = TRUE,
                                    loess = FALSE) {

  if (!requireNamespace("ggplot2", quietly = TRUE))
    stop("Package 'ggplot2' is required.")

  # -------------------------
  # Prepare data
  # -------------------------
  df <- data[, c(time_col, conc_col, group_col), drop = FALSE]
  df <- stats::na.omit(df)
  colnames(df)[1:2] <- c("time", "conc")

  df <- df[df$time >= 0 & df$conc > 0, ]

  # -------------------------
  # Group handling
  # -------------------------
  if (!is.null(group_col)) {
    df$group <- as.factor(df[[group_col]])
  } else {
    df$group <- factor("Experimental")
  }

  # Validate group sizes
  grp_n <- table(df$group)
  if (any(grp_n < 4)) {
    warning("Some groups have fewer than 4 observations; parameter estimates may be unreliable.")
  }

  # -------------------------
  # Helper: PK model
  # -------------------------
  pk_fun <- function(time, ka, kel, vd) {
    if (abs(ka - kel) < 1e-6) return(rep(NA, length(time)))
    (dose * ka / (vd * (ka - kel))) *
      (exp(-kel * time) - exp(-ka * time))
  }

  # -------------------------
  # Model fitting by group
  # -------------------------
  fit_results <- do.call(rbind, lapply(split(df, df$group), function(d) {

    # ---- Improved starting values ----
    # kel from terminal slope
    tail_n <- max(3, floor(nrow(d) / 3))
    term <- tail(d[order(d$time), ], tail_n)
    kel_start <- abs(coef(lm(log(conc) ~ time, term))[2])
    kel_start <- ifelse(is.finite(kel_start), kel_start, 0.1)

    # ka from Tmax approximation
    tmax_obs <- d$time[which.max(d$conc)]
    ka_start <- max(1 / tmax_obs, kel_start * 2)

    vd_start <- dose / max(d$conc)

    # ---- Fit with bounds and guard ----
    fit <- stats::nls(
      conc ~ (dose * ka / (vd * (ka - kel))) *
        (exp(-kel * time) - exp(-ka * time)),
      data = d,
      start = list(ka = ka_start, kel = kel_start, vd = vd_start),
      algorithm = "port",
      lower = c(ka = 1e-4, kel = 1e-4, vd = 1e-4),
      control = stats::nls.control(maxiter = 300)
    )

    coef_fit <- coef(fit)
    ka  <- coef_fit["ka"]
    kel <- coef_fit["kel"]
    vd  <- coef_fit["vd"]

    # ---- Safety check for Tmax / Cmax ----
    if (ka <= kel) {
      tmax <- NA
      cmax <- NA
    } else {
      tmax <- log(ka / kel) / (ka - kel)
      cmax <- pk_fun(tmax, ka, kel, vd)
    }

    # ---- Model-consistent AUC & CL/F ----
    cl  <- kel * vd
    auc <- dose / cl

    # ---- Fit diagnostics ----
    pred <- predict(fit)
    rmse <- sqrt(mean((d$conc - pred)^2))

    n <- length(d$conc)
    p <- length(coef_fit)
    rss <- sum((d$conc - pred)^2)

    aic <- n * log(rss / n) + 2 * p
    bic <- n * log(rss / n) + log(n) * p

    data.frame(
      group = unique(d$group),
      ka = ka,
      kel = kel,
      Tmax = tmax,
      Cmax = cmax,
      Vd_F = vd,
      CL_F = cl,
      AUC = auc,
      RMSE = rmse,
      AIC = aic,
      BIC = bic
    )
  }))

  # -------------------------
  # Plotting
  # -------------------------
  if (plot) {

    # Generate prediction grid
    pred_df <- do.call(rbind, lapply(split(df, df$group), function(d) {
      pars <- fit_results[fit_results$group == unique(d$group), ]
      t_seq <- seq(min(d$time), max(d$time), length.out = 200)
      data.frame(
        time = t_seq,
        conc = pk_fun(t_seq, pars$ka, pars$kel, pars$Vd_F),
        group = pars$group
      )
    }))

    p <- ggplot2::ggplot(df, ggplot2::aes(time, conc, color = group)) +
      ggplot2::geom_point(size = 3) +
      ggplot2::geom_line(data = pred_df,
                         ggplot2::aes(time, conc, color = group),
                         linewidth = 1) +
      ggplot2::labs(
        title = "One-Compartment Oral Pharmacokinetic Model",
        subtitle = "First-order absorption and elimination",
        x = "Time (hours)",
        y = "Plasma Concentration (mg/L)",
        color = "Group"
      ) +
      ggplot2::theme_bw(base_size = 14) +
      ggplot2::theme(
        plot.title = ggplot2::element_text(
          face = "bold",
          hjust = 0.5
        ),
        plot.subtitle = ggplot2::element_text(
          hjust = 0.5
        ),
        panel.grid.major = ggplot2::element_blank(),
        panel.grid.minor = ggplot2::element_blank()
      )

    if (loess) {
      p <- p +
        ggplot2::geom_smooth(method = "loess", se = FALSE,
                             linetype = "dashed", alpha = 0.5)
    }

    # ---- Annotation ----
    if (annotate && nrow(fit_results) <= 1) {
      ann <- fit_results
      ann$label <- paste0(
        "ka = ", signif(ann$ka, 3), "\n",
        "kel = ", signif(ann$kel, 3), "\n",
        "Tmax = ", round(ann$Tmax, 2), "\n",
        "Cmax = ", round(ann$Cmax, 2), "\n",
        "Vd/F = ", round(ann$Vd_F, 2), "\n",
        "CL/F = ", round(ann$CL_F, 2), "\n",
        "AUC = ", round(ann$AUC, 2), "\n",
        "RMSE = ", round(ann$RMSE, 3), "\n",
        "AIC = ", round(ann$AIC, 1), "\n",
        "BIC = ", round(ann$BIC, 1)
      )

      p <- p +
        ggplot2::geom_text(
          data = ann,
          ggplot2::aes(x = Inf, y = Inf, label = label, color = group),
          hjust = 1.1, vjust = 1.1,
          show.legend = FALSE
        )
    }

    print(p)
  }

  return(list(
    fitted_parameters = fit_results,
    data = df
  ))
}
