% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/output.R
\name{predict.gamlasso}
\alias{predict.gamlasso}
\title{Prediction from a fitted gamlasso model}
\usage{
\method{predict}{gamlasso}(
  object,
  newdata = NULL,
  type = "link",
  s = "lambda.min",
  new.event.times = NULL,
  ...
)
}
\arguments{
\item{object}{fitted model object of the class \code{gamlasso} as
produced by \code{gamlasso}}

\item{newdata}{A data frame with the values of the linear and smooth
variables for which predictions are to be made. If not provided then
predictions corresponding to the original data used to fit \code{object}
is returned. If provided then the variable names (column names) should
match with the variable names used to fit \code{object}: the code throws
an error if not.}

\item{type}{When this has the value \code{"link"} (default) then the
linear predictor (with offset added if needed) is returned. When
\code{type = "response"} predictions on the response scale is returned,
depending on the family used while fitting \code{object}.}

\item{s}{Value of the lasso penalty parameter \code{lambda} at which
predictions are required. Default is \code{"lambda.min"} but alternatively
\code{"lambda.1se"} can be used.}

\item{new.event.times}{A vector of new event times to be used for predicting
survival times when \code{type = "response"} for a gamlasso object fitted
with \code{family = "cox"}}

\item{...}{Other arguments}
}
\value{
Returns a vector of the same length as \code{nrow(newdata)} with
  the values of the linear predictor or on the response scale depending
  on \code{type}. For \code{type = "link"} the value is simply the elementwise
  sum of the predictions from the gam and lasso models in \code{object}.
  For \code{type = "response"} the values are on the response scale, for
  example exponential of the linear response is returned if
  \code{object$inherit$family = "poisson"}
}
\description{
Takes a fitted \code{gamlasso} object produced by \code{gamlasso} and
  returns predictions given a new set of values of the linear and smooth
  variables.
}
\details{
Lasso models do not have standard errors so \code{predict.gamlasso}
  does not provide them either. The standard errors for the gam part of the
  model can be accesed by using \code{mgcv::predict.gam} with suitable options.
  Offsets are always included in the prediction if present in the original call
  to \code{gamlasso}. Also if \code{type} is anything other than \code{"link"}
  or \code{"response"} then the function throws an error.
}
\examples{
library(plsmselect)

data(simData)

## Fit poisson gamlasso model using the term specification approach:
## (L2-penalty on linear terms & L2-penalty on smooth terms)
pfit = gamlasso(response="Yp",
                linear.terms=paste0("x",1:10),
                smooth.terms=paste0("z",1:4),
                data=simData,
                linear.penalty = "l2",
                smooth.penalty = "l2",
                family="poisson",
                num.knots = 5,
                seed=1)

## fitted values (of linear predictor):
fitted.values <- predict(pfit)

## predicted values on response scale:
pred.response <- predict(pfit, type="response", newdata=simData)

\donttest{## For same model as above, but with L1-penalty on linear terms
## i.e. L1-penalty on the model matrix (X) we can use formula approach:
simData$X = model.matrix(~x1+x2+x3+x4+x5+x6+x7+x8+x9+x10, data=simData)[,-1]

pfit = gamlasso(Yp ~ X +
                   s(z1, k=5) + # L2-penalty (bs="tp") is default (see ?mgcv::s)
                   s(z2, k=5) +
                   s(z3, k=5) +
                   s(z4, k=5),
                 family="poisson",
                 data = simData,
                 seed=1)
}
# See ?gamlasso for an example fitting a gaussian response model
# See ?summary.gamlasso for an example fitting a binomial response model
# See ?cumbasehaz for an example fitting a survival response model
}
\seealso{
\code{\link{gamlasso}}, \code{\link[mgcv]{predict.gam}},
  \code{\link[glmnet]{predict.glmnet}}.
}
