#include <RcppArmadillo.h>
// [[Rcpp::depends(RcppArmadillo)]]

using namespace Rcpp;
using namespace arma;

arma::vec fill_v(int a, int b, arma::vec x){
  int n = b - a + 1;
  arma::vec y(n);
  for(int i=0; i<n; i++){
    y(i) = x(i+a-1);
  }
  return(y);
}

//' @name Rho Koenker
//'
//' @param x generic vector
//' @param tau percentile
//' 
//' @return y vector, linear transformation by rho 
//' @keywords internal
//' @noRd
arma::vec rho_koenker(arma::vec x, double tau){
  arma::vec y = x;                             // copy x
  arma::uvec neg = arma::find(x < 0);          // indices where x < 0
  arma::uvec pos = arma::find(x >= 0);         // indices where x >= 0
  y(neg) *= (tau - 1.0);
  y(pos) *= tau;
  return y;
}

//' @name Rho M-quantile
//'
//' @param x generic vector
//' @param tau percentile
//' @param c tuning
//' 
//' @return y vector, linear transformation by m-rho
//' @keywords internal
//' @noRd
// [[Rcpp::export]]
arma::vec rho_mq(arma::vec x, double tau, double c){
  arma::vec a = arma::abs(x);
  arma::vec y(a.n_elem);
  // Indicators
  arma::vec b1 = arma::conv_to<arma::vec>::from(a > c);   // large residuals
  arma::vec b2 = 1.0 - b1;                                // small residuals
  // tau weight depending on sign of x
  arma::vec w = arma::conv_to<arma::vec>::from(x < 0);
  w = (1.0 - w) * tau + w * (1.0 - tau);                   // if x<0 → 1-tau else → tau
  // Compute y1 and y2
  arma::vec y1 = w % (c * a - 0.5 * c * c);                // linear for |x|>c
  arma::vec y2 = w % (0.5 * arma::square(a));              // quadratic for |x|<=c
  // Combine
  y = b1 % y1 + b2 % y2;
  return y;
}

//' @name Psi M-quantile
//'
//' @param x generic vector
//' @param tau percentile
//' @param c tuning
//' 
//' @keywords internal
//' @noRd
arma::vec psi_mq(arma::vec x, double tau, double c){
  arma::vec a = arma::abs(x);
  arma::vec y(a.n_elem);
  // Indicators
  arma::vec b1 = arma::conv_to<arma::vec>::from(a > c);   // large residuals
  arma::vec b2 = 1.0 - b1;                                // small residuals
  // tau weight depending on sign of x
  arma::vec w = arma::conv_to<arma::vec>::from(x < 0);
  w = (1.0 - w) * tau + w * (1.0 - tau);                   // if x<0 → 1-tau else → tau
  // psi contributions
  arma::vec y1 = w % (c * x / a);     // for |x| > c
  arma::vec y2 = w % x;               // for |x| ≤ c
  // Combine regions
  y = b1 % y1 + b2 % y2;
  return y;
}

//' @name D Psi M-quantile
//'
//' @description Derivative of psi M-quantile
//'
//' @param x generic vector
//' @param tau percentile
//' @param c tuning
//' 
//' @return y vector, linear transformation by second derivative m-rho
//' @keywords internal
//' @noRd
arma::vec d_psi_mq(arma::vec x, double tau, double c){
  arma::vec a = arma::abs(x);
  arma::vec y(a.n_elem);
  // Region indicators
  arma::vec b2 = arma::conv_to<arma::vec>::from(a <= c);   // |x| ≤ c
  arma::vec b1 = 1.0 - b2;                                 // |x| > c
  // Weight depending on sign of x
  arma::vec w = arma::conv_to<arma::vec>::from(x < 0);
  w = (1.0 - w) * tau + w * (1.0 - tau);    // tau or (1 - tau)
  // For |x|>c derivative=0, for |x|≤c derivative=w
  y = b2 % w;
  return y;
}

arma::vec rho_als(arma::vec x, double tau){
  arma::vec y = arma::square(x);  // x^2 for all elements
  // Weight depending on sign of x
  arma::vec w = arma::conv_to<arma::vec>::from(x < 0);
  w = (1.0 - w) * tau + w * (1.0 - tau);   // if x<0 → 1−tau else → tau
  return y % w;
}

//' @name Psi ALS
//'
//' @description Psi asymetric least square
//'
//' @param x generic vector
//' @param tau percentile
//' 
//' @return y vector, linear transformation by ALS psi
//' @keywords internal
//' @noRd
arma::vec psi_als(arma::vec x, double tau){
  arma::vec y = 2 * x;  // multiply all elements by 2
  // Weight depending on sign of x
  arma::vec w = arma::conv_to<arma::vec>::from(x < 0);
  w = (1.0 - w) * tau + w * (1.0 - tau);   // if x<0 → 1−tau else → tau
  return y % w;
}

//' @name D Psi ALS
//'
//' @description Derivative of Psi asymetric least square
//'
//' @param x generic vector
//' @param tau percentile
//' 
//' @return y vector, linear transformation by derivative ALS psi
//' @keywords internal
//' @noRd
arma::vec d_psi_als(arma::vec x, double tau){
  // Weight depending on sign of x
  arma::vec w = arma::conv_to<arma::vec>::from(x < 0);
  w = (1.0 - w) * (2 * tau) + w * (2 * (1.0 - tau));  // if x<0 → 2*(1-tau), else 2*tau
  return w;
}

//' @name Loss quantile regression
//' 
//' @description This function returns the core of quantile regression to be minimized
//'
//' @param beta initial values
//' @param x design matrix
//' @param y vector output
//' @param tau percentile
//' @param N sample size
//' @param d columns of x  
//' 
//' @return eta Numeric, sum of quantile regression
//' @keywords internal
//' @noRd
// [[Rcpp::export]]
double loss_qr_simple(arma::vec beta, arma::mat x, arma::vec y, double tau, int N, int d){
  double eta = 0;
  arma::vec res(N);
  arma::vec rho(N);
  res = y - (x * beta);
  rho = rho_koenker(res,tau);
  eta = accu(rho);
  return(log(eta));
}

//' @name Loss quantile regression with fixed effects
//' 
//' @description This function returns the core of quantile regression with fixed effects to be minimized
//'
//' @param theta initial values
//' @param x design matrix
//' @param y vector output
//' @param z incident matrix
//' @param tau percentile
//' @param n N sample size
//' @param d columns of x
//' @param mm n columns of z   
//' 
//' @return eta Numeric, sum of quantile regression with fixed effects
//' @keywords internal
//' @noRd
// [[Rcpp::export]]
double loss_qrfe(arma::vec theta,arma::mat x,arma::vec y,arma::mat z,double tau,int n,int d,int mm){
  double eta;
  arma::vec beta(d);
  arma::vec alpha(mm);
  arma::vec res(n);
  arma::vec rho(n);
  beta = fill_v(1,d, theta); 
  alpha = fill_v((d+1),d+mm, theta); 
  // If first column of x is all 1 → enforce sum(alpha) = 0
  if (arma::all(x.col(0) == 1.0)) {
    alpha -= arma::mean(alpha);  // subtract mean to enforce sum(alpha) = 0
  }
  res = y -  (z * alpha) -  (x * beta);
  rho = rho_koenker(res,tau);
  eta = accu(rho);
  return(log(eta));
}

//' @name Loss lasso quantile regression with fixed effects
//'
//' @description This function returns the core of lasso quantile regression with fixed effects to be minimized
//'
//' @param theta initial values
//' @param x design matrix
//' @param y vector output
//' @param z incident matrix
//' @param tau percentile
//' @param n N sample size
//' @param d columns of x
//' @param mm n columns of z  
//' @param lambda constriction parameter
//' 
//' @return eta Numeric, sum of lasso quantile regression with fixed effects
//' @keywords internal
//' @noRd
// [[Rcpp::export]]
double loss_qrlasso(arma::vec theta,arma::mat x,arma::vec y,arma::mat z,double tau,int n,int d,int mm, double lambda){
  double eta;
  arma::vec beta(d);
  arma::vec alpha(mm);
  arma::vec res(n);
  arma::vec rho(n);
  beta = fill_v(1,d, theta); 
  alpha = fill_v(d+1,d+mm, theta); 
  // If first column of x is all 1 → enforce sum(alpha) = 0
  if (arma::all(x.col(0) == 1.0)) {
    alpha -= arma::mean(alpha);  // subtract mean to enforce sum(alpha) = 0
  }
  res = y -  (z * alpha) -  (x * beta);   
  rho = rho_koenker(res,tau);
  eta = accu(rho)/n + lambda * accu(abs(alpha));
  return(log(eta));
}

//' @name Loss M-quantile regression
//'
//' @description This function returns the core of M-quantile regression to be minimized
//'
//' @param beta initial values
//' @param x design matrix
//' @param y vector output
//' @param tau percentile
//' @param N sample size
//' @param d columns of x  
//' @param c tuning
//' 
//' @return eta Numeric, sum of M-quantile regression
//' @keywords internal
//' @noRd
// [[Rcpp::export]]
double loss_mqr(arma::vec beta, arma::mat x, arma::vec y, double tau, int N, int d, double c){
  double eta = 0;
  arma::vec res(N);
  arma::vec rho(N);
  res = y -  (x * beta);
  rho = rho_mq(res,tau,c);
  eta = accu(rho);
  return(log(eta));
}

//' @name Loss M-quantile regression with fixed effects
//'
//' @description This function returns the core of M-quantile regression with fixed effects to be minimized
//'
//' @param theta initial values
//' @param x design matrix
//' @param y vector output
//' @param z incident matrix
//' @param tau percentile
//' @param n N sample size
//' @param d columns of x
//' @param mm n columns of z 
//' @param c tuning
//' 
//' @return eta Numeric, sum of M-quantile regression with fixed effects
//' @keywords internal
//' @noRd
// [[Rcpp::export]]
double loss_mqrfe(arma::vec theta,arma::mat x,arma::vec y,arma::mat z,double tau,int n,int d,int mm, double c){
  double eta;
  arma::vec beta(d);
  arma::vec alpha(mm);
  arma::vec res(n);
  arma::vec rho(n);
  beta = fill_v(1,d, theta); 
  alpha = fill_v((d+1),d+mm, theta); 
  // If first column of x is all 1 → enforce sum(alpha) = 0
  if (arma::all(x.col(0) == 1.0)) {
    alpha -= arma::mean(alpha);  // subtract mean to enforce sum(alpha) = 0
  }
  res = y -  (z * alpha) -  (x * beta);
  rho = rho_mq(res,tau,c);
  eta = accu(rho);
  return(log(eta));
}

//' @name Loss lasso M-quantile regression with fixed effects
//' 
//' @description This function returns the core of lasso M-quantile regression with fixed effects to be minimized   
//'
//' @param theta initial values
//' @param x design matrix
//' @param y vector output
//' @param z incident matrix
//' @param tau percentile
//' @param n N sample size
//' @param d columns of x
//' @param mm n columns of z  
//' @param c tuning
//' @param lambda constriction parameter
//' 
//' @return eta Numeric, sum of lasso M-quantile regression with fixed effects
//' @keywords internal
//' @noRd
// [[Rcpp::export]]
double loss_mqrlasso(arma::vec theta,arma::mat x,arma::vec y,arma::mat z,double tau,int n,int d,int mm, double c, double lambda){
  double eta;
  arma::vec beta(d);
  arma::vec alpha(mm);
  arma::vec res(n);
  arma::vec rho(n);
  beta = fill_v(1,d, theta); 
  alpha = fill_v(d+1,d+mm, theta); 
  // If first column of x is all 1 → enforce sum(alpha) = 0
  if (arma::all(x.col(0) == 1.0)) {
    alpha -= arma::mean(alpha);  // subtract mean to enforce sum(alpha) = 0
  }
  res = y -  (z * alpha) -  (x * beta);   
  rho = rho_mq(res,tau,c);
  eta = accu(rho)/n + lambda * accu(abs(alpha));
  return(log(eta));
}

//' @name Loss expectile regression
//'
//' @description This function returns the core of expectile regression to be minimized   
//' 
//' @param beta initial values
//' @param x design matrix
//' @param y vector output
//' @param tau percentile
//' @param N sample size
//' @param d columns of x  
//' 
//' @return eta Numeric, sum of expectile regression
//' @keywords internal
//' @noRd
// [[Rcpp::export]]
double loss_er(arma::vec beta, arma::mat x, arma::vec y, double tau, int N, int d){
  double eta = 0;
  arma::vec res(N);
  arma::vec rho(N);
  res = y -  (x * beta);
  rho = rho_als(res,tau);
  eta = accu(rho);
  return(log(eta));
}

//' @name Loss expectile regression with fixed effects
//'
//' @description This function returns the core of expectile regression with fixed effects to be minimized   
//'
//' @param theta initial values
//' @param x design matrix
//' @param y vector output
//' @param z incident matrix
//' @param tau percentile
//' @param n N sample size
//' @param d columns of x
//' @param mm n columns of z  
//' 
//' @return eta Numeric, sum of expectile regression with fixed effects
//' @keywords internal
//' @noRd
// [[Rcpp::export]]
double loss_erfe(arma::vec theta,arma::mat x,arma::vec y,arma::mat z,double tau,int n,int d,int mm){
  double eta;
  arma::vec beta(d);
  arma::vec alpha(mm);
  arma::vec res(n);
  arma::vec rho(n);
  beta = fill_v(1,d, theta); 
  alpha = fill_v((d+1),d+mm, theta); 
  // If first column of x is all 1 → enforce sum(alpha) = 0
  if (arma::all(x.col(0) == 1.0)) {
    alpha -= arma::mean(alpha);  // subtract mean to enforce sum(alpha) = 0
  }
  res = y -  (z * alpha) -  (x * beta);
  rho = rho_als(res,tau);
  eta = accu(rho);
  return(log(eta));
}

//' @name Loss lasso expectile regression with fixed effects
//' 
//' @description This function returns the core of lasso expectile regression with fixed effects to be minimized   
//'  
//' @param theta initial values
//' @param x design matrix
//' @param y vector output
//' @param z incident matrix
//' @param tau percentile
//' @param n N sample size
//' @param d columns of x
//' @param mm n columns of z  
//' @param lambda constriction parameter
//' 
//' @return eta Numeric, sum of lasso expectile regression with fixed effects
//' @keywords internal
//' @noRd
// [[Rcpp::export]]
double loss_erlasso(arma::vec theta,arma::mat x,arma::vec y,arma::mat z,double tau,int n,int d,int mm, double lambda){
  double eta;
  arma::vec beta(d);
  arma::vec alpha(mm);
  arma::vec res(n);
  arma::vec rho(n);
  beta = fill_v(1,d, theta); 
  alpha = fill_v(d+1,d+mm, theta); 
  // If first column of x is all 1 → enforce sum(alpha) = 0
  if (arma::all(x.col(0) == 1.0)) {
    alpha -= arma::mean(alpha);  // subtract mean to enforce sum(alpha) = 0
  }
  res = y -  (z * alpha) -  (x * beta);   
  rho = rho_als(res,tau);
  eta = accu(rho)/n + lambda * accu(abs(alpha));
  return(log(eta));
}

