% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/spec_watson_targets.R
\name{spec_watson_targets}
\alias{spec_watson_targets}
\title{Specify targets following Watson \emph{et al.} (2010)}
\usage{
spec_watson_targets(
  rare_area_threshold = 10000,
  rare_relative_target = 1,
  rare_area_target = 1000,
  common_relative_target = 0.1,
  cap_area_target = 1e+06,
  area_units = "km^2"
)
}
\arguments{
\item{rare_area_threshold}{\code{numeric} value indicating the threshold
area for rare identifying rare features.
Defaults to 10000 (i.e., 10000 km^2).}

\item{rare_relative_target}{\code{numeric} value denoting the
relative target for features with a spatial distribution
that is smaller than \code{rare_area_threshold}.
Note that this value must be a proportion between 0 and 1.
Defaults to 1 (i.e., 100\%).}

\item{rare_area_target}{\code{numeric} value indicating the
area-based target for features with a spatial distribution
that is smaller than \code{rare_area_threshold}.
Defaults to 1000 (i.e., 1000 km^2).}

\item{common_relative_target}{\code{numeric} value denoting the
relative target for features with a spatial distribution
that is greater than \code{common_area_threshold}.
Defaults to 0.1 (i.e., 10\%).
Since this default value is based on historical levels of global protected
area coverage, it may be appropriate to set this value based on current
levels of protected area coverage (e.g., 17.6\% for terrestrial and 8.4\% for
marine systems globally; UNEP-WCMC and IUCN 2025).}

\item{cap_area_target}{\code{numeric} value denoting the area-based target cap.
To avoid setting a target cap, a missing (\code{NA}) value can be specified.
Defaults to 1000000
(i.e., 1,000,000 \ifelse{html}{\out{km<sup>2</sup>}}{\eqn{km^2}}).}

\item{area_units}{\code{character} value denoting the unit of measurement
for the area-based arguments.
Defaults to \code{"km^2"}
(i.e., \ifelse{html}{\out{km<sup>2</sup>}}{\eqn{km^2}}).}
}
\value{
An object (\code{\linkS4class{TargetMethod}}) for specifying targets that
can be used with \code{\link[=add_auto_targets]{add_auto_targets()}} and \code{\link[=add_group_targets]{add_group_targets()}}
to add targets to a \code{\link[=problem]{problem()}}.
}
\description{
Specify targets based on the methodology outlined by
Watson \emph{et al.} (2010).
Briefly, this method involves setting targets thresholds as a percentage
based on whether or not a feature is considered rare.
To help prevent widespread features from obscuring priorities,
targets are capped following Butchart \emph{et al.} (2015).
This method was designed for species protection at national-scales.
Note that this function is designed to be used with \code{\link[=add_auto_targets]{add_auto_targets()}}
and \code{\link[=add_group_targets]{add_group_targets()}}.
}
\details{
This target setting method was designed to protect species in
a national-scale prioritization (Watson \emph{et al.} 2010).
Although similar methods have also been successfully to national-scale
prioritizations (e.g., Kark \emph{et al.} 2009),
it may fail to identify meaningful priorities for
prioritizations conducted at smaller geographic scales
(e.g., local or county-level scales).
For example, if this target setting method is applied to
such geographic scales, then the resulting prioritizations
may select an overly large percentage of the study area,
or be biased towards over-representing common and widespread species.
This is because the thresholds
(i.e., \code{rare_area_threshold} and \code{cap_area_threshold})
were originally developed based on criteria for national-scales.
As such, if you working at the sub-national scale, it is recommended to set
thresholds based on that criteria are appropriate to the spatial extent
of the planning region.
Please note that this function is provided as convenient method to
set targets for problems with a single management zone, and cannot
be used for those with multiple management zones.
}
\section{Mathematical formulation}{

This method involves setting target thresholds based on the spatial
extent of the features.
By default, this method identifies rare features as those with a
spatial distribution smaller than 10,000
\ifelse{html}{\out{km<sup>2</sup>}}{\eqn{km^2}}
(per \code{rare_area_threshold} and \code{area_units})
and common features as those with a larger spatial distribution.
Given this, rare features are assigned target threshold of 100\%
(per \code{rare_relative_target})
or 1,000 \ifelse{html}{\out{km<sup>2</sup>}}{\eqn{km^2}}
(per \code{rare_area_threshold})
(whichever of these two values is smaller), and
common features are assigned a target threshold of 10\%
(per \code{common_relative_target}).
Additionally, following Butchart \emph{et al.} (2015), a cap of 1,000,000
\ifelse{html}{\out{km<sup>2</sup>}}{\eqn{km^2}} is applied to target
thresholds  (per \code{cap_area_threshold} and \code{area_units}).
}

\section{Data calculations}{

This function involves calculating targets based on the spatial extent
of the features in \code{x}.
Although it can be readily applied to \code{\link[=problem]{problem()}} objects that
have the feature data provided as a \code{\link[terra:rast]{terra::rast()}} object,
you will need to specify the spatial units for the features
when initializing the \code{\link[=problem]{problem()}} objects if the feature data
are provided in a different format. In particular, if the feature
data are provided as a \code{data.frame} or \code{character} vector,
then you will need to specify an argument to \code{feature_units} when
using the \code{\link[=problem]{problem()}} function.
See the Examples section of the documentation for \code{\link[=add_auto_targets]{add_auto_targets()}}
for a demonstration of specifying the spatial units for features.
}

\examples{
\dontrun{
# set seed for reproducibility
set.seed(500)

# load data
sim_complex_pu_raster <- get_sim_complex_pu_raster()
sim_complex_features <- get_sim_complex_features()

# create problem with Watson et al. (2010) targets
p1 <-
  problem(sim_complex_pu_raster, sim_complex_features) \%>\%
  add_min_set_objective() \%>\%
  add_auto_targets(method = spec_watson_targets()) \%>\%
  add_binary_decisions() \%>\%
  add_default_solver(verbose = FALSE)

# solve problem
s1 <- solve(p1)

# plot solution
plot(s1, main = "solution", axes = FALSE)
}
}
\references{
Butchart SHM, Clarke M, Smith RJ, Sykes RE, Scharlemann JPW, Harfoot M,
Buchanan GM, Angulo A, Balmford A, Bertzky B, Brooks TM, Carpenter KE,
Comeros‐Raynal MT, Cornell J, Ficetola GF, Fishpool LDC, Fuller RA,
Geldmann J, Harwell H, Hilton‐Taylor C, Hoffmann M, Joolia A, Joppa L,
Kingston N, May I, Milam A, Polidoro B, Ralph G, Richman N, Rondinini C,
Segan DB, Skolnik B, Spalding MD, Stuart SN, Symes A, Taylor J, Visconti P,
Watson JEM, Wood L, Burgess ND (2015) Shortfalls and solutions for meeting
national and global conservation area targets. \emph{Conservation Letters},
8: 329--337.

Kark S, Levin N, Grantham HS, Possingham HP (2009) Between-country
collaboration and consideration of costs increase conservation planning
efficiency in the Mediterranean Basin.
\emph{Proceedings of the National Academy of Sciences}, 106: 15368--15373.

UNEP-WCMC and IUCN (2025) Protected Planet Report 2024.
Cambridge, UK: UNEP-WCMC and IUCN. Available at <www.protectedplanet.net>.

Watson JEM, Evans MC, Carwardine J, Fuller RA, Joseph LN, Segan DB,
Taylor MFJ, Fensham RJ, Possingham HP (2010) The capacity of Australia's
protected-area system to represent threatened species.
\emph{Conservation Biology},25: 324--332.
}
\seealso{
Other target setting methods:
\code{\link{spec_absolute_targets}()},
\code{\link{spec_area_targets}()},
\code{\link{spec_duran_targets}()},
\code{\link{spec_interp_absolute_targets}()},
\code{\link{spec_interp_area_targets}()},
\code{\link{spec_jung_targets}()},
\code{\link{spec_max_targets}()},
\code{\link{spec_min_targets}()},
\code{\link{spec_polak_targets}()},
\code{\link{spec_pop_size_targets}()},
\code{\link{spec_relative_targets}()},
\code{\link{spec_rl_ecosystem_targets}()},
\code{\link{spec_rl_species_targets}()},
\code{\link{spec_rodrigues_targets}()},
\code{\link{spec_rule_targets}()},
\code{\link{spec_ward_targets}()},
\code{\link{spec_wilson_targets}()}
}
\concept{methods}
