% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/commensurate.t2e.R
\name{commensurate.t2e}
\alias{commensurate.t2e}
\title{Bayesian analysis with commensurate prior for time-to-event outcome}
\usage{
commensurate.t2e(
  formula, data, method.borrow,
  chains=2, iter=4000, warmup=floor(iter/2), thin=1,
  alternative="greater", sig.level=0.025,
  seed=sample.int(.Machine$integer.max,1))
}
\arguments{
\item{formula}{Object of class \code{formula}, which is a symbolic
description of the model to be fitted. The dependent variable must be an
object of class \code{Surv}. The explanatory variables only include
covariates of interest, which must be specified in the form of linear
combination.}

\item{data}{Data frame, which must have variables named \code{study} for
study indicator (0 for external control, and 1 for current trial) and
\code{treat} for treatment indicator (0 for concurrent and external control,
and 1 for treatment).}

\item{method.borrow}{List of information borrowing method. \code{"noborrow"}
uses the concurrent data only. \code{"fullborrow"} uses the external control
data without discounting. \code{"cauchy"} uses the commensurate prior to
dynamically borrow the external control data, and the commensurability
parameter is assumed to follow a half-Cauchy distribution. \code{"normal"}
uses the commensurate prior to dynamically borrow the external control data,
and the commensurability parameter is assumed to follow a half-normal
distribution. \code{"cauchy"} and \code{"normal"} require to specify the
scale parameter \code{scale} of half-Cauchy and half-normal distribution
respectively.}

\item{chains}{Number of Markov chains in MCMC sampling. The default value is
\code{chains=2}.}

\item{iter}{Number of iterations for each chain (including warmup) in MCMC
sampling. The default value is \code{iter=4000}.}

\item{warmup}{Number of warmup (burnin) iterations per chain in MCMC
sampling. The default value is \code{warmup=floor(iter/2)}.}

\item{thin}{Period for saving samples in MCMC sampling. The default value
is \code{thin=1}.}

\item{alternative}{Alternative hypothesis to be tested ("greater" or "less").
The default value is \code{alternative="greater"}.}

\item{sig.level}{Significance level. The default value is
\code{sig.level=0.025}.}

\item{seed}{Setting a seed for MCMC sampling.}
}
\value{
The \code{commensurate.t2e} returns a list containing the following objects:
\item{reject}{Data frame containing results of Bayesian one-sided hypothesis
testing (whether or not the posterior probability that the log hazard ratio
is greater or less than 0 exceeds 1 minus significance level): \code{TRUE}
when significant, otherwise \code{FALSE}.}
\item{theta}{Data frame containing posterior mean, median, and sd of log
hazard ratio.}
\item{stan.obj}{Stanfit object.}
}
\description{
Bayesian analysis for time-to-event outcome is implemented via MCMC, where a
commensurate prior is used for incorporating data from external controls.
No borrowing and full borrowing are also applicable.
}
\details{
The time to event outcome is assumed to follow a Weibull
distribution. Given more than one covariates, a Weibull proportional
hazards model is built and its Bayesian estimation is performed via
MCMC. Commensurate prior is used to dynamically discount the information to
be borrowed from external control based on the similarity between the current
trial and external controls, where the commensurability parameter determines
the extent of borrowing. The commensurability parameter is assumed to follow
a half-cauchy or a half-normal distribution, and its scale parameter needs to
be carefully specified. No borrowing approach is to perform the analysis
without incorporating the external controls. Full borrowing approach is just
to pool the concurrent and external controls, which is used as a comparator
in the analysis.
}
\examples{
n.CT       <- 100
n.CC       <- 50
nevent.C   <- 100
n.ECp      <- 200
nevent.ECp <- 180
accrual    <- 16

out.mevent.CT <- 6
out.mevent.CC <- 6
driftHR       <- 1

cov.C <- list(list(dist="norm",mean=0,sd=1,lab="cov1"),
              list(dist="binom",prob=0.4,lab="cov2"))

cov.cor.C <- rbind(c(  1,0.1),
                   c(0.1,  1))

cov.EC <- list(list(dist="norm",mean=0,sd=1,lab="cov1"),
               list(dist="binom",prob=0.4,lab="cov2"))

cov.cor.EC <- rbind(c(  1,0.1),
                    c(0.1,  1))

cov.effect <- c(0.9,0.9)

indata <- trial.simulation.t2e(
  n.CT=n.CT, n.CC=n.CC, nevent.C=nevent.C,
  n.ECp=n.ECp, nevent.ECp=nevent.ECp, accrual=accrual,
  out.mevent.CT, out.mevent.CC, driftHR,
  cov.C=cov.C, cov.cor.C=cov.cor.C,
  cov.EC=cov.EC, cov.cor.EC=cov.cor.EC, cov.effect=cov.effect)

n.EC <- 50

method.whomatch <- "conc.treat"
method.matching <- "optimal"
method.psorder  <- NULL

out.psmatch <- psmatch(
  study~cov1+cov2, data=indata, n.EC=n.EC,
  method.whomatch=method.whomatch, method.matching=method.matching,
  method.psorder=method.psorder)

indata.match <- rbind(indata[indata$study==1,],indata[out.psmatch$subjid.EC,])

method.borrow <- list(list(prior="cauchy",scale=2.0),
                      list(prior="normal",scale=0.5))

commensurate.t2e(
  survival::Surv(time,status)~cov1+cov2,data=indata.match,
  method.borrow=method.borrow,chains=1,iter=100)
}
\references{
Hobbs BP, Carlin BP, Mandrekar SJ, Sargent DJ. Hierarchical commensurate and
power prior models for adaptive incorporation of historical information in
clinical trials. \emph{Biometrics} 2011; 67:1047-1056.

Hobbs BP, Sargent DJ, Carlin BP. Commensurate priors for incorporating
historical information in clinical trials using general and generalized
linear models. \emph{Bayesian Analysis} 2012; 7:639-674.
}
