% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/simulate_sensor_mode.R
\name{simulate_sensor_mode}
\alias{simulate_sensor_mode}
\title{Simulate Atmospheric Concentration at Sensor Locations}
\usage{
simulate_sensor_mode(start_time, end_time, source_coords, emission_rate, wind_data,
  sensor_coords, sim_dt, puff_dt, output_dt, puff_duration, ws, wd)
}
\arguments{
\item{start_time}{POSIXct. Start time of the simulation.}

\item{end_time}{POSIXct. End time of the simulation.}

\item{source_coords}{Numeric vector or matrix. Source coordinates in meters (x, y, z). If a single source,
pass as a vector. For multiple sources, use a matrix where each row is a source.}

\item{emission_rate}{Numeric. Emission rate from each source in kg/hr. Applied uniformly to all sources.}

\item{wind_data}{Data frame. Must contain either columns `wind_u` and `wind_v` (wind vector components in x/y directions)
or columns representing wind speed and direction, declared as `ws` and `wd`.}

\item{sensor_coords}{Numeric matrix. Sensor coordinates in meters (x, y, z); one row per sensor.}

\item{sim_dt}{Integer. Simulation time step in seconds (default: 1). Controls how often the simulation
updates concentrations.}

\item{puff_dt}{Integer. Puff emission interval in seconds (default: 1). Controls how often a new puff is emitted.}

\item{output_dt}{Integer. Desired resolution in seconds for output concentrations.}

\item{puff_duration}{Numeric. Lifetime of each puff in seconds (default: 1200). Puffs are removed after this time.}

\item{ws}{Optional character. If your `wind_data` uses wind speed and direction instead of `wind_u`/`wind_v`, supply the name of the wind speed column here (e.g., `"ws"` or `"wind_speed"`).}

\item{wd}{Optional character. If your `wind_data` uses wind direction in degrees, supply the name of the wind direction column here (e.g., `"wd"` or `"wind_direction"`).}
}
\value{
A data frame with aggregated sensor concentrations across time.
  Rows represent time intervals (`output_dt`), columns represent sensors (`Sensor_1`, `Sensor_2`, etc.).
}
\description{
This function simulates atmospheric methane concentrations at one or more sensor locations
using a Gaussian puff forward model. It supports one or multiple emission sources and assumes
each puff maintains a constant wind speed and direction throughout its lifetime. The function
accounts for puff dispersion based on wind conditions and atmospheric stability class.
}
\details{
- Each source emits puffs at regular intervals (`puff_dt`) with a fixed mass based on `emission_rate`.
- Wind speed and direction at the time of puff emission are used to advect the puff and determine dispersion.
- Puff position is analytically computed at each timestep based on wind, without tracking in-between steps.
- Puff dispersion is computed using stability-class-based sigma values from a fast lookup.
- Total sensor concentration is the sum of all active puff contributions at each timestep.
- Concentrations are aggregated into intervals matching `output_dt` before being returned.
}
\note{
All time parameters should be positive, with `puff_dt > sim_dt` and `out_dt > sim_dt`. Also, `puff_dt` should be a positive integer multiple of `sim_dt`, i.e. `puff_dt = n*sim_dt` for some positive integer `n`. This prevents the code having to interpolate the concentration values in time, although it is likely that this constraint could be avoided.
}
\examples{
set.seed(123)
sim_dt <- 10
puff_dt <- 10
output_dt <- 60
start_time <- as.POSIXct("2024-01-01 12:00:00")
end_time <- as.POSIXct("2024-01-01 13:00:00")

source_coords <- matrix(c(0, 0, 2.5), ncol = 3, byrow = TRUE)

sensor_coords <- matrix(c(
   -20, 0, 2.0,
     0, -20, 2.0,
    20, 0, 2.0,
     0, 20, 2.0,
    10, 10, 2.0
), ncol = 3, byrow = TRUE)

wind_data <- data.frame(
   wind_u = runif(3601, min = -3, max = 0.7),
   wind_v = runif(3601, min = -3, max = 1.5)
)

out <- simulate_sensor_mode(
   start_time = start_time,
   end_time = end_time,
   source_coords = source_coords,
   emission_rate = 3.5,
   wind_data = wind_data,
   sensor_coords = sensor_coords,
   sim_dt = sim_dt,
   puff_dt = puff_dt,
   output_dt = output_dt,
   puff_duration = 1200
)
}
\references{
Jia, M., Fish, R., Daniels, W., Sprinkle, B. and Hammerling, D. (2024) <doi:10.26434/chemrxiv-2023-hc95q-v3>
}
