#' Formats raw price data.
#'
#' Tidies raw prices and returns a tidied, usable data frame. 
#' Raw data should be structured identically to that produced
#' by get_prices(), as this function depends on that structure.
#' 
#' \code{tidy_prices} produces a data frame that is 'tidy' or 
#' more readily readable by a user and usable by other functions
#' within this package.
#' 
#' @param x Raw daily data, as produced by get_prices()
#' 
#' @return Returns a data set that's been 'tidied' up for use 
#' by other functions in this package.
#' 
#' @seealso \code{\link{get_prices}}
#' 
#' @examples
#' my_companies <- data.frame(ticker=c('GOOG', 'IBM'))
#' raw_price_data <- get_prices(my_companies)
#' prices <- tidy_prices(raw_price_data)
#' 
#' @importFrom dplyr bind_rows %>%
#' @return data.frame of cleaned prices
#' @export

tidy_prices <- function(x) {
  colNames <- names(x)
  closing <- x[, grep(".Close", colNames)]
  pret <- x[, grep("pret", colNames)]
  
  #' @describeIn pricecombine combines relevant columns from the original price
  #' data set.
  pricecombine <- function(colReference) {
    dates <- rownames(closing)
    ticker <- gsub(".Close", "", names(closing)[colReference])
    ticker <- rep(ticker, length(dates))
    cbind(data.frame(ticker, date = dates, pret = pret[, colReference], close = closing[, colReference]))
  }
  
  data <- lapply(1:ncol(closing), pricecombine)
  
  #' Convert certain columns into character in order to
  #' bypass the warning generated by dplyr::bind_rows()
  convertcols <- function(dataframe) {
    dataframe[,1] <- as.character(dataframe[,1])  # Convert ticker to character
    dataframe[,2] <- as.character(dataframe[,2])  # Convert date to character.
    dataframe
  }
  
  data <- lapply(data, convertcols)
  
  ## Aggregate data sets and convert pret and close columns to type numeric.
  result <- dplyr::bind_rows(data)
  result$pret <- as.numeric(result$pret)
  result$close <- as.numeric(result$close)
  result
} 
