# quiltr <img src="man/figures/quiltrhex.png" width="160px" align="right" />

Qualtrics imports for labelling text using R

## Installation

You can install the package with:

``` r
devtools::install_github("cjbarrie/quiltr")
```

To use the package, it first needs to be loaded with:

```r

library(quiltr)

```

The <tt>quiltr</tt> package is designed for generating Qualtrics importable forms for data labelling, in particular text labelling. 

There are two workhorse functions, which are `quilt_form_data()` and `quilt_form`. The first of these takes a dataset of text for labelling and structures it appropriately for writing the Qualtrics data labelling form. The second takes the `data.frame` produced by `quilt_form_data()` and generates a .txt file that is importable into Qualtrics.

In the package are several bundled datasets, which are useful for testing and understanding how the package functions. These are named: `lipsum_text`, `arabiya_text`, and `qdat`.


| Dataset | Description |
|----------|:-------------:|
|lipsum_text | Dataset of (pseudo) random lipsum text generated using the <tt>stringi</tt> package
|arabiya_text | Dataset of Arabic text data sampled from [https://data.mendeley.com/datasets/57zpx667y9/2](https://data.mendeley.com/datasets/57zpx667y9/2)
|qdat | Dataset containing one hundred paragraphs of lipsum text with columns for text (prompt), response_type and ID as generated by `quilt_form_data`

We first load one of these datasets into memory:

```r

data(lipsum_text)

```

We can then call `quilt_form_data()`, which takes the following arguments:

```r

qdat <- quilt_form_data(prompt = "Label this text: ",
                     text = lipsum_text$text, response_type = "scale",
                     nlow = 1, nhigh = 10, addID = T)

```

The `prompt` argument is optional and gives the user the ability to add a prompt before the text for labelling. The second argument is the text to be labelled. Here, we are specifying our dummy `lipsum_text` data and, specifically, the column `lipsum_text$text`. The `response_type` parameter has several options, including "scale" for an integer scale. For this, we then need to specify an upper and lower bound with the `nlow` and `nhigh` parameters. The other options here are "yesno" for a binary yes or no response, and "options," which then requires the additional `options` parameter.

Once we have our data in the appropriate format, we can then generate our Qualtrics importable form with `quilt_form()` as follows:

```r

quilt_form(input_data = qdat,
            page_break_every = 1, question_type = "multianswer",
            filename = "quilted_survey.txt")

```

## Importing to Qualtrics

After we've generated the file in the appropriate format, we can import to Qualtrics. First we go to the survey in question, here named "import_test." We then click on tools and select the "Import/Export" > "Import survey." 

![](man/figures/quiltr1.gif)

Qualtrics will then prompt us to select the relevant .txt file, which we previously generated with `quilt_form`. We select here "quilted_survey.txt" and press "Import." We have now successfully imported our survey into Qualtrics!

## Non-Latin language support

The package works equally well with Latin languages as it does with non-Latin languages, including languages written from right-left such as Arabic. We can see this below where we use a mixture of Arabic text and English prompts.

```r

qdat <- quilt_form_data(prompt = "Label this text: ",
                     text = arabiya_text$text, response_type = "scale",
                     nlow = 1, nhigh = 10, addID = T)
```

```r

quilt_form(input_data = qdat,
            page_break_every = 1, question_type = "multianswer",
            filename = "quilted_survey.txt")

```

After importing into Qualtrics, we see that the prompt and Arabic text is correctly displayed:

![](man/figures/quiltr3.png)

## Question types

There are several options for question types for text labelling tasks. The first group allow the user to select from a set of options (e.g., "Yes/No/Maybe"), and allow respondents to select either one or multiple answers depending on whether we specify e.g., "select" or "multiselect" as our `question_type` argument. 

The table below details the different question types, and how they are rendered in Qualtrics.

| `question_type` | Description | Example |
|----------|:------------- | ----
| "dropdown" | Dropdown select box | ![](man/figures/dropdown.png) 
| "select" | Vertical list select | ![](man/figures/select.png)
| "multiselect" | Vertical list multiselect | ![](man/figures/multiselect.png)
| "singleanswer" | Horizontal list select | ![](man/figures/singleanswer.png)
| "multianswer" | Horizontal list multiselect | ![](man/figures/multianswer.png)
| "rankorder" | Vertical list of draggable ranked options | ![](man/figures/rankorder.png)
| "singleline" | Single line text entry | ![](man/figures/singleline.png)
| "essay" | Multiline text entry | ![](man/figures/essay.png)

## Acknowledgments

The <tt>quiltr</tt> package has been adapted from the <tt>qualtricsR</tt> [here](https://github.com/saberry/qualtricsR). This package was designed for generating bespoke surveys in a format appropriate for importing into Qualtrics but lacked support for many of the requirements of large text labelling inputs.
