% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/boatman_2017.R
\name{boatman_2017}
\alias{boatman_2017}
\title{Boatman model for fitting thermal performance curves}
\usage{
boatman_2017(temp, rmax, tmin, tmax, a, b)
}
\arguments{
\item{temp}{temperature in degrees centigrade}

\item{rmax}{the rate at optimum temperature}

\item{tmin}{low temperature (ºC) at which rates become negative}

\item{tmax}{high temperature (ºC) at which rates become negative}

\item{a}{shape parameter to adjust the skewness of the curve}

\item{b}{shape  parameter to adjust the kurtosis of the curve}
}
\value{
a numeric vector of rate values based on the temperatures and parameter values provided to the function
}
\description{
Boatman model for fitting thermal performance curves
}
\details{
Equation:
\deqn{rate= r_{max} \cdot \left(sin\bigg(\pi\left(\frac{temp-t_{min}}{t_{max} - t_{min}}\right)^{a}\bigg)\right)^{b}}{%
rate = rmax.(sin(pi.((temp - tmin)/(tmax - tmin))^a))^b}

Start values in \code{get_start_vals} are derived from the data or sensible values from the literature.

Limits in \code{get_lower_lims} and \code{get_upper_lims} are derived from the data or based extreme values that are unlikely to occur in ecological settings.
}
\note{
Generally we found this model easy to fit.
}
\examples{
# load in ggplot
library(ggplot2)

# subset for the first TPC curve
data('chlorella_tpc')
d <- subset(chlorella_tpc, curve_id == 1)

# get start values and fit model
start_vals <- get_start_vals(d$temp, d$rate, model_name = 'boatman_2017')
# fit model
mod <- nls.multstart::nls_multstart(rate~boatman_2017(temp = temp, rmax, tmin, tmax, a, b),
data = d,
iter = c(4,4,4,4,4),
start_lower = start_vals - 10,
start_upper = start_vals + 10,
lower = get_lower_lims(d$temp, d$rate, model_name = 'boatman_2017'),
upper = get_upper_lims(d$temp, d$rate, model_name = 'boatman_2017'),
supp_errors = 'Y',
convergence_count = FALSE)

# look at model fit
summary(mod)

# get predictions
preds <- data.frame(temp = seq(min(d$temp), max(d$temp), length.out = 100))
preds <- broom::augment(mod, newdata = preds)

# plot
ggplot(preds) +
geom_point(aes(temp, rate), d) +
geom_line(aes(temp, .fitted), col = 'blue') +
theme_bw()
}
\references{
Boatman, T. G., Lawson, T., & Geider, R. J. A key marine diazotroph in a changing ocean: The interacting effects of temperature, CO2 and light on the growth of Trichodesmium erythraeum IMS101. PLoS ONE, 12, e0168796 (2017)
}
