rcd3 <- function(n_factors = 3, show_efficiency = TRUE, show_replications = TRUE, verbose = TRUE) {
  old_opts <- options()
  on.exit(options(old_opts), add = TRUE)
  options(max.print = 1500)

  get_principal_blocks <- function(n) {
    factors <- LETTERS[1:n]
    all_runs <- expand.grid(replicate(n, 0:2, simplify = FALSE))
    all_runs <- as.matrix(all_runs)
    zero_run <- rep(0, n)
    blocks <- list()
    for (i in 1:(nrow(all_runs) - 1)) {
      for (j in (i + 1):nrow(all_runs)) {
        r1 <- all_runs[i, ]; r2 <- all_runs[j, ]
        block <- rbind(zero_run, r1, r2)
        if (all(apply(block, 2, function(col) length(unique(col)) == 3))) {
          df <- as.data.frame(block)
          colnames(df) <- factors
          blocks[[length(blocks) + 1]] <- df
        }
      }
    }
    blocks
  }

  label_effect <- function(vec, factors) {
    label <- ""
    superscripts <- c("0" = "", "1" = "", "2" = "\u00B2")
    for (i in seq_along(vec)) {
      if (vec[i] != 0) label <- paste0(label, factors[i], superscripts[as.character(vec[i])])
    }
    if (label == "") "I" else label
  }

  generate_effects <- function(n) {
    total <- 3^n
    effects <- matrix(NA, nrow = total - 1, ncol = n)
    row <- 1
    for (i in 1:(total - 1)) {
      vec <- integer(n); num <- i
      for (j in 1:n) { vec[j] <- num %% 3; num <- num %/% 3 }
      effects[row, ] <- vec
      row <- row + 1
    }
    effects
  }

  is_confounded <- function(block, effect) {
    res <- (as.matrix(block) %*% effect) %% 3
    length(unique(as.vector(res))) == 1
  }

  normalize_effect <- function(eff) {
    first_nonzero <- which(eff != 0)[1]
    if (length(first_nonzero) == 0) return(eff)
    if (eff[first_nonzero] == 2) eff <- (3 - eff) %% 3
    eff
  }

  classify_effect <- function(eff) {
    nonzeros <- sum(eff != 0)
    if (nonzeros == 1) "Main" else if (nonzeros == 2) "2FI" else "Other"
  }

  compute_efficiency_factors <- function(blocks, normalized_effects, effect_labels) {
    num_blocks <- length(blocks)
    estimable_count <- numeric(length(effect_labels))
    for (blk in blocks) {
      confounded <- apply(normalized_effects, 1, function(eff) is_confounded(blk, eff))
      estimable_count <- estimable_count + as.numeric(!confounded)
    }
    eff_factor <- round(estimable_count / num_blocks, 2)
    names(eff_factor) <- effect_labels
    eff_factor
  }

  reorder_block_by_effect <- function(block, effect) {
    vals <- (as.matrix(block) %*% effect) %% 3
    target <- 0:2
    new_order <- sapply(target, function(v) which(vals == v)[1])
    block[new_order, , drop = FALSE]
  }

  reorder_block_by_index <- function(block, index) {
    patterns <- list(c(1, 2, 3), c(2, 3, 1), c(3, 1, 2))
    pattern <- patterns[[((index - 1) %% 3) + 1]]
    block[pattern, , drop = FALSE]
  }

  generate_complete_blocks <- function(pb) {
    k <- ncol(pb)
    derived_blocks <- list()
    for (i in 0:(3^(k - 1) - 1)) {
      shift <- integer(k); num <- i
      for (j in 2:k) { shift[j] <- num %% 3; num <- num %/% 3 }
      derived <- (pb + matrix(rep(shift, each = 3), ncol = k, byrow = FALSE)) %% 3
      derived_blocks[[length(derived_blocks) + 1]] <- derived
    }
    derived_blocks
  }

  find_valid_higher_order_effect <- function(blocks, effects, n_factors) {
    full_order <- effects[apply(effects, 1, function(e) sum(e != 0) == n_factors), , drop = FALSE]
    for (i in 1:nrow(full_order)) {
      eff <- full_order[i, ]; ok <- TRUE
      for (blk in blocks) {
        vals <- (as.matrix(blk) %*% eff) %% 3
        if (length(unique(vals)) < 3) { ok <- FALSE; break }
      }
      if (ok) return(eff)
    }
    NULL
  }

  factors <- LETTERS[1:n_factors]
  all_blocks <- get_principal_blocks(n_factors)
  all_effects <- generate_effects(n_factors)
  classified <- apply(all_effects, 1, classify_effect)
  target_effects <- all_effects[classified %in% c("Main", "2FI"), , drop = FALSE]
  normalized_effects <- unique(t(apply(target_effects, 1, normalize_effect)))
  valid_rows <- apply(normalized_effects, 1, function(eff) {
    first <- which(eff != 0)[1]
    !is.na(first) && eff[first] == 1
  })
  normalized_effects <- normalized_effects[valid_rows, , drop = FALSE]
  effect_labels <- apply(normalized_effects, 1, function(eff) label_effect(eff, factors))
  block_estimable <- lapply(all_blocks, function(blk) {
    apply(normalized_effects, 1, function(eff) !is_confounded(blk, eff))
  })

  covered <- which(block_estimable[[1]])
  chosen_blocks <- list(as.matrix(all_blocks[[1]])); used_indices <- 1
  while (length(covered) < nrow(normalized_effects)) {
    best_block <- NULL; max_new <- 0
    for (i in seq_along(block_estimable)) {
      if (i %in% used_indices) next
      new_cov <- which(block_estimable[[i]])[!(which(block_estimable[[i]]) %in% covered)]
      if (length(new_cov) > max_new) { max_new <- length(new_cov); best_block <- i }
    }
    if (is.null(best_block)) break
    covered <- union(covered, which(block_estimable[[best_block]]))
    chosen_blocks[[length(chosen_blocks) + 1]] <- as.matrix(all_blocks[[best_block]])
    used_indices <- c(used_indices, best_block)
  }

  if (n_factors == 2) {
    if (length(all_blocks) < 2) stop("Not enough principal blocks available for n_factors = 2.")
    chosen_blocks <- list(as.matrix(all_blocks[[2]]))
  }

  if (verbose) message("Factors: ", paste(factors, collapse = " "))

  num_rep <- length(chosen_blocks)
  if (n_factors != 2 && num_rep %% 3 != 0) {
    extra_needed <- 3 - (num_rep %% 3)
    if (verbose)
      message("Number of replications (", num_rep, ") not divisible by 3; adding ", extra_needed, " extra replication(s).")
    for (i in 1:extra_needed) chosen_blocks[[length(chosen_blocks) + 1]] <- chosen_blocks[[i]]
    num_rep <- length(chosen_blocks)
  } else if (n_factors == 2) num_rep <- 1

  if (verbose)
    message("Final number of principal blocks selected: ", num_rep)

  candidate_eff <- NULL
  if (n_factors > 2) {
    candidate_eff <- find_valid_higher_order_effect(chosen_blocks, all_effects, n_factors)
    if (!is.null(candidate_eff)) {
      if (verbose)
        message("Higher-order effect confounded: ", label_effect(candidate_eff, factors))
      chosen_blocks <- lapply(chosen_blocks, function(blk) reorder_block_by_effect(blk, candidate_eff))
    } else if (verbose) {
      message("No suitable higher-order effect found. Sacrificing highest-order 2FI.")
    }
  } else if (verbose) {
    message("For n_factors = 2: Higher-order effect to confound: A^2B.")
  }

  for (i in seq_along(chosen_blocks)) {
    block <- reorder_block_by_index(chosen_blocks[[i]], i)
    colnames(block) <- factors
    chosen_blocks[[i]] <- block
    if (verbose) {
      message(sprintf("Principal Block %d:", i))
      print(as.data.frame(block), row.names = FALSE)
    }
  }

  eff_factors <- NULL
  if (show_efficiency) {
    if (n_factors == 2) {
      types <- apply(normalized_effects, 1, classify_effect)
      ef <- ifelse(types == "Main", 1, 0)
      names(ef) <- effect_labels
      eff_factors <- ef
    } else {
      eff_factors <- compute_efficiency_factors(chosen_blocks, normalized_effects, effect_labels)
    }
    if (verbose) {
      message("\nEfficiency Factors:")
      print(eff_factors)
    }
  }

  if (n_factors == 2) {
    base_pb <- chosen_blocks[[1]]
    a2b_eff <- c(2,1)
    rep1_blocks <- lapply(generate_complete_blocks(base_pb), function(derived){
      reorder_block_by_effect(derived, a2b_eff)
    })
    cycle_rows <- function(block, shift){
      n <- nrow(block)
      new_order <- ((seq_len(n)-1+shift) %% n) + 1
      block[new_order,,drop=FALSE]
    }
    rep_full_blocks <- list(
      lapply(rep1_blocks, cycle_rows, shift=0),
      lapply(rep1_blocks, cycle_rows, shift=1),
      lapply(rep1_blocks, cycle_rows, shift=2)
    )
  } else {
    rep_full_blocks <- lapply(seq_along(chosen_blocks), function(pb_index){
      pb <- chosen_blocks[[pb_index]]
      full_blocks <- lapply(generate_complete_blocks(pb), function(derived){
        if(n_factors>2 && !is.null(candidate_eff)){
          derived <- reorder_block_by_effect(derived, candidate_eff)
        }
        derived <- reorder_block_by_index(derived, pb_index)
        derived
      })
      full_blocks
    })
  }

  if (show_replications && verbose) {
    for (pb_index in seq_along(rep_full_blocks)) {
      message(sprintf("\nReplication %d", pb_index))
      full_blocks <- rep_full_blocks[[pb_index]]
      n_blocks <- length(full_blocks)
      factor_names <- factors
      block_strings <- list()
      for (i in seq_along(full_blocks)) {
        block <- full_blocks[[i]]
        header <- sprintf("Col %d", i)
        rows <- apply(block,1,function(r) paste(r,collapse=" "))
        block_strings[[i]] <- c(header, paste(factor_names, collapse=" "), rows)
      }
      output_matrix <- block_strings[[1]]
      if(length(block_strings) > 1){
        for(i in 2:length(block_strings)){
          output_matrix <- cbind(output_matrix,"", block_strings[[i]])
        }
      }
      output_matrix <- cbind(c("", "", paste0("Row ",1:3)), output_matrix)
      output_df <- as.data.frame(output_matrix, stringsAsFactors=FALSE)
      colnames(output_df) <- rep("", ncol(output_df))
      print(output_df, row.names = FALSE, quote = FALSE, right = FALSE)
    }
  }

  invisible(list(
    factors = factors,
    normalized_effects = normalized_effects,
    effect_labels = effect_labels,
    chosen_principal_blocks = chosen_blocks,
    efficiency_factors = if(show_efficiency) eff_factors else NULL,
    replications = lapply(seq_along(rep_full_blocks), function(pb_index){
      lapply(rep_full_blocks[[pb_index]], function(block){
        colnames(block) <- factors
        block
      })
    })
  ))
}
