#' Check alternative splice acceptors for a splice acceptor SNV
#'
#' @param roi.aln.df A data frame of the detection window alignment
#' @param minus.strand A logical value indicating whether the pathogenic mutation on the minus strand
#' @param snv.flank.bp A non-negative integer specifying the length of flanking regions to be added to both ends of the splice acceptor for detecting SNV reversions
#' 
#' @return A list including original splice acceptor positions and alternative splice acceptor positions 
#' 
#' @examples
#' \dontrun{
#'     checkSpliceAcceptorSNV()
#' }
#'
#' @noRd
checkSpliceAcceptorSNV <- function(roi.aln.df, minus.strand, snv.flank.bp=10){
    
    ref.bases <- roi.aln.df$ref_pos[ !is.na(roi.aln.df$ref_pos) ]
    roi.start <- min(ref.bases)
    roi.end <- max(ref.bases)
    
    snv.base.ref <- roi.aln.df$ref_seq[ roi.aln.df$pathog_mut==1 ]
    snv.base <- roi.aln.df$ref_pos[ roi.aln.df$pathog_mut==1 ]
    ref.cds.bases <- roi.aln.df$ref_pos[ roi.aln.df$ref_cds==1 ]
    
    cryptic.acceptor <- NULL
    nWindow <- floor(snv.flank.bp/3)
    
    if(!minus.strand){
        
        if(snv.base.ref=="A"){
            snv.window.right <- snv.base + 1
        }else if(snv.base.ref=="G"){
            snv.window.right <- snv.base 
        }
        snv.window.mid <- snv.window.right - 1
        snv.window.left <- snv.window.right - 2
        
        real.acceptor <- c( snv.window.mid, snv.window.right )
        
        if( roi.start<=snv.window.mid & roi.end>=snv.window.right & nWindow>0 ){
            
            for(i in 1:nWindow){
                
                right.window <- snv.window.left:snv.window.right + 3*i  
                check.bases <- snv.window.mid:right.window[3]
                check.df <- roi.aln.df[ roi.aln.df$ref_pos %in% check.bases, ]
                if( nrow(check.df)==length(check.bases) & sum(check.df$cigar=="M")==length(check.bases) & sum(!is.na(check.df$ins))==0 ){
                    if( sum(!is.na(check.df$snv[check.df$ref_pos %in% c(right.window[2],right.window[3])]))>0 ){
                        if( paste(check.df$reads_seq[check.df$ref_pos %in% c(right.window[2],right.window[3])], collapse="")=="AG" ){
                            cryptic.acceptor <- c(right.window[2],right.window[3]) 
                        }
                    }
                }
                if( !is.null(cryptic.acceptor) ){ break }
                
                left.window <- snv.window.left:snv.window.right - 3*i
                check.bases <- left.window[2]:snv.window.right
                check.df <- roi.aln.df[ roi.aln.df$ref_pos %in% check.bases, ]
                if( nrow(check.df)==length(check.bases) & sum(check.df$cigar=="M")==length(check.bases) & sum(!is.na(check.df$ins))==0 ){
                    if( sum(!is.na(check.df$snv[check.df$ref_pos %in% c(left.window[2],left.window[3])]))>0 ){
                        if( paste(check.df$reads_seq[check.df$ref_pos %in% c(left.window[2],left.window[3])], collapse="")=="AG" ){
                            cryptic.acceptor <- c(left.window[2],left.window[3]) 
                        }      
                    }
                }
                if( !is.null(cryptic.acceptor) ){ break }
                
            }
        }
        
    }else{
        
        if(snv.base.ref=="C"){
            snv.window.left <- snv.base 
        }else if(snv.base.ref=="T"){
            snv.window.left <- snv.base - 1 
        }
        snv.window.mid <- snv.window.left + 1
        snv.window.right <- snv.window.left + 2
        
        real.acceptor <- c( snv.window.left, snv.window.mid )
        
        if( roi.start<=snv.window.left & roi.end>=snv.window.mid & nWindow>0 ){
        
            for(i in 1:nWindow){
                
                left.window <- snv.window.left:snv.window.right - 3*i
                check.bases <- left.window[1]:snv.window.mid
                check.df <- roi.aln.df[ roi.aln.df$ref_pos %in% check.bases, ]
                
                if( nrow(check.df)==length(check.bases) & sum(check.df$cigar=="M")==length(check.bases) & sum(!is.na(check.df$ins))==0 ){
                    if( sum(!is.na(check.df$snv[check.df$ref_pos %in% c(left.window[1],left.window[2])]))>0 ){
                        if( paste(check.df$reads_seq[check.df$ref_pos %in% c(left.window[1],left.window[2])], collapse="")=="CT" ){
                            cryptic.acceptor <- c(left.window[1],left.window[2]) 
                        }
                    }
                }
                if( !is.null(cryptic.acceptor) ){ break }
                
                right.window <- snv.window.left:snv.window.right + 3*i 
                check.bases <- snv.window.left:right.window[2]
                check.df <- roi.aln.df[ roi.aln.df$ref_pos %in% check.bases, ]
                
                if( nrow(check.df)==length(check.bases) & sum(check.df$cigar=="M")==length(check.bases) & sum(!is.na(check.df$ins))==0 ){
                    if( sum(!is.na(check.df$snv[check.df$ref_pos %in% c(right.window[1],right.window[2])]))>0 ){
                        if( paste(check.df$reads_seq[check.df$ref_pos %in% c(right.window[1],right.window[2])], collapse="")=="CT" ){
                            cryptic.acceptor <- c(right.window[1],right.window[2]) 
                        }
                    }
                }
                if( !is.null(cryptic.acceptor) ){ break }
                
            }
        }
        
    }
    
    acceptors <- list( real.acceptor=real.acceptor, cryptic.acceptor=cryptic.acceptor )
    return(acceptors)
}






