% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/rmargint-fn.R
\name{margint.rob}
\alias{margint.rob}
\title{Robust marginal integration procedures for additive models}
\usage{
margint.rob(
  formula,
  data,
  subset,
  point = NULL,
  windows,
  prob = NULL,
  sigma.hat = NULL,
  win.sigma = NULL,
  epsilon = 1e-06,
  type = "0",
  degree = NULL,
  typePhi = "Huber",
  k.h = 1.345,
  k.t = 4.685,
  max.it = 20,
  qderivate = FALSE,
  orderkernel = 2,
  Qmeasure = NULL
)
}
\arguments{
\item{formula}{an object of class \code{formula} (or one that can be coerced to that class): a symbolic description of the model to be fitted.}

\item{data}{an optional data frame, list or environment (or object coercible by \link{as.data.frame} to a data frame) containing the variables in the model. If not found in \code{data}, the variables are taken from \code{environment(formula)}, typically the environment from which the function was called.}

\item{subset}{an optional vector specifying a subset of observations to be used in the fitting process.}

\item{point}{a matrix of points where predictions will be computed and returned.}

\item{windows}{a vector or a squared matrix of bandwidths for the smoothing estimation procedure.}

\item{prob}{a vector of probabilities of observing each response (n). Defaults to \code{NULL}.}

\item{sigma.hat}{estimate of the residual standard error. If \code{NULL} we use the mad of the residuals obtained with local medians.}

\item{win.sigma}{a vector of bandwidths for estimating sigma.hat. If \code{NULL} it uses the argument windows if it is a vector or its diagonal if it is a matrix.}

\item{epsilon}{convergence criterion.}

\item{type}{three different type of estimators can be selected: type \code{'0'} (local constant on all the covariates), type \code{'1'} (local linear smoother on all the covariates), type \code{'alpha'} (local polynomial smoother only on the direction of interest).}

\item{degree}{degree of the local polynomial smoother in the direction of interest when using the estimator of type \code{'alpha'}. Defaults to \code{NULL} for the case when using estimators of type \code{'0'} or \code{'1'}.}

\item{typePhi}{one of either \code{'Tukey'} or \code{'Huber'}.}

\item{k.h}{tuning constant for a Huber-type loss function. Defaults to \code{1.345}.}

\item{k.t}{tuning constant for a Tukey-type loss function. Defaults to \code{4.685}.}

\item{max.it}{maximum number of iterations for the algorithm.}

\item{qderivate}{if TRUE, it calculates \code{g^(q+1)/(q+1)!} for each component only for the type \code{'alpha'} method. Defaults to \code{FALSE}.}

\item{orderkernel}{order of the kernel used in the nuisance directions when using the estimator of type \code{'alpha'}. Defaults to \code{2}.}

\item{Qmeasure}{a matrix of points where the integration procedure ocurrs. Defaults to \code{NULL} for calcuting the integrals over the sample.}
}
\value{
A list with the following components:
\item{mu }{Estimate for the intercept.}
\item{g.matrix }{Matrix of estimated additive components (n by p).}
\item{sigma.hat }{Estimate of the residual standard error.}
\item{prediction }{Matrix of estimated additive components for the points listed in the argument point.}
\item{mul }{A vector of size p showing in each component the estimated intercept that considers only that direction of interest when using the type \code{'alpha'} method.}
\item{g.derivative }{Matrix of estimated derivatives of the additive components (only when qderivate is \code{TRUE}) (n by p).}
\item{prediction.derivate }{Matrix of estimated derivatives of the additive components for the points listed in the argument point (only when qderivate is \code{TRUE}).}
\item{Xp}{Matrix of explanatory variables.}
\item{yp}{Vector of responses.}
\item{formula}{Model formula}
}
\description{
This function computes robust marginal integration procedures for additive models.
}
\details{
This function computes three types of robust marginal integration procedures for additive models.
}
\examples{
function.g1 <- function(x1) 24*(x1-1/2)^2-2
function.g2 <- function(x2) 2*pi*sin(pi*x2)-4
set.seed(140)
n <- 150
x1 <- runif(n)
x2 <- runif(n)
X <- cbind(x1, x2)
eps <- rnorm(n,0,sd=0.15)
regresion <- function.g1(x1) + function.g2(x2)
y <- regresion + eps
bandw <- matrix(0.25,2,2)
set.seed(8090)
nQ <- 80 
Qmeasure <- matrix(runif(nQ*2), nQ, 2)
fit.rob <- margint.rob(y ~ X, windows=bandw, type='alpha', degree=1, Qmeasure=Qmeasure) 

}
\references{
Boente G. and Martinez A. (2017). Marginal integration M-estimators for additive models. TEST, 26(2), 231-260. https://doi.org/10.1007/s11749-016-0508-0
}
\author{
Alejandra Martinez, \email{ale_m_martinez@hotmail.com}, Matias Salibian-Barrera
}
